#include <Inventor/ViewerComponents/Win/WinTimer.h>

#include <Inventor/ViewerComponents/Win/RenderAreaInteractive.h>

#include <Inventor/ViewerComponents/SoRenderAreaCore.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/SoDB.h>

#include <Inventor/ViewerComponents/nodes/SceneInteractor.h>

//------------------------------------------------------------------------------
RenderAreaInteractive::RenderAreaInteractive()
  : RenderArea()
  , m_countDownButtons( 0 )
{
  init(true);
}

//------------------------------------------------------------------------------
RenderAreaInteractive::RenderAreaInteractive( bool buildRootSceneGraph )
  : RenderArea()
  , m_countDownButtons( 0 )
{
  init(buildRootSceneGraph);
}

//------------------------------------------------------------------------------
RenderAreaInteractive::~RenderAreaInteractive()
{
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::startRender( SiRenderArea::RenderEventArg& )
{
  this->paintGL();
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::init( bool buildRootSceneGraph )
{
  m_clippingMode = AUTO;
  m_interactiveMode = SoInteractiveComplexity::AUTO;
  m_isAutoInteractive = true;

  if ( buildRootSceneGraph )
  {
    m_rootSceneGraph = new SceneInteractor();
    buildSceneGraph();
  }

  SoDB::setSystemTimer( new WinTimer );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::buildSceneGraph()
{
  m_appSceneGraph = new SoGroup;
  m_rootSceneGraph->addChild(m_appSceneGraph);
  RenderArea::setSceneGraph(m_rootSceneGraph);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setSceneGraph( SoNode *sceneGraph )
{
  m_appSceneGraph->removeAllChildren();
  m_appSceneGraph->addChild(sceneGraph);
}

//------------------------------------------------------------------------------
SceneInteractor*
RenderAreaInteractive::getRootSceneGraph()
{
  return m_rootSceneGraph;
}

//------------------------------------------------------------------------------
SceneInteractor*
RenderAreaInteractive::getSceneInteractor() const
{
  return m_rootSceneGraph;
}

//------------------------------------------------------------------------------
SoRenderAreaCore::RenderStatus
RenderAreaInteractive::render()
{
  if ( m_clippingMode == AUTO )
  {
    // Update camera clipping planes before each rendering.
    m_rootSceneGraph->adjustClippingPlanes(m_renderAreaCore->getSceneManager()->getViewportRegion());
  }

  // render scene
  return RenderArea::render();
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode)
{
  m_clippingMode = mode;
}

//------------------------------------------------------------------------------
RenderAreaInteractive::ClippingPlanesAdjustMode
RenderAreaInteractive::getClippingPlanesAdjustMode()
{
  return m_clippingMode;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::viewAll( const SbViewportRegion &viewport )
{
  m_rootSceneGraph->viewAll(viewport);
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::processEvents( UINT uMsg, WPARAM wParam, LPARAM lParam )
{
  switch ( uMsg )
  {
  case WM_KEYDOWN:
  {
    keyPressEvent(wParam);
    break;
  }
  case WM_KEYUP:
  {
    keyReleaseEvent(wParam);
    break;
  }

  case WM_LBUTTONDOWN:
  {
    mousePressEvent( lParam, SoMouseButtonEvent::BUTTON1);
    break;
  }
  case WM_LBUTTONUP:
  {
    mouseReleaseEvent( lParam, SoMouseButtonEvent::BUTTON1);
    break;
  }
  case WM_LBUTTONDBLCLK:
  {
    mouseDoubleClickEvent( lParam, SoMouseButtonEvent::BUTTON1);
    break;
  }

  case WM_MBUTTONDOWN:
  {
    mousePressEvent( lParam, SoMouseButtonEvent::BUTTON2);
    break;
  }
  case WM_MBUTTONUP:
  {
    mouseReleaseEvent( lParam, SoMouseButtonEvent::BUTTON2);
    break;
  }
  case WM_MBUTTONDBLCLK:
  {
    mouseDoubleClickEvent( lParam, SoMouseButtonEvent::BUTTON2);
    break;
  }

  case WM_RBUTTONDOWN:
  {
    mousePressEvent( lParam, SoMouseButtonEvent::BUTTON3);
    break;
  }
  case WM_RBUTTONUP:
  {
    mouseReleaseEvent( lParam, SoMouseButtonEvent::BUTTON3);
    break;
  }
  case WM_RBUTTONDBLCLK:
  {
    mouseDoubleClickEvent( lParam, SoMouseButtonEvent::BUTTON3);
    break;
  }

  case WM_MOUSEMOVE:
  {
    // This code discard move events created by touch screen
    const LPARAM touchScreenEvent = 0x82;
    if ( (GetMessageExtraInfo() & touchScreenEvent) == touchScreenEvent )
      break;

    mouseMoveEvent( lParam );
    break;
  }
  
  case WM_MOUSEWHEEL:
  {
    wheelEvent( GET_WHEEL_DELTA_WPARAM(wParam) );
    break;
  }

  case WM_TOUCH:
  {
    touchEvent( wParam, lParam );
    break;
  }

  }

  RenderArea::processEvents(uMsg, wParam, lParam );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::initializeGL(int width, int height)
{
  RenderArea::initializeGL(width, height);
  m_renderAreaCore->onStartRender().add( *this, &RenderAreaInteractive::startRender );
  m_renderAreaCore->getSceneManager()->setAutoInteractiveMode( m_isAutoInteractive );
  m_renderAreaCore->getSceneManager()->setInteractive( m_interactiveMode == SoInteractiveComplexity::FORCE_INTERACTION );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setInteractiveMode( SoInteractiveComplexity::InteractiveMode mode )
{
  m_interactiveMode = mode;
  m_isAutoInteractive = ( mode == SoInteractiveComplexity::AUTO );

  if ( m_renderAreaCore.ptr() != NULL )
  {
    m_renderAreaCore->getSceneManager()->setAutoInteractiveMode( m_isAutoInteractive );
    m_renderAreaCore->getSceneManager()->setInteractive( m_interactiveMode == SoInteractiveComplexity::FORCE_INTERACTION );
  }
}

//------------------------------------------------------------------------------
SoInteractiveComplexity::InteractiveMode
RenderAreaInteractive::getInteractiveMode() const
{
  return m_interactiveMode;
}

//------------------------------------------------------------------------------
SbBool
RenderAreaInteractive::processEvent( const SoEvent *event )
{
  if ( m_renderAreaCore.ptr() != NULL )
    return m_renderAreaCore->processEvent(event);

  SoDebugError::postWarning("RenderAreaInteractive::processEvents",
    "Rendering area is not initialized, events cannot be processed.");
  return FALSE;
}

//------------------------------------------------------------------------------
SbBool
RenderAreaInteractive::processEvents( const std::vector<const SoEvent*>& eventList )
{
  if ( m_renderAreaCore.ptr() != NULL )
    return m_renderAreaCore->processEvents(eventList);

  SoDebugError::postWarning("RenderAreaInteractive::processEvents",
    "Rendering area is not initialized, events cannot be processed.");
  return FALSE;
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::wheelEvent( short zDelta )
{
  SoEvent* event = WinEventToSoEvent::getMouseWheelEvent( zDelta, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::keyPressEvent( WPARAM param )
{
  SoEvent* event = WinEventToSoEvent::getKeyPressEvent( param, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::keyReleaseEvent( WPARAM param )
{
  SoEvent* event = WinEventToSoEvent::getKeyReleaseEvent( param, this  );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::mousePressEvent( LPARAM loc, SoMouseButtonEvent::Button button )
{
  setCaptureOnButtonDown();
  SoEvent* event = WinEventToSoEvent::getMousePressEvent( loc, button, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::mouseReleaseEvent( LPARAM loc, SoMouseButtonEvent::Button button )
{
  releaseCaptureOnButtonUp();
  SoEvent* event = WinEventToSoEvent::getMouseReleaseEvent( loc, button, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::mouseMoveEvent( LPARAM loc )
{
  SoEvent* event = WinEventToSoEvent::getMouseMoveEvent( loc, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::enterEvent()
{
  SoEvent* event = WinEventToSoEvent::getMouseEnterEvent( this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::leaveEvent()
{
  SoEvent* event = WinEventToSoEvent::getMouseLeaveEvent( this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::mouseDoubleClickEvent( LPARAM loc, SoMouseButtonEvent::Button button )
{
  SoEvent* event = WinEventToSoEvent::getMouseDoubleClickEvent( loc, button, this );
  processEvent( event );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::touchEvent( WPARAM wParam, LPARAM lParam )
{
  std::vector<const SoEvent*> eventsList;
  WinEventToSoEvent::getTouchEvents( wParam, lParam, this, eventsList );
  processEvents( eventsList );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::setCaptureOnButtonDown()
{
  m_countDownButtons++;
  SetCapture( this->m_hWnd );
}

//------------------------------------------------------------------------------
void
RenderAreaInteractive::releaseCaptureOnButtonUp()
{
  m_countDownButtons--;
  if ( m_countDownButtons == 0 )
  {
    ReleaseCapture();
  }
}
