#include "InteractionOrbiter.h"
#include <Inventor/SoPickedPoint.h>
#include <Inventor/elements/SoViewportRegionElement.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/sensors/SoAlarmSensor.h>
#include <Inventor/ViewerComponents/SoCameraInteractor.h>
#include <algorithm>
#include <cmath>
#include <Inventor/touch/SoTouchManager.h>
#include <Inventor/gestures/events/SoRotateGestureEvent.h>
#include <Inventor/gestures/events/SoScaleGestureEvent.h>

using openinventor::inventor::Axis;

const float InteractionOrbiter::s_selectionEpsilon = 0.05f;
const int InteractionOrbiter::s_nbMovesBeforeStartMoving = 5;
const double InteractionOrbiter::s_interaction_time = 0.5;
const double InteractionOrbiter::s_maxInteractionSpeed = 3.0;

InteractionOrbiter::InteractionOrbiter( SoCameraInteractor* cameraInteractor )
  : m_cameraInteractor( cameraInteractor )
  , m_orbitMethod( SceneOrbiter::TURNTABLE )
  , m_upAxis( Axis::Y )
  , m_turntableConstraintLevel( 0.f )
  , m_selectionAllowed( false )
  , m_isButton1Down( false )
  , m_mousePositionNorm( 0.f, 0.f )
  , m_mousePosition( 0, 0 )
  , m_eventTime( 0 )
  , m_waitBeforeMoving( s_nbMovesBeforeStartMoving )
  , m_canMove( false )
  , m_interactiveCount( 0 )
  , m_currentAction( nullptr )
{
  m_mouseWheelDelta = SoPreferences::getInt( "OIV_WHEEL_DELTA", 120 );

  m_interactionEndSensor = new SoAlarmSensor();
  m_interactionEndSensor->setData( this );
  m_interactionEndSensor->setFunction( []( void* data, SoSensor* /*sensor*/ )
  {
    // stop interaction
    InteractionOrbiter* interaction = static_cast<InteractionOrbiter*>(data);
    interaction->interactiveCountDec();
  } );
}

InteractionOrbiter::InteractionOrbiter( const InteractionOrbiter& copyFrom )
  : m_cameraInteractor( copyFrom.m_cameraInteractor )
  , m_orbitMethod( copyFrom.m_orbitMethod )
  , m_upAxis( copyFrom.m_upAxis )
  , m_turntableConstraintLevel( copyFrom.m_turntableConstraintLevel )
  , m_selectionAllowed( copyFrom.m_selectionAllowed )
  , m_isButton1Down( copyFrom.m_isButton1Down )
  , m_mousePositionNorm( copyFrom.m_mousePositionNorm )
  , m_mousePosition( copyFrom.m_mousePosition )
  , m_mouseWheelDelta( copyFrom.m_mouseWheelDelta )
  , m_eventTime( 0 )
  , m_waitBeforeMoving( copyFrom.m_waitBeforeMoving )
  , m_canMove( copyFrom.m_canMove )
  , m_interactiveCount( copyFrom.m_interactiveCount )
  , m_currentAction( nullptr )
{
  m_interactionEndSensor = new SoAlarmSensor();
  m_interactionEndSensor->setData( this );
  m_interactionEndSensor->setFunction( []( void* data, SoSensor* /*sensor*/ )
  {
    // stop interaction
    InteractionOrbiter* interaction = static_cast<InteractionOrbiter*>(data);
    interaction->interactiveCountDec();
  } );
}

InteractionOrbiter::~InteractionOrbiter()
{
  delete m_interactionEndSensor;
}

void
InteractionOrbiter::reset()
{
  // reset mouse buttons state
  m_isButton1Down = false;
  m_currentAction = nullptr;
}

void
InteractionOrbiter::setCameraInteractor( SoCameraInteractor* cameraInteractor )
{
  m_cameraInteractor = cameraInteractor;
}

void
InteractionOrbiter::setConstraintLevel( float level )
{
  m_turntableConstraintLevel = level;
}

float
InteractionOrbiter::getConstraintLevel()
{
  return m_turntableConstraintLevel;
}

void
InteractionOrbiter::setRotationMethod( SceneOrbiter::RotationMethod style )
{
  m_orbitMethod = style;
}

SceneOrbiter::RotationMethod
InteractionOrbiter::getRotationMethod()
{
  return m_orbitMethod;
}

void
InteractionOrbiter::setUpAxis( Axis::Type axis )
{
  m_upAxis = axis;
}

Axis::Type
InteractionOrbiter::getUpAxis()
{
  return m_upAxis;
}

void
InteractionOrbiter::beginAction( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action )
{
  const SoMouseButtonEvent::Button button = mouseEvent->getButton();
  const SoMouseButtonEvent::State state = mouseEvent->getState();
  if ( button == SoMouseButtonEvent::BUTTON1 )
  {
    m_isButton1Down = ( state == SoMouseButtonEvent::DOWN );

    const bool ctrlDown = mouseEvent->wasCtrlDown();
    const bool shiftDown = mouseEvent->wasShiftDown();
    if (!m_isButton1Down && m_selectionAllowed && !ctrlDown && !shiftDown)
    {
      // we let pass the event
      return;
    }

    m_mousePositionNorm = mouseEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));
    beginAction(action,mouseEvent->wasCtrlDown(),mouseEvent->wasShiftDown());

    if ( !m_isButton1Down && !m_selectionAllowed )
    {
      action->setHandled();
    }
  }
}

void
InteractionOrbiter::beginAction( SoKeyboardEvent* keyEvent, SoHandleEventAction* action )
{
  const SoKeyboardEvent::Key key = keyEvent->getKey();
  const SoKeyboardEvent::State state = keyEvent->getState();

  if ( key == SoKeyboardEvent::LEFT_CONTROL || key == SoKeyboardEvent::LEFT_SHIFT )
  {
    m_selectionAllowed = false;
  }

  switch ( key )
  {
  case SoKeyboardEvent::LEFT_CONTROL:
    beginAction( action, state == SoKeyboardEvent::DOWN, keyEvent->wasShiftDown() );
    break;
  case SoKeyboardEvent::LEFT_SHIFT:
    beginAction( action, keyEvent->wasCtrlDown(), state == SoKeyboardEvent::DOWN );
    break;
  default:
    break;
  }
}

void
InteractionOrbiter::beginAction( SoHandleEventAction* action, bool ctrlDown, bool shiftDown )
{
  m_currentAction = action;

  if ( m_isButton1Down && ctrlDown )
  {
    // BUTTON 1 + CTRL = pan
    beginPan( action );
  }
  else if ( m_isButton1Down && !shiftDown )
  {
    // BUTTON 1 without modifier = orbit
    beginSelection( action );
  }
}

void
InteractionOrbiter::doAction( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action )
{
  m_currentAction = action;

  // ZOOM
  int wheelDelta = wheelEvent->getDelta() / m_mouseWheelDelta;
  float scaleFactor = std::pow( 2.f, ( float )( wheelDelta * M_PI / 180.0f ) );

  doDollyWithCenter( m_mousePositionNorm, scaleFactor, action );
}

void
InteractionOrbiter::doAction( SoLocation2Event* mouseEvent, SoHandleEventAction* action )
{
  m_currentAction = action;

  const bool ctrlDown = mouseEvent->wasCtrlDown();
  const bool shiftDown = mouseEvent->wasShiftDown();

  if ( m_isButton1Down && ctrlDown )
  {
    // BUTTON 1 + CTRL = pan
    doPan( mouseEvent, action );
  }
  else if ( m_isButton1Down && shiftDown )
  {
    // BUTTON 1 + SHIFT = dolly
    doDolly( mouseEvent, action );
  }
  else if ( m_isButton1Down )
  {
    if ( m_selectionAllowed )
    {
      doIdentifyGesture( mouseEvent, action );
    }
    else
    {
      // BUTTON 1 without modifier = orbit
      doOrbit( mouseEvent, action );
    }
  }

  if ( !action->isHandled() )
  {
    // update mouse cursor's position
    m_mousePositionNorm = mouseEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));
  }
}

void
InteractionOrbiter::doAction( SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  m_currentAction = action;

  const SoTouchEvent::State state = touchEvent->getState();
  int numFinger = touchEvent->getTouchManager()->getFingerNumber();

  // One Finger
  if ( numFinger == 1 )
  {
    // On Move
    if ( state == SoTouchEvent::MOVE )
    {
      // Wait before start moving in order to handle selection
      if ( m_waitBeforeMoving > 0 )
      {
        --m_waitBeforeMoving;
        action->setHandled();
        return;
      }

      if ( !m_canMove )
      {
        // Saving current pos as ref for orbiting
        m_mousePositionNorm = touchEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));
        beginOrbit( action );
        m_canMove = true;
      }
      else
        doOrbit( touchEvent, action );
    }

    // On finger Release
    if ( state == SoTouchEvent::UP )
    {
      if ( m_canMove )
      {
        m_canMove = false;
        m_waitBeforeMoving = s_nbMovesBeforeStartMoving; // Reset counter
        action->setHandled(); // Blocking
      }
      else
      {
        // let pass the event for selection
      }
    }
  }
  // Multi touch
  else if ( numFinger > 1 )
  {
    m_canMove = false;
    if ( state == SoTouchEvent::MOVE )
      doTranslate( touchEvent, action );
  }
}

void
InteractionOrbiter::doTranslate( const SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  startInteraction();

  // NOTE: We are normalizing a distance, not a position. Cannot use vpRegion.normalize!
  const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
  const SbVec2i32& vpSize = vpRegion.getViewportSizePixels_i32();
  SbVec2f displacement = touchEvent->getDisplacement(); // Returns an object, not a reference
  SbVec2f normDisp( 0.5f * displacement[0] / vpSize[0], 0.5f * displacement[1] / vpSize[1] );
  m_cameraInteractor->translate( normDisp, vpRegion );
  action->setHandled();
}

void
InteractionOrbiter::doAction( SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action )
{
  m_currentAction = action;

  const SbVec2f normPosition = SoViewportRegionElement::get(action->getState()).normalize(scaleEvent->getPosition());
  doDollyWithCenter( normPosition, 1.f / scaleEvent->getDeltaScaleFactor(), action );
}

void
InteractionOrbiter::doAction( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action )
{
  m_currentAction = action;
  startInteraction();

  const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
  const SbVec2f& eventNormPosition = rotateEvent->getNormalizedPosition(vpRegion);
  float distFromEye = m_cameraInteractor->getCamera()->getViewVolume().getNearDist();
  SbVec3f rotCenter = m_cameraInteractor->projectToPlane( eventNormPosition, distFromEye, vpRegion );

  m_cameraInteractor->setRotationAxis( SbVec3f( 0, 0, 1.0f ) );
  m_cameraInteractor->setRotationCenter( rotCenter );
  m_cameraInteractor->rotate( -rotateEvent->getDeltaRotation() * 0.5f );
  action->setHandled();
}

void
InteractionOrbiter::beginPan( SoHandleEventAction* action )
{
  m_cameraInteractor->activatePanning(m_mousePositionNorm, SoViewportRegionElement::get(action->getState()));
  action->setHandled();
}

void
InteractionOrbiter::beginSelection( SoHandleEventAction* /*action*/ )
{
  m_selectionAllowed = true;
}

void
InteractionOrbiter::beginOrbit( SoHandleEventAction* action )
{
  switch ( m_orbitMethod )
  {
  case SceneOrbiter::TRACKBALL:
    m_cameraInteractor->activateOrbiting( m_mousePositionNorm );
    m_cameraInteractor->setRotationCenter( m_cameraInteractor->getFocalPoint() );
    break;
  case SceneOrbiter::TURNTABLE:
    m_mousePosition = action->getEvent()->getPosition();
    m_eventTime = action->getEvent()->getTime().getMsecValue();
    break;
  default:
    // do nothing
    break;
  }
  action->setHandled();
}

void
InteractionOrbiter::doIdentifyGesture( SoEvent* anyEvent, SoHandleEventAction* action )
{
  const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
  const SbVec2f& current_mousePositionNorm = anyEvent->getNormalizedPosition(vpRegion);
  SbVec2f delta = current_mousePositionNorm - m_mousePositionNorm;
  float length = delta.length();
  if ( length > s_selectionEpsilon )
  {
    m_selectionAllowed = false;
    m_mousePositionNorm = anyEvent->getNormalizedPosition( vpRegion );
    beginOrbit( action );
  }
  action->setHandled();
}

void
InteractionOrbiter::doPan( const SoEvent* anyEvent, SoHandleEventAction* action )
{
  startInteraction();

  const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
  m_mousePositionNorm = anyEvent->getNormalizedPosition( vpRegion );
  m_cameraInteractor->pan( m_mousePositionNorm, vpRegion );
  action->setHandled();
}

void
InteractionOrbiter::doOrbit( SoEvent* anyEvent, SoHandleEventAction* action )
{
  startInteraction();

  switch ( m_orbitMethod )
  {
  case SceneOrbiter::TURNTABLE:
    doTurntableOrbit( anyEvent, action );
    break;
  case SceneOrbiter::TRACKBALL:
  default:
    m_mousePositionNorm = anyEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));
    m_cameraInteractor->orbit( m_mousePositionNorm );
    break;
  }
  action->setHandled();
}

void
InteractionOrbiter::doTurntableOrbit( SoEvent* anyEvent, SoHandleEventAction* action )
{
  const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());

  // step 1 : Calculate the amount of rotation given the mouse movement.
  // a movement from left to right or top to bottom = 2*PI
  double twoPIperPixel = 2 * M_PI / std::min( vpRegion.getViewportSizePixels_i32()[0], vpRegion.getViewportSizePixels_i32()[1] ) * .8;

  SbVec2s newPos = anyEvent->getPosition();
  unsigned long newEventTime = anyEvent->getTime().getMsecValue();

  int xDiff = m_mousePosition[0] - newPos[0];
  int yDiff = newPos[1] - m_mousePosition[1];

  double xAngle = xDiff * twoPIperPixel;
  double yAngle = yDiff * twoPIperPixel;

  unsigned long dt = newEventTime - m_eventTime;
  double distance = std::sqrt( xDiff * xDiff + yDiff * yDiff );
  // interaction speed: pixels/ms
  double speed = dt == 0 ? 0 : distance / dt;
  // apply constraint level
  double level = speed > s_maxInteractionSpeed ? 0 : 1 - speed / s_maxInteractionSpeed;
  bool constrained = level < m_turntableConstraintLevel;

  SbMatrix mx;
  mx = m_cameraInteractor->getCamera()->orientation.getValue();
  SbVec3f focalPt = m_cameraInteractor->getFocalPoint();

  xDiff = std::abs( xDiff );
  yDiff = std::abs( yDiff );

  // if it's a fast movement, do this rotation only if it's an "X" rotation
  if ( !constrained || xDiff >= yDiff )
  {
    // step 2: Rotate the camera around the pivot point on the up axis.
    SbVec3f rotAxis;
    switch ( m_upAxis )
    {
    case Axis::X:
      rotAxis.setValue( mx[0][0], mx[1][0], mx[2][0] );
      break;
    case Axis::Z:
      rotAxis.setValue( mx[0][2], mx[1][2], mx[2][2] );
      break;
    case Axis::Y:
    default:
      rotAxis.setValue( mx[0][1], mx[1][1], mx[2][1] );
      break;
    }

    m_cameraInteractor->setRotationAxis( rotAxis );
    m_cameraInteractor->setRotationCenter( focalPt );
    m_cameraInteractor->rotate( (float) xAngle );
  }
  // if it's a fast movement, do this rotation only if it's an "Y" rotation
  if ( !constrained || yDiff >= xDiff )
  {
    // step 3: Rotate the camera around the pivot point on the second axis.
    SbVec3f horiz( 1, 0, 0 );
    m_cameraInteractor->setRotationAxis( horiz );
    m_cameraInteractor->setRotationCenter( focalPt );
    m_cameraInteractor->rotate( (float) yAngle );
  }

  // Update the mouse position for the next rotation
  m_mousePosition = newPos;
  m_eventTime = newEventTime;
}

void
InteractionOrbiter::doDolly( SoEvent* anyEvent, SoHandleEventAction* action )
{
  startInteraction();

  const SbVec2f& newLocator = anyEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));
  const float dollyScaleFactor = float( pow( 2.0f, 10.0f * ( newLocator[1] - m_mousePositionNorm[1] ) ) );
  m_cameraInteractor->dolly( dollyScaleFactor );

  m_mousePositionNorm = newLocator;
  action->setHandled();
}

void
InteractionOrbiter::doDollyWithCenter( const SbVec2f& center, float scaleFactor, SoHandleEventAction* action )
{
  startInteraction();

  m_cameraInteractor->dollyWithZoomCenter(center, scaleFactor, SoViewportRegionElement::get(action->getState()));

  action->setHandled();
}

void
InteractionOrbiter::interactiveCountInc()
{
  m_interactiveCount++;
  if ( m_interactiveCount == 1 && m_currentAction != nullptr )
  {
    // an interaction starts, set interactive mode in scene manager
    m_currentAction->getSceneManager()->setInteractive( true );
    // force a redraw, so graph will see the change in
    // the SoInteractionElement
    m_currentAction->getSceneManager()->scheduleRedraw();
  }
}

void
InteractionOrbiter::interactiveCountDec()
{
  if ( m_interactiveCount > 0 )
  {
    m_interactiveCount--;

    if ( m_interactiveCount == 0 && m_currentAction != nullptr )
    {
      // interaction is finished, disable interactive mode in scene manager
      m_currentAction->getSceneManager()->setInteractive( false );
      // force a redraw, so graph will see the change in
      // the SoInteractionElement
      m_currentAction->getSceneManager()->scheduleRedraw();
    }
  }
}

void
InteractionOrbiter::startInteraction()
{
  if ( m_interactionEndSensor->isScheduled() )
  {
    // already interacting, unschedule previous timer
    m_interactionEndSensor->unschedule();
  }
  else
  {
    // interaction starts
    interactiveCountInc();
  }

  if (s_interaction_time == 0)
    return;

  // wait 0.5s... if scene is still, consider interaction is finished
  m_interactionEndSensor->setTimeFromNow( SbTime( s_interaction_time ) );
  m_interactionEndSensor->schedule();
}
