#include "OrbitAnimator.h"

#include <Inventor/nodes/SoCamera.h>

static const size_t ROTATION_SAMPLE_COUNT = 3;

OrbitAnimator::OrbitAnimator(SoCameraInteractor* cameraInteractor)
  : m_cameraInteractor( cameraInteractor )
  , m_currentRotationSampleIndex( 0 )
  , m_sceneGraph( nullptr )
  , m_isAutoClippingPlanes( SoPreferences::getBool("OIV_SCENE_EXAMINER_AUTO_CLIPPING_PLANES", TRUE) )
{
  m_sphereProjector.setViewVolume( m_cameraInteractor->getCamera()->getViewVolume() );
  m_sphereProjector.setSphere( SbSphere( SbVec3f( 0.f, 0.f, 0.f ), .7f ) );

  m_rotationSamples.resize( ROTATION_SAMPLE_COUNT );
}

void
OrbitAnimator::updateRotation()
{
  float angle;
  SbVec3f averageAxis;

  // get average axis of rotation
  m_rotationSamples[m_currentRotationSampleIndex].getValue( averageAxis, angle );

  // get average angle of rotation
  float finalAngle = 0;
  for ( size_t i = m_currentRotationSampleIndex; i < m_currentRotationSampleIndex + ROTATION_SAMPLE_COUNT; i++ )
  {
    SbVec3f axis;
    m_rotationSamples[i % ROTATION_SAMPLE_COUNT].getValue( axis, angle );

    // only consider non-zero values
    finalAngle += angle == 0.f ? finalAngle : angle;
  }

  m_rotation.setValue( averageAxis, -finalAngle );
}

void
OrbitAnimator::addScreenPoint( const SbVec2f& screenPosition )
{
  SbRotation currentRotation;
  m_sphereProjector.projectAndGetRotation( screenPosition, currentRotation );

  m_rotationSamples[m_currentRotationSampleIndex] = currentRotation;

  // compute the average rotation
  updateRotation();

  m_currentRotationSampleIndex = ( m_currentRotationSampleIndex + 1 ) % ROTATION_SAMPLE_COUNT;
}

void
OrbitAnimator::setStartPoint( const SbVec2f& initialOrbitPosition )
{
  for ( size_t i = 0; i < ROTATION_SAMPLE_COUNT; ++i )
  {
    m_rotationSamples[i] = SbRotation();
  }

  m_sphereProjector.project( initialOrbitPosition );

  updateRotation();

  m_startPoint = initialOrbitPosition;
}

SbVec2f OrbitAnimator::startPoint() const
{
  return m_startPoint;
}

void
OrbitAnimator::animate()
{
  if ( !m_rotation.equals( m_rotation.identity(), 0.00001f ) )
  {
    m_cameraInteractor->orbit( m_rotation );

    if ( m_isAutoClippingPlanes )
    {
      m_cameraInteractor->adjustClippingPlanes( m_sceneGraph, m_vpRegion );
    }
  }
  else
    stop();
}

void
OrbitAnimator::setSceneGraph(SoNode* sceneGraph)
{
  m_sceneGraph = sceneGraph;
}

void
OrbitAnimator::setViewportRegion(const SbViewportRegion& vpRegion)
{
  m_vpRegion = vpRegion;
}
