/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Mar 2014)
**=======================================================================*/
#include <SoInputGenericReader.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/SoInput.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/SoLists.h>

// Node used to generate a RasterImage file scenegraph
#include <Inventor/image/SbRasterImage.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoShapeHints.h>

// Node used to generate a VolumeReader file scenegraph
#include <LDM/readers/SoVolumeReader.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>


/*************************************************************************/
void initPlugin()
{
  if (SoPreferences::getBool("OIV_GENERIC_READER_ENABLE_VVIZ", TRUE))
    SoVolumeRendering::init();
  SoInputGenericReader::initClass();
}

/*************************************************************************/
void exitPlugin()
{
  SoInputGenericReader::exitClass();
  if (SoPreferences::getBool("OIV_GENERIC_READER_ENABLE_VVIZ", TRUE))
    SoVolumeRendering::finish();
}

/*************************************************************************/
class SoInputGenericReaderImpl
{
public:
  // return a scene graph based on a user template scenegraph file
  // if template file is not foiund then a hardcodede scene graph will be generated
  static SoSeparator* generateRasterImageScenegraph(const SbString& filename);
  static SoSeparator* generateVolumeReaderScenegraph(const SbString& filename);

private:
  // load the given template scenegrap filename
  static SoSeparator* loadTemplateSceneGraph(const SbString& templateFilename);

  // return a prebuilt scene graph
  static SoSeparator* defaultVolumeReaderSceneGraph( );
  static SoSeparator* defaultRasterImageSceneGraph( );

  // default template scenegraph filename
  // TODO: could be configured by an envvar or a method
  static SbString s_defaultVVizTemplate;
  static SbString s_defaultTextureTemplate;

  // return the first node of the given type
  template<typename T>
  static T* searchNode(const SoNode* scene)
  {
    scene->ref();
    SoSearchAction sa;
    sa.setFind(SoSearchAction::TYPE);
    sa.setSearchingAll(true);
    sa.setType(T::getClassTypeId());
    sa.apply((SoNode*)scene);

    SoPath* path = sa.getPath();
    if ( !path )
    {
      scene->unrefNoDelete();
      return(NULL);
    }
    T* ret = dynamic_cast<T*>(path->getTail());
    scene->unrefNoDelete();
    return ret;
  }

};

SO_TYPED_CLASS_SOURCE(SoInputReader, SoInputGenericReader, "InputGenericReader");

/*************************************************************************/
SoInputGenericReader::SoInputGenericReader()
{}

/*************************************************************************/
SoInputGenericReader::~SoInputGenericReader()
{}

/*************************************************************************/
bool SoInputGenericReader::canConvert(SoInput* in)
{
  // 1- check if we can load this file with a VolumeViz reader
  SoRef<SoVolumeReader> volumeReader = SoVolumeReader::getAppropriateReader(in->getCurStringFileName());
  if ( volumeReader.ptr()!=NULL)
    return true;

  // 2- check if we can load this file with an OIV rasterReader
  SbRasterImage* rasterimage = SoTexture::readTexture( in->getCurStringFileName() );
  if ( rasterimage!=NULL )
  {
    delete rasterimage;
    return true;
  }

  // we do not support this file format
  return false;
}

/*************************************************************************/
SoSeparator* SoInputGenericReader::convert( SoInput* in )
{
  SoSeparator* sep = NULL;

  // 1- check if we can load this file with a VolumeViz reader
  // and return a typical scenegraph
  SoRef<SoVolumeReader> volumeReader = SoVolumeReader::getAppropriateReader(in->getCurStringFileName());
  if ( volumeReader.ptr()!=NULL)
  {
    sep = SoInputGenericReaderImpl::generateVolumeReaderScenegraph(in->getCurStringFileName());
    return sep;
  }

  // 2- check if we can load this file with an OIV rasterReader
  // and send
  SbRasterImage* rasterimage = SoTexture::readTexture( in->getCurStringFileName() );
  if ( rasterimage!=NULL )
  {
    sep = SoInputGenericReaderImpl::generateRasterImageScenegraph(in->getCurStringFileName());
    delete rasterimage;
    return sep;
  }


  // we do not support this file format
  return sep;
}


// init static members
SbString SoInputGenericReaderImpl::s_defaultVVizTemplate("$OIVHOME/source/IvTune/IvGenericFileReader/VolumeViz-scenegraph-Template.iv");
SbString SoInputGenericReaderImpl::s_defaultTextureTemplate("$OIVHOME/source/IvTune/IvGenericFileReader/Texture-scenegraph-Template.iv");

/*************************************************************************/
SoSeparator* SoInputGenericReaderImpl::loadTemplateSceneGraph(const SbString& templateFilename)
{
  SoSeparator* rootSep = NULL;
  SoInput myInput;
  if (myInput.openFile(templateFilename) )
  {
    rootSep = SoDB::readAll(&myInput);
    if ( rootSep == NULL )
    {
#if defined(_DEBUG)
      SoDebugError::postWarning(OIV_FUNCTION,"some issue while reading %s\n",templateFilename.toLatin1());
#endif
    }
  }
  else
  {
#if defined(_DEBUG)
    SoDebugError::postWarning(OIV_FUNCTION,"unable to open %s\n",templateFilename.toLatin1());
#endif
  }
  return rootSep;
}


/*************************************************************************/
SoSeparator* SoInputGenericReaderImpl::generateVolumeReaderScenegraph( const SbString& filename )
{
  // this is the root of the scenegraph that we will return.
  SoRef<SoSeparator> rootSep;

  // try the template file
  rootSep = loadTemplateSceneGraph(s_defaultVVizTemplate);

  // else use the builin scenegraph
  if ( rootSep.ptr()==NULL )
    rootSep = defaultVolumeReaderSceneGraph();

  // From the template, update the VolumeData filename with the one in the param
  SoVolumeData * volumeData = searchNode<SoVolumeData>(rootSep.ptr());
  if ( volumeData!=NULL )
  {
    volumeData->fileName = filename;
  }
  else
  {
#if defined(_DEBUG)
    SoDebugError::postWarning(OIV_FUNCTION,"unable to find a node of type SoVolumeData in the scenegraph\n");
#endif
    return NULL;
  }

  return rootSep.releaseNoDelete();
}

/*************************************************************************/
SoSeparator* SoInputGenericReaderImpl::defaultVolumeReaderSceneGraph()
{
  // for now we just return a simple SoVolumeData node
  SoVolumeData* volumeData = new SoVolumeData();

  SoVolumeRenderingQuality* volumerenderingQuality = new SoVolumeRenderingQuality();
  volumerenderingQuality->ambientOcclusion = TRUE;
  volumerenderingQuality->deferredLighting = TRUE;

  SoVolumeRender* volumeRender = new SoVolumeRender();
  volumeRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
  volumeRender->subdivideTile = TRUE;
  volumeRender->samplingAlignment = SoVolumeRender::SMOOTH_BOUNDARY_ALIGNED;
  volumeRender->opacityThreshold = 0.05f;

  SoSeparator* rootSep = new SoSeparator();
  rootSep->addChild(volumeData);
  rootSep->addChild(new SoDataRange());
  rootSep->addChild(new SoTransferFunction());
  rootSep->addChild(volumerenderingQuality);
  rootSep->addChild(volumeRender);

  return rootSep;
}

/*************************************************************************/
SoSeparator* SoInputGenericReaderImpl::defaultRasterImageSceneGraph()
{
  SoVertexProperty* vpquad = new SoVertexProperty();
  vpquad->vertex.set1Value(0,SbVec3f(-1,-1, 0));
  vpquad->vertex.set1Value(1,SbVec3f(-1, 1, 0));
  vpquad->vertex.set1Value(2,SbVec3f( 1, 1, 0));
  vpquad->vertex.set1Value(3,SbVec3f( 1,-1, 0));

  vpquad->texCoord.set1Value(0,SbVec2f(0,0));
  vpquad->texCoord.set1Value(1,SbVec2f(0,1));
  vpquad->texCoord.set1Value(2,SbVec2f(1,1));
  vpquad->texCoord.set1Value(3,SbVec2f(1,0));


  SoFaceSet* quad = new SoFaceSet();
  quad->vertexProperty.setValue(vpquad);
  quad->numVertices = 4;

  SoTexture2* texture2 = new SoTexture2();

  SoShapeHints* shapeHint = new SoShapeHints();
  shapeHint->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;

  SoSeparator* rootSep = new SoSeparator();
  rootSep->addChild(shapeHint);
  rootSep->addChild(texture2);
  rootSep->addChild(quad);

  return rootSep;
}

/*************************************************************************/
SoSeparator* SoInputGenericReaderImpl::generateRasterImageScenegraph( const SbString& filename )
{
  // this is the root of the scenegraph that we will return.
  SoRef<SoSeparator> rootSep;

  // try the template file
  rootSep = loadTemplateSceneGraph(s_defaultTextureTemplate);

  // else use the builin scenegraph
  if ( rootSep.ptr()==NULL )
    rootSep = defaultRasterImageSceneGraph();

  // From the template, update the Texture2 filename with the one in the param
  SoTexture2* texture = searchNode<SoTexture2>(rootSep.ptr());
  if ( texture!=NULL )
    texture->filename = filename;
  else
  {
#if defined(_DEBUG)
    SoDebugError::postWarning(OIV_FUNCTION,"unable to find a node of type SoVolumeData in the scenegraph\n");
#endif
    return NULL;
  }

  return rootSep.releaseNoDelete();
}
