/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T.MEHAMLI (Dec 2010)
**=======================================================================*/

#include <Inventor/sys/SoGL.h>
#include <Ivt_DirectionalLightEditor.h>

#include <IvTune/IvTuneExtender/IvtShell.h>
#include <IvTune/IvTuneExtender/IvtServices.h>

#include <Inventor/SbViewportRegion.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoSceneManager.h>

#include <Inventor/actions/SoSearchAction.h>

#include <Inventor/devices/SoGLContext.h>

#include <Inventor/events/SoLocation2Event.h> // For mouse move events
#include <Inventor/events/SoMouseButtonEvent.h> // For mouse press/release events

#include <Inventor/helpers/SbGlContextHelper.h>

#include <Inventor/Gui/viewers/SoGuiAlgoViewers.h>

#include <Inventor/manips/SoDirectionalLightManip.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoEnvironment.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSphere.h>

#include <Inventor/sensors/SoNodeSensor.h>

#include <QColorDialog>
#include <QDockWidget>
#include <QFrame>
#include <QMouseEvent>
#include <QPalette>

#include <QOpenGLWidget>

//------------------------------------------------------------------------------
// OIV Geometry
//------------------------------------------------------------------------------
char* Ivt_DirectionalLightEditor::m_editorContentBuffer = (char*) // Taken from SoQtDirLitEd.cxx
"#Inventor V2.0 ascii\n\
Separator {\n\
    DEF DIR_LIGHT_EDITOR_ROTATOR Group {\n\
	LightModel { model PHONG }\n\
	Separator {\n\
	    Transform {\n\
		scaleFactor .2 .2 .2\n\
		rotation    1 0 0 -1.57079632679489661923  # PI/2\n\
		translation 0 0 1.2\n\
	    }\n\
	    Cone {}\n\
	}\n\
	Separator {\n\
	    Transform {\n\
		scaleFactor .08 .3 .08\n\
		rotation    1 0 0 -1.57079632679489661923  # PI/2\n\
		translation 0 0 1.7\n\
	    }\n\
	    Cylinder {}\n\
	}\n\
	Separator {\n\
	    Transform {\n\
		scaleFactor .1 .1 .1\n\
		rotation    1 0 0 -1.57079632679489661923  # PI/2\n\
		translation 0 0 -1.1\n\
	    }\n\
	    Cone {}\n\
	}\n\
    }\n\
    DEF dirLightEditorRotator Separator {\n\
	Material {\n\
	    diffuseColor	[ 0.5 0.5 0.5 ]\n\
	    emissiveColor	[ 0.5 0.5 0.5 ]\n\
	}\n\
	DrawStyle { lineWidth 2 }\n\
	USE DIR_LIGHT_EDITOR_ROTATOR\n\
    }\n\
    DEF dirLightEditorRotatorActive Separator {\n\
	Material {\n\
	    diffuseColor	[ 0.5 0.5 0.0 ]\n\
	    emissiveColor	[ 0.5 0.5 0.0 ]\n\
	}\n\
	DrawStyle { lineWidth 3 }\n\
	USE DIR_LIGHT_EDITOR_ROTATOR\n\
    }\n\
}\n\
";

#ifdef _WIN32
#pragma warning( push )
#pragma warning( disable:4996 )
#endif

//------------------------------------------------------------------------------
Ivt_DirectionalLightEditor::Ivt_DirectionalLightEditor()
  : IvtEditor()
  , m_glRAAlgos( NULL )
  , m_dirLightManip( NULL )
  , m_root( NULL )
{
  m_info.name = "Directional Light Editor";
  m_info.author ="Thermo Fisher Scientific";
  m_info.description = "An editor for the directional lights.";
  m_info.version = "1.0.0";

  m_handledType = SoDirectionalLight::getClassTypeId();
}

//------------------------------------------------------------------------------
Ivt_DirectionalLightEditor::~Ivt_DirectionalLightEditor()
{
  delete m_buttonEvent;
  delete m_callbackList;
  delete m_cameraSensor;
  delete m_loc2Event;

  //m_cameraToWatch->unref();
  if( m_dirLightManip )
    m_dirLightManip->unref();
  if( m_glRAAlgos )
    m_glRAAlgos->unref();
  if( m_root )
    m_root->unref();

}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::setUp()
{
  m_dirLight = (SoDirectionalLight*) m_editedObject;

  m_ignoreCallback = TRUE;
  copyLight( m_dirLightManip, m_dirLight );
  m_ignoreCallback = FALSE;
  updateLocalComponents();

  // Look for the camera before the light in the user scene graph
  SoPath* fullPathToLight = new SoPath;
  SoSearchAction sa;
  fullPathToLight->setHead( m_usg );
  fullPathToLight->ref();
  sa.setNode( m_dirLight );
  sa.setInterest( SoSearchAction::LAST );
  sa.apply( fullPathToLight );
  if ( sa.getPath() )
  {
    SoFullPath* camPath = (SoFullPath*)sa.getPath();
    SoSearchAction sa2;
    sa2.setType( SoCamera::getClassTypeId() );
    sa2.apply( camPath );
    if ( sa2.getPath() ) 
    {
      m_cameraToWatch = (SoCamera *)((SoFullPath *)sa2.getPath())->getTail();
      m_cameraToWatch->ref();
      cameraSensorCB( this, NULL );
      m_cameraSensor->attach( m_cameraToWatch );
    }
  }

  fullPathToLight->unref();

  updateRA();
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::openColorEditor()
{
#if (QT_VERSION < 0x040500)
  QColor color = QColorDialog::getColor();
  colorDialogCB( color );
#else
  if ( !m_colorDialog->isVisible() )
    m_colorDialog->show();
  else
    m_colorDialog->hide();
#endif
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::intensitySliderCB( int value )
{
  double fValue = (double)value/100.00f;
  intensityLE->setText( QString::number(fValue, 'g', 2) );

  // Update our local light
  m_ignoreCallback = TRUE; // ignore resulting manip callback
  m_dirLightManip->intensity.setValue( (float)fValue );
  m_ignoreCallback = FALSE;

  m_dirLight->intensity.setValue( (float)fValue );
  
  // Invoke the callbacks with the new values
  m_callbackList->invokeCallbacks( m_dirLightManip );
  
  updateRA();
}

//------------------------------------------------------------------------------
bool
Ivt_DirectionalLightEditor::eventFilter( QObject* watched, QEvent* event )
{
#if (QT_VERSION < 0x040500)
  if ( watched == m_glRA )
#else
  if ( watched == m_colorDialog )
  {
    if ( event->type() == QEvent::Close )
    {
      m_colorDialog->hide();
      return true;
    }
    else
      return false;
  }
  else if ( watched == m_glRA )
#endif
  {
    if ( event->type() == QEvent::Paint )
    {
      updateRA();
      return true;
    }
    else if ( dynamic_cast<QMouseEvent*>(event) )
    {
      bool result = m_glRAAlgos->getSceneManager()->processEvent( translateEvent((QMouseEvent*)event) );
      updateRA();
      return result;
    }
    else
      return false;
  }
  else
    return QObject::eventFilter( watched, event );
}

//------------------------------------------------------------------------------
void
Ivt_DirectionalLightEditor::updateRA()
{
  m_oglContext->bind();
  {
    m_glRAAlgos->actualRedraw();
    m_oglContext->swapBuffers();
  }
  m_oglContext->unbind();
}

//------------------------------------------------------------------------------
const SoEvent*
Ivt_DirectionalLightEditor::translateEvent( QMouseEvent* event )
{
  SoEvent* tevent = NULL;

  switch (event->type()) 
  {
  case QEvent::MouseButtonPress :
    tevent = translateButtonEvent( event, SoButtonEvent::DOWN );
    break;
  case QEvent::MouseButtonDblClick :
    tevent = translateButtonEvent( event, SoButtonEvent::DBCLK );
    break;
  case QEvent::MouseButtonRelease :
    tevent = translateButtonEvent( event, SoButtonEvent::UP );
    break;
  case QEvent::MouseMove :
    tevent = translateMotionEvent( event );
    break;
  default:
    break;
  }
  tevent->setPosition( SbVec2f(event->x(), 
                       (((float)glRenderAreaContainer->size().height()) - 21.f) - event->y()) );

  return tevent;
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::intensityLECB()
{
  double value = intensityLE->text().toDouble();
  if ( value > 1 )
    value = 1.00f;
  
  intensitySlider->setValue( (int)(value*100) );
}

//------------------------------------------------------------------------------
void
Ivt_DirectionalLightEditor::colorDialogCB( const QColor& currentColor )
{
  float r = ((float)currentColor.red())/255.00f;
  float g = ((float)currentColor.green())/255.00f;
  float b = ((float)currentColor.blue())/255.00f;
  SbColor col = SbColor( r, g, b );

  if ( currentColor.isValid() )
  {
    QPalette p = intensitySlider->palette();
    p.setColor( QPalette::Window, currentColor );
    intensitySlider->setPalette( p );
    
    if ( m_dirLight )
      m_dirLight->color.setValue( col );

    // Update our local light
    m_ignoreCallback = TRUE;
    m_dirLightManip->color.setValue( col );
    m_ignoreCallback = FALSE;

    // Invoke the callbacks with the new values
    m_callbackList->invokeCallbacks( m_dirLightManip );
  }

  updateRA();
}

//------------------------------------------------------------------------------
void
Ivt_DirectionalLightEditor::initGL()
{
  m_glRAAlgos = new SoGuiAlgoViewers;
  m_glRAAlgos->ref(); // Ref removed from SoGui so it must be made explicitly
  m_glRAAlgos->setHeadlight( FALSE ); // No need for head light

  // Part of the following code is taken from SoQtDirLitEd.cxx
  m_dirLight = NULL;
  m_callbackList = new SoCallbackList;
  
  // create the dir light manip with our own geom
  SoInput in;
  in.setBuffer( (void*)m_editorContentBuffer, (size_t)strlen(m_editorContentBuffer) );
  SoNode* ecb;
  SoDB::read( &in, ecb );
  m_dirLightManip = new SoDirectionalLightManip();
  m_dirLightManip->ref();
  
  // Set the rotator to be our arrow.
  SoDragger* liteDragger = m_dirLightManip->getDragger();
  liteDragger->setPart( "rotator.rotator", SoNode::getByName("dirLightEditorRotator") );
  liteDragger->setPart( "rotator.rotatorActive", SoNode::getByName("dirLightEditorRotatorActive") );
  
  // Set the other parts to NULL
  SoSeparator* dummySep = new SoSeparator;
  dummySep->ref();
  liteDragger->setPart( "translator.xTranslator.translator", dummySep );
  liteDragger->setPart( "translator.yTranslator.translator", dummySep );
  liteDragger->setPart( "translator.zTranslator.translator", dummySep );
  liteDragger->setPart( "translator.xTranslator.translatorActive", dummySep );
  liteDragger->setPart( "translator.yTranslator.translatorActive", dummySep );
  liteDragger->setPart( "translator.zTranslator.translatorActive", dummySep );
  liteDragger->setPart( "translator.yzTranslator.translator", dummySep );
  liteDragger->setPart( "translator.xzTranslator.translator", dummySep );
  liteDragger->setPart( "translator.xyTranslator.translator", dummySep );
  liteDragger->setPart( "translator.yzTranslator.translatorActive", dummySep );
  liteDragger->setPart( "translator.xzTranslator.translatorActive", dummySep );
  liteDragger->setPart( "translator.xyTranslator.translatorActive", dummySep );
  liteDragger->setPart( "rotator.feedback", dummySep );
  liteDragger->setPart( "rotator.feedbackActive", dummySep );
  dummySep->unref();
  liteDragger->addValueChangedCallback( Ivt_DirectionalLightEditor::dirLightManipCB, this );
  
  // Callbacks
  m_ignoreCallback = FALSE;
  
  // Set up the camera sensor - this will keep our camera oriented to 
  // match the scene the light is in.
  m_cameraSensor = new SoNodeSensor;
  m_cameraSensor->setFunction( (void(*)(void*, SoSensor*))Ivt_DirectionalLightEditor::cameraSensorCB );
  m_cameraSensor->setData( (void*)this );
  m_cameraToWatch = NULL;
  
  // Local scene graph for direct manipulation
  m_root = new SoSeparator;
  m_editorCamera = new SoPerspectiveCamera;
  m_litStuff = new SoSeparator;
  SoEnvironment* environment = new SoEnvironment;
  SoMaterial* material = new SoMaterial;
  SoComplexity* complexity = new SoComplexity;
  SoSphere* sphere = new SoSphere;
  m_litStuff->addChild( environment );
  m_litStuff->addChild( material );
  m_litStuff->addChild( complexity );
  m_litStuff->addChild( sphere );
  
  m_root->ref();
  m_root->addChild( m_editorCamera );
  m_root->addChild( m_dirLightManip );
  m_root->addChild( m_litStuff );
  
  // Some ambient light
  environment->ambientColor.setValue( 1.0, 1.0, 1.0 );
  environment->ambientIntensity.setValue( 0.5 );
  
  // Let's have an interesting material! ...
  material->ambientColor.setValue( .2f, .2f, .2f );
  material->diffuseColor.setValue( .55f, .55f, .55f );
  material->specularColor.setValue( .70f, .70f, .70f );
  material->shininess.setValue( 1.0 );
  
  // ... and some complexity
  complexity->value.setValue( 0.6f );

  // Set up rendering context
  m_glRA->makeCurrent();

  m_oglContext = new SoGLContext(true);
  m_oglContext->bind();
  {
    m_glRAAlgos->resetRenderAction();
    m_glRAAlgos->getSceneManager()->setNeedToSetViewport( 0 );

    glEnable( GL_DEPTH_TEST );
  }
  m_oglContext->unbind();

  // Set up the scene and the rendering callback
  SoSceneManager* sceneManager = m_glRAAlgos->getSceneManager();
  sceneManager->setRenderCallback( Ivt_DirectionalLightEditor::editorRenderCB, this );
  SbViewportRegion vpRegion( SbVec2s(glRenderAreaContainer->size().width()-15, glRenderAreaContainer->size().height()-20) );
  sceneManager->setViewportRegion( vpRegion );
  m_glRAAlgos->setSceneGraph( m_root );
  m_editorCamera->viewAll( m_litStuff, SbViewportRegion(SbVec2s(glRenderAreaContainer->size().width(), 
                                                                glRenderAreaContainer->size().height())), 2.0 );
}

//------------------------------------------------------------------------------
void
Ivt_DirectionalLightEditor::copyLight( SoDirectionalLight* dst, const SoDirectionalLight* src )
{
  dst->color.setValue( src->color.getValue() );
  dst->intensity.setValue( src->intensity.getValue() );
  dst->direction.setValue( src->direction.getValue() );
}

//------------------------------------------------------------------------------
void
Ivt_DirectionalLightEditor::updateLocalComponents()
{
  float* col = (float*)(m_dirLightManip->color.getValue().getValue());
  QColor color = QColor( (int)(col[0]*255), (int)(col[1]*255), (int)(col[2]*255) );
#if (QT_VERSION < 0x040500)
#else
  m_colorDialog->setCurrentColor( color );
#endif

  intensitySlider->setValue( (int)(m_dirLightManip->intensity.getValue()*100) );
  QPalette p = intensitySlider->palette();
  p.setColor( QPalette::Window, color );
  intensitySlider->setPalette( p );
}

//------------------------------------------------------------------------------
SoMouseButtonEvent*
Ivt_DirectionalLightEditor::translateButtonEvent( QMouseEvent* be, SoButtonEvent::State whichState )
{
  SoMouseButtonEvent::Button whichButton;

  if ( be->button() == Qt::LeftButton )
    whichButton = SoMouseButtonEvent::BUTTON1;
  else if ( be->button() == Qt::MiddleButton )
    whichButton = SoMouseButtonEvent::BUTTON2;
  else if ( be->button() == Qt::RightButton )
    whichButton = SoMouseButtonEvent::BUTTON3;
  else 
    whichButton = SoMouseButtonEvent::ANY;

  m_buttonEvent->setTime( SoDB::getCurrentTime() );
  m_buttonEvent->setShiftDown( be->modifiers() & Qt::ShiftModifier );
  m_buttonEvent->setCtrlDown( be->modifiers() & Qt::ControlModifier );
  m_buttonEvent->setAltDown( be->modifiers() & Qt::AltModifier );

  m_buttonEvent->setState( whichState );
  m_buttonEvent->setButton( whichButton );

  return m_buttonEvent;
}

//------------------------------------------------------------------------------
SoLocation2Event*
Ivt_DirectionalLightEditor::translateMotionEvent( QMouseEvent* me )
{
  m_loc2Event->setTime( SoDB::getCurrentTime() );
  m_loc2Event->setShiftDown( me->modifiers() & Qt::ShiftModifier );
  m_loc2Event->setCtrlDown( me->modifiers() & Qt::ControlModifier );
  m_loc2Event->setAltDown( me->modifiers() & Qt::AltModifier );
  m_loc2Event->setButton1Down( me->modifiers() & Qt::LeftButton );

  return m_loc2Event;
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::cameraSensorCB( Ivt_DirectionalLightEditor* editor, SoSensor* /*sensor*/ )
{
  if ( editor->m_cameraToWatch )
  {
    SbRotation newRot = editor->m_cameraToWatch->orientation.getValue();
    editor->m_editorCamera->orientation.setValue( newRot );

    editor->m_editorCamera->viewAll( editor->m_litStuff, 
                                     SbViewportRegion(SbVec2s(editor->glRenderAreaContainer->size().width(), 
                                                              editor->glRenderAreaContainer->size().height())), 2.0 );

    editor->updateRA();
  }
}

//------------------------------------------------------------------------------
void
Ivt_DirectionalLightEditor::dirLightManipCB(void* data, SoDragger* /*dragger*/ )
{
  Ivt_DirectionalLightEditor* ed = (Ivt_DirectionalLightEditor*)data;
  
  if ( ed->m_ignoreCallback ) 
    return;
  
  // Update the attached light, if it exists
  if ( ed->m_dirLight != NULL )
    ed->m_dirLight->direction.setValue( ed->m_dirLightManip->direction.getValue() );
  
  // Invoke the callbacks
  ed->m_callbackList->invokeCallbacks( ed->m_dirLightManip );
}

//------------------------------------------------------------------------------
void
Ivt_DirectionalLightEditor::editorRenderCB( void* data, SoSceneManager* /*mgr*/ )
{
  Ivt_DirectionalLightEditor* editor = (Ivt_DirectionalLightEditor*)data;

  if ( editor )
    editor->updateRA();
}

//------------------------------------------------------------------------------
void
Ivt_DirectionalLightEditor::activate()
{
  IvtShell::getInstance()->registerViewMenu( this ); // Register this editor to the view menu of IvTune

  IvtEditor::activate();
  show(); // Show this editor.

  setUp(); // Initialize this editor regarding the selected node
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::deactivate()
{
  IvtEditor::deactivate(); // Call it first

  IvtShell::getInstance()->unregisterViewMenu( this ); // Unregister this editor from the view menu of IvTune
 
  hide(); // Hide this editor
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::hide()
{
  IvtEditor::hide(); // Call it first
  // Hide the GUI elements of this editor
 
  m_container->hide();
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::load()
{
  // Containers instanciation
  m_container = new QFrame();
  m_widget = m_container;
  
  // Instantiation of the editor
  setupUi( m_container );

  // Create the gl render area
  m_glRA = new QOpenGLWidget( glRenderAreaContainer );
  m_glRA->installEventFilter( this );
  m_glRA->setAttribute( Qt::WA_NoSystemBackground );
  m_glRA->setAttribute( Qt::WA_PaintOnScreen );
  QVBoxLayout* glLayout = new QVBoxLayout( glRenderAreaContainer );
  glLayout->addWidget( m_glRA );

  // Inventor events init for the GL render area
  m_buttonEvent = new SoMouseButtonEvent;
  m_loc2Event = new SoLocation2Event;

  // Default color and color editor init;
#if (QT_VERSION < 0x040500)
#else
  m_colorDialog = new QColorDialog( Qt::white );
  m_colorDialog->installEventFilter( this );
  m_colorDialog->hide();
#endif
  intensitySlider->setPalette( Qt::white );
  
  intensitySlider->setAutoFillBackground( true );

  // Create connections
#if (QT_VERSION < 0x040500)
#else
  connect( m_colorDialog, SIGNAL(currentColorChanged(const QColor&)), this, SLOT(colorDialogCB(const QColor&)) );
#endif
  connect( colorDialogButton, SIGNAL(clicked()), this, SLOT(openColorEditor()) );
  connect( intensitySlider, SIGNAL(valueChanged(int)), this, SLOT(intensitySliderCB(int)) );
  connect( intensityLE, SIGNAL(editingFinished()), this, SLOT(intensityLECB()) );

  // Retrieve user scene graph
  m_usg = IvtServices::getInstance()->getSceneGraph();

  // Init GL render area
  initGL();

  hide(); // Hide this editor as it will be shown when it will be invoked
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::show()
{
  IvtEditor::show();

  // Show the GUI elements of this editor
  m_container->show();
}

//------------------------------------------------------------------------------
void 
Ivt_DirectionalLightEditor::unload()
{
  // Delete instanciated objects in load()
  delete m_container;
}

#ifdef _WIN32
#pragma warning( pop )
#endif
