/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Tristan MEHAMLI (Dec 2010)
**=======================================================================*/

#include <Ivt_ProcessingMenu.h>
#include <Ivt_ProcessingMenuActionsManager.h>

#include <IvTune/IvTuneExtender/IvtShell.h>
#include <IvTune/IvTuneExtender/IvtServices.h>

#include <Inventor/Gui/viewers/SoGuiAlgoViewers.h>

#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoShape.h>

#include <QAction>
#include <QMenu>

//------------------------------------------------------------------------------
Ivt_ProcessingMenu::Ivt_ProcessingMenu() 
: IvtExtension()
{
  m_info.name = "Processing Menu";
  m_info.author ="Thermo Fisher Scientific";
  m_info.description = "A menu based on the processing menu from the demo QtLargeModelViewer.";
  m_info.version = "1.0.0";
}

//------------------------------------------------------------------------------
Ivt_ProcessingMenu::~Ivt_ProcessingMenu()
{
}

//------------------------------------------------------------------------------
void
Ivt_ProcessingMenu::activate()
{
  IvtShell::getInstance()->registerContextualMenu( SoNode::getClassTypeId(), false, m_processingMenu ); // Register the menu
  IvtShell::getInstance()->registerViewMenu( this ); // Register this extension to the view menu of IvTune

  IvtExtension::activate();
}

//------------------------------------------------------------------------------
void
Ivt_ProcessingMenu::deactivate()
{
  IvtExtension::deactivate();
  // Unregister the menu and this extension
  IvtShell::getInstance()->unregisterContextualMenu( m_processingMenu );
  IvtShell::getInstance()->unregisterViewMenu( this );
}

//------------------------------------------------------------------------------
void
Ivt_ProcessingMenu::hide()
{
  IvtExtension::hide();
  m_processingMenu->hide(); // Hide the menu
}

//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::load()
{
  // Inits
  // Create the actions manager for the processing menu
  m_acma = new Ivt_ProcessingMenuActionsManager();

  // Processing menu
  buildProcessingMenu();
}

//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::show()
{
  IvtExtension::show();
  m_processingMenu->show(); // Show the menu
}

//------------------------------------------------------------------------------
void
Ivt_ProcessingMenu::unload()
{
  // Delete what was instanced in load()
  delete m_acma;
  m_acma = NULL;

  delete m_processingMenu;
  m_processingMenu = NULL;
}

//------------------------------------------------------------------------------
// Processing menu slots
//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::pmReorganizeProcess()
{
  if ( m_acma->reorganizeProcess(m_currentProcessedNode) )
  {
    // Apply changes on the menu items according to the result
    if ( !m_reorganizeItem->isChecked() )
      m_reorganizeItem->setChecked( true );
    m_reorganizeItem->setEnabled( false );
    if ( m_shapeSimplifyItem->isChecked() )
      m_shapeSimplifyItem->setChecked( false );
    m_shapeSimplifyItem->setEnabled( false );
    if ( m_globalSimplifyItem->isChecked() )
      m_globalSimplifyItem->setChecked( false );
    m_globalSimplifyItem->setEnabled( false );
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::pmShapeSimplifyProcess()
{
  m_acma->shapeSimplifyProcess( m_currentProcessedNode );

  // Apply changes on the menu items according to the result
  if ( !m_shapeSimplifyTSItem->isChecked() )
    m_shapeSimplifyItem->setChecked( true );

  m_reorganizeItem->setEnabled( false );
  m_shapeSimplifyItem->setEnabled( false );
  m_shapeSimplifyTSItem->setEnabled( false );
  m_globalSimplifyItem->setEnabled( false );
  m_globalSimplifyTSItem->setEnabled( false );
}

//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::pmGlobalSimplifyProcess()
{
  if ( m_acma->globalSimplifyProcess(m_currentProcessedNode) )
  {
    // Apply changes on the menu items according to the result
    if ( !m_globalSimplifyTSItem->isChecked() )
      m_globalSimplifyItem->setChecked( true );

    m_reorganizeItem->setEnabled( false );
    m_shapeSimplifyItem->setEnabled( false );
    m_shapeSimplifyTSItem->setEnabled( false );
    m_globalSimplifyItem->setEnabled( false );
    m_globalSimplifyTSItem->setEnabled( false );
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::pmShapeSimplifyTSProcess()
{
  m_acma->shapeSimplifyProcess( m_currentProcessedNode, true );

  // Apply changes on the menu items according to the result
  if ( !m_shapeSimplifyTSItem->isChecked() )
    m_shapeSimplifyTSItem->setChecked( true );

  if ( !m_reorganizeItem->isChecked() )
    m_reorganizeItem->setChecked( true );
  m_reorganizeItem->setEnabled( false );
  if ( m_shapeSimplifyItem->isChecked() )
    m_shapeSimplifyItem->setChecked( false );
  m_shapeSimplifyItem->setEnabled( false );
  m_shapeSimplifyTSItem->setEnabled( false );
  if ( m_globalSimplifyItem->isChecked() )
    m_globalSimplifyItem->setChecked( false );
  m_globalSimplifyItem->setEnabled( false );
  m_globalSimplifyTSItem->setEnabled( false );
}

//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::pmGlobalSimplifyTSProcess()
{
  if ( m_acma->globalSimplifyProcess(m_currentProcessedNode, true) )
  {
    // Apply changes on the menu items according to the result
    if ( !m_globalSimplifyTSItem->isChecked() )
      m_globalSimplifyTSItem->setChecked( true );

    if ( !m_reorganizeItem->isChecked() )
      m_reorganizeItem->setChecked( true );
    m_reorganizeItem->setEnabled( false );
    if ( m_shapeSimplifyItem->isChecked() )
      m_shapeSimplifyItem->setChecked( false );
    m_shapeSimplifyItem->setEnabled( false );
    m_shapeSimplifyTSItem->setEnabled( false );
    if ( m_globalSimplifyItem->isChecked() )
      m_globalSimplifyItem->setChecked( false );
    m_globalSimplifyItem->setEnabled( false );
    m_globalSimplifyTSItem->setEnabled( false );
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::pmSplitShapesProcess()
{
  // Open Split Shapes dialog
  m_acma->splitShapesProcess( m_currentProcessedNode );
}

//------------------------------------------------------------------------------
void
Ivt_ProcessingMenu::pmCheckNode( SoFieldContainer* fc )
{
  SoNode* node = dynamic_cast<SoNode*>(fc);
  if ( node != m_currentProcessedNode ) // A new node is about to be processed
  {
    m_currentProcessedNode = node;
    processingMenuReset(); // Reset the processing menu
  }
}

//------------------------------------------------------------------------------
// Processing menu from the Large Model Viewer
//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::buildProcessingMenu()
{
  m_currentProcessedNode = NULL;
  
  m_processingMenu = new QMenu( tr("Processing") );

  // Create actions
  m_reorganizeItem = m_processingMenu->addAction( tr("Reorganize (triangle strips)"), this, SLOT(pmReorganizeProcess()) );
  m_reorganizeItem->setCheckable( true );
  m_shapeSimplifyItem = m_processingMenu->addAction( tr("Shape Simplify"), this, SLOT(pmShapeSimplifyProcess()) );
  m_shapeSimplifyItem->setCheckable( true );
  m_globalSimplifyItem = m_processingMenu->addAction( tr("Global Simplify"), this, SLOT(pmGlobalSimplifyProcess()) );
  m_globalSimplifyItem->setCheckable( true );
  m_shapeSimplifyTSItem = m_processingMenu->addAction( tr("Shape Simplify (triangle strips)"), this, SLOT(pmShapeSimplifyTSProcess()) );
  m_shapeSimplifyTSItem->setCheckable( true );
  m_globalSimplifyTSItem = m_processingMenu->addAction( tr("Global Simplify (triangle strips)"), this, SLOT(pmGlobalSimplifyTSProcess()) );
  m_globalSimplifyTSItem->setCheckable( true );
  m_processingMenu->addSeparator();
  m_splitShapeItem = m_processingMenu->addAction( tr("Split Shapes..."), this, SLOT(pmSplitShapesProcess()) );

  // Connect this to the treeview to check whether it is a new node that will be processed
  connect( IvtServices::getInstance(), SIGNAL(selectedNodeChanged(SoFieldContainer*)), this, SLOT(pmCheckNode(SoFieldContainer*)) );
}

//------------------------------------------------------------------------------
void 
Ivt_ProcessingMenu::processingMenuReset()
{
  if ( m_currentProcessedNode &&
       m_currentProcessedNode->isOfType(SoGroup::getClassTypeId()) ) // SoGroup and classes derived from SoGroup
  {
    m_reorganizeItem->setChecked( false );
    m_reorganizeItem->setEnabled( true );
  }
  else
  {
    m_reorganizeItem->setChecked( false );
    m_reorganizeItem->setEnabled( false );
  }
  if ( m_currentProcessedNode && 
       m_currentProcessedNode->isOfType(SoShape::getClassTypeId()) ) // SoShape and classes derived from SoShape
  {
    m_shapeSimplifyItem->setChecked( false );
    m_shapeSimplifyItem->setEnabled( true );
    m_shapeSimplifyTSItem->setChecked( false );
    m_shapeSimplifyTSItem->setEnabled( true );
  }
  else
  {
    m_shapeSimplifyItem->setChecked( false );
    m_shapeSimplifyItem->setEnabled( false );
    m_shapeSimplifyTSItem->setChecked( false );
    m_shapeSimplifyTSItem->setEnabled( false );
  }
  m_globalSimplifyItem->setChecked( false );
  m_globalSimplifyItem->setEnabled( true );
  m_globalSimplifyTSItem->setChecked( false );
  m_globalSimplifyTSItem->setEnabled( true );
}
