/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T.MEHAMLI (Dec 2010)
**=======================================================================*/

#include <Ivt_ShaderEditor.h>
#include <Ivt_ShaderSyntaxHighlight.h>

#include <IvTune/IvTuneExtender/IvtShell.h>
#include <IvTune/IvTuneExtender/IvtServices.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <Inventor/nodes/SoShaderObject.h>

#include <QtCore/QFile>

#include <QDialog>
#include <QDockWidget>
#include <QFileDialog>
#include <QFrame>
#include <QMessageBox>

//------------------------------------------------------------------------------
Ivt_ShaderEditor::Ivt_ShaderEditor()
: IvtEditor()
{
  m_info.name = "Shader Editor";
  m_info.author ="Thermo Fisher Scientific";
  m_info.description = "An editor for shader programs.";
  m_info.version = "1.0.0";

  m_handledType = SoShaderObject::getClassTypeId();
}

//------------------------------------------------------------------------------
Ivt_ShaderEditor::~Ivt_ShaderEditor()
{
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::setUp()
{
  m_doItOnce = false;

  m_shaderNode = (SoShaderObject*) m_editedObject;
  
  SbString fileBaseName;
  fileBaseName.makeEmpty();

  // Load the shader and display it in the editor
  if ( m_shaderNode->sourceType.getValue() == SoShaderObject::FILENAME )
  {
    SbString fileName = m_shaderNode->getFileAbsolutePath( m_shaderNode->sourceProgram.getValue() );
    m_fileName = fileName.toLatin1();

    QFile file( m_fileName );
    if ( file.open(QFile::ReadOnly | QFile::Text) )
      m_fileBuffer = file.readAll();

    fileBaseName = SbFileHelper::getBaseName( m_fileName.toLatin1().data() );
  }
  else
  {
    m_fileBuffer = QString( m_shaderNode->sourceProgram.getValue().toLatin1() );

    m_fileName = "From buffer";
  }

  if ( !m_fileBuffer.isEmpty() )
  {
    if ( !fileBaseName.isEmpty() ) // From file
      fileGroupBox->setTitle( fileBaseName.toLatin1() );
    else
      fileGroupBox->setTitle( m_fileName );

    textEditor->setPlainText( m_fileBuffer );
  }

  m_doItOnce = true;
}

//------------------------------------------------------------------------------
// Slots
// Buttons callbacks
//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::clear()
{
  textEditor->clear();
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::copy()
{
  textEditor->copy();
  
  pasteButton->setEnabled( true );
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::cut()
{
  textEditor->cut();

  pasteButton->setEnabled( true );
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::paste()
{
  textEditor->paste();
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::redo()
{
  textEditor->redo();
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::reload()
{
  QString filename = fileGroupBox->title();
  filename.remove( "*" ); // Remove the '*' indicating that the shader was modified
  fileGroupBox->setTitle( filename );

  textEditor->setPlainText( m_fileBuffer );
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::save()
{
  QString filename = fileGroupBox->title();
  filename.remove( "*" ); // Remove the '*' indicating that the shader was modified
  fileGroupBox->setTitle( filename );
  m_doItOnce = true;
  
  // Run a message box to determine what's wanted by the user
  QMessageBox overWriteFileDialog( m_container );
  overWriteFileDialog.setModal( true );
  overWriteFileDialog.setText( "The shader has been modified." );

  if ( m_shaderNode->sourceType.getValue() == SoShaderObject::FILENAME ) // The shader can be overwritten or saved in a new file
  {
    overWriteFileDialog.setInformativeText( "Do you want to overwrite it?" );
    overWriteFileDialog.setStandardButtons( QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel );
    overWriteFileDialog.setDefaultButton( QMessageBox::Cancel );
    int ret = overWriteFileDialog.exec();

    switch (ret)
    {
    case QMessageBox::Yes:
      break; // Nothing to do, just save it
    case QMessageBox::No:
      {
        m_fileName.clear();
        QString fullPath = SbFileHelper::getDirName(m_fileName.toLatin1().data()).toLatin1();
        m_fileName = QFileDialog::getSaveFileName( m_container, tr("Save the shader"), fullPath, tr("GLSL Shaders (*.glsl)") ); // Get the new file name
        if ( !m_fileName.isEmpty() ) // Otherwise the user has canceled the file dialog
          m_shaderNode->sourceProgram.setValue( m_fileName.toLatin1().data() ); // Make the source point to it
        break;
      }
    case QMessageBox::Cancel:
      return;
    default:
      return;
    }

    // Save the file
    QFile file( m_fileName );
    if ( file.open(QFile::ReadWrite | QFile::Text) )
      file.write( textEditor->toPlainText().toLatin1().data() );
  }
  else
  {
    overWriteFileDialog.setInformativeText( "Do you want to save it in a new file?" );
    int ret = overWriteFileDialog.exec();

    switch (ret)
    {
    case QMessageBox::Yes:
      {
        m_fileName.clear();
        m_fileName = QFileDialog::getSaveFileName( m_container, tr("Save the shader"), ".", tr("GLSL Shaders (*.glsl)") );
        QFile file( m_fileName );
        if ( file.open(QFile::ReadWrite | QFile::Text) )
          file.write( textEditor->toPlainText().toLatin1().data() );

        if ( !m_fileName.isEmpty() ) // Otherwise the user has canceled the file dialog
        {
          m_shaderNode->sourceProgram.setValue( m_fileName.toLatin1().data() ); // Make the source point to it
          m_shaderNode->sourceType.setValue( SoShaderObject::FILENAME ); // Change the source type
        }
        break;
      }
    case QMessageBox::No:
       m_shaderNode->sourceProgram.setValue( textEditor->toPlainText().toLatin1().data() );
      break;
    case QMessageBox::Cancel:
      return;
    default:
      return;
    }
  }

  m_fileBuffer.clear();
  m_fileBuffer = textEditor->toPlainText();

  // Setup button states
  reloadButton->setEnabled( false );
  saveButton->setEnabled( false );
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::undo()
{
  textEditor->undo();
}

//------------------------------------------------------------------------------
// Edition callbacks
//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::setCopyAvailable( bool available )
{
  copyButton->setEnabled( available );
  cutButton->setEnabled( available );
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::setFileEdited()
{
  if ( m_doItOnce )
  {
    QString filename = fileGroupBox->title();
    filename.append( "*" );
    fileGroupBox->setTitle( filename );

    saveButton->setEnabled( true );
    reloadButton->setEnabled( true );

    m_doItOnce = false;
  }
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::setRedoAvailable( bool available )
{
  redoButton->setEnabled( available );
}

//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::setUndoAvailable( bool available )
{
  undoButton->setEnabled( available );
}

//------------------------------------------------------------------------------
// Redifined methods from IvtExtension, IvtEditor and IvtNodeEditor
//------------------------------------------------------------------------------
void
Ivt_ShaderEditor::activate()
{
  IvtShell::getInstance()->registerViewMenu( this ); // Register this editor to the view menu of IvTune

  IvtEditor::activate();

  // This editor will be invoked as soon as a node of the same type of SoTransferFunction is selected in IvTune's tree view
  show(); // Show this editor.

  setUp(); // Initialize this editor regarding the selected node
}

//------------------------------------------------------------------------------
void 
Ivt_ShaderEditor::deactivate()
{
  IvtEditor::deactivate();

  IvtShell::getInstance()->unregisterViewMenu( this ); // Unregister this editor from the view menu of IvTune
  // This editor will be revoked as soon as a node which is not of type SoTransferFunction is selected in IvTune's tree view
 
  hide(); // Hide this editor
}

//------------------------------------------------------------------------------
void 
Ivt_ShaderEditor::hide()
{
  IvtEditor::hide();
  // Hide the GUI elements of this editor
 
  m_container->hide();
}

//------------------------------------------------------------------------------
void 
Ivt_ShaderEditor::load()
{
  // Containers instanciation
  m_container = new QFrame();
  m_widget = m_container;
  
  // Instanciation of the editor
  setupUi( m_container );

  // Setup text editor
  QFont font;
  font.setFamily( "Courier" );
  font.setFixedPitch( true );
  font.setPointSize( 10 );

  textEditor->setFont( font );
  m_syntaxHighLight = new Ivt_ShaderSyntaxHighlight( textEditor->document() );

  // Setup button states
  copyButton->setEnabled( false );
  cutButton->setEnabled( false );
  pasteButton->setEnabled( false );
  redoButton->setEnabled( false );
  reloadButton->setEnabled( false );
  saveButton->setEnabled( false );
  undoButton->setEnabled( false );

  // Create connections
  // Buttons callbacks
  connect( clearButton, SIGNAL(clicked()), this, SLOT(clear()) );
  connect( copyButton, SIGNAL(clicked()), this, SLOT(copy()) );
  connect( cutButton, SIGNAL(clicked()), this, SLOT(cut()) );
  connect( pasteButton, SIGNAL(clicked()), this, SLOT(paste()) );
  connect( redoButton, SIGNAL(clicked()), this, SLOT(redo()) );
  connect( reloadButton, SIGNAL(clicked()), this, SLOT(reload()) );
  connect( saveButton, SIGNAL(clicked()), this, SLOT(save()) );
  connect( undoButton, SIGNAL(clicked()), this, SLOT(undo()) );
  // Edition callbacks
  connect( textEditor, SIGNAL(copyAvailable(bool)), this, SLOT(setCopyAvailable(bool)) );
  connect( textEditor, SIGNAL(redoAvailable(bool)), this, SLOT(setRedoAvailable(bool)) );
  connect( textEditor, SIGNAL(textChanged()), this, SLOT(setFileEdited()) );
  connect( textEditor, SIGNAL(undoAvailable(bool)), this, SLOT(setUndoAvailable(bool)) );

  // Hidden to prevent artifacts in its parent
  hide();

  m_doItOnce = false;
}

//------------------------------------------------------------------------------
void 
Ivt_ShaderEditor::show()
{
  IvtEditor::show();
  // Show the GUI elements of this editor
 
  m_container->show();
}

//------------------------------------------------------------------------------
void 
Ivt_ShaderEditor::unload()
{
  // Delete instanciated objects in load()
  delete m_container;
}
