/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Jerome Hummel (Nov 2009)
** Modified by : T.MEHAMLI (Dec 2010)
**=======================================================================*/

#include <Ivt_AlphaCurveEditor.h>

#include <IvTune/IvTuneExtender/IvtShell.h>
#include <IvTune/IvTuneExtender/IvtServices.h>

#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/SoLDMLargeDataManagement.h>

#include <QtCore/QEvent>
#include <QDockWidget>
#include <QFrame>
#include <QHBoxLayout>
#include <QImage>
#include <QLabel>
#include <QMouseEvent>
#include <QPainter>
#include <QPaintEvent>
#include <QPixmap>
#include <QResizeEvent>
#include <QScrollArea>

#define PENWIDTH 3 //Corresponds to one entry color value width in the colormap
#define IMAGEHEIGHT 100
#define WIDGETBORDER 10
#define KEYSIZE 7
#define CHECKERSIZE 10

#ifdef _WIN32
#pragma warning( push )
#pragma warning( disable:4996 )
#endif

//------------------------------------------------------------------------------
Ivt_AlphaCurveEditor::Ivt_AlphaCurveEditor()
  : IvtEditor()
  , m_leftKey( NULL )
  , m_pickedKey( NULL )
  , m_rightKey( NULL )
{
  SoLDMLargeDataManagement::init();

  m_info.name = "Alpha Curve Editor";
  m_info.author ="Thermo Fisher Scientific";
  m_info.description = "An editor for color maps alpha values edition.";
  m_info.version = "1.0.0";

  m_handledType = SoTransferFunction::getClassTypeId();
}

//------------------------------------------------------------------------------
Ivt_AlphaCurveEditor::~Ivt_AlphaCurveEditor()
{
  delete [] m_xColorCoord;
  SoLDMLargeDataManagement::finish();
}

//------------------------------------------------------------------------------
void 
Ivt_AlphaCurveEditor::createColormapImage()
{
  m_colorList.clear();

  int num = m_transferFunction->actualColorMap.getNum();
  //for now assume its RGBA
  const float* colors = m_transferFunction->actualColorMap.getValues( 0 );
  for ( int i = 0; i < num; i += 4 )
  {
    float r = colors[i]*255;
    float g = colors[i+1]*255;
    float b = colors[i+2]*255;
    float a = colors[i+3]*255;
    QColor color( (int)r,(int)g,(int)b,(int)a );
    m_colorList.append( color );
  }
  //Create checkerboard:
  QPixmap checker( CHECKERSIZE*2, CHECKERSIZE*2 );
  QPainter p( &checker );
  QBrush brushw( Qt::white );
  p.setBrush( brushw );
  p.setPen( Qt::NoPen );
  p.drawRect( QRect(0,0, CHECKERSIZE*2, CHECKERSIZE*2) );
  QBrush brushb( Qt::gray );
  p.setBrush( brushb );
  p.drawRect( QRect(0,0, CHECKERSIZE, CHECKERSIZE) );
  p.drawRect( QRect(CHECKERSIZE, CHECKERSIZE, CHECKERSIZE, CHECKERSIZE) );
  p.end();

  //Create colormap image:
  m_editorWidth = m_container->size().width() - WIDGETBORDER*4; // Width of the colormap image container
  m_penWidth = m_editorWidth / m_numColors;
  if ( m_penWidth < 1 )
    m_penWidth = 1;
  QPixmap pixmap( m_editorWidth, IMAGEHEIGHT );
  QPainter painter( &pixmap );
  QBrush brush;
  brush.setTexture( checker );
  brush.setStyle( Qt::TexturePattern );
  painter.setBrush( brush );
  painter.drawRect( 0 ,0, m_editorWidth, IMAGEHEIGHT );

  painter.setBrush( Qt::NoBrush );

  QColor color;
  int x = 0, colorIndex = 0;
  float ratio = ((float)m_numColors/(float)(m_editorWidth/m_penWidth));
  float step = 0.0;
  m_xColorCoord = new int[m_numColors];

  for ( int i = 0; i < m_editorWidth; i += m_penWidth )
  {
    QPen pen( m_colorList[colorIndex] );
    pen.setWidth( m_penWidth );

    painter.setPen( pen );
    painter.drawLine( x, 0, x, IMAGEHEIGHT );

    x += m_penWidth;

    // Handle the resizing of the editor by keeping homogenous color repartition
    step += ratio;
    if ( step > 1.0 )
    {
      step -= 1.0;
      m_xColorCoord[colorIndex] = x;
      colorIndex++;
    }
    // Avoid overflow
    if ( colorIndex >= m_numColors )
      colorIndex = m_numColors - 1;
  }
  painter.end();
  m_xColorCoord[colorIndex] = x;

  m_colormapImage = pixmap.toImage();
}

//------------------------------------------------------------------------------
void 
Ivt_AlphaCurveEditor::computeAlphaKeys()
{
  m_keys.clear();
  int num = m_transferFunction->actualColorMap.getNum();
  //for now assume it is RGBA
  const float* colors = m_transferFunction->actualColorMap.getValues( 0 );

  AlphaKey* key = new AlphaKey();
  key->m_alpha = colors[4];
  key->m_x = 0; 
  key->m_y = key->m_alpha * IMAGEHEIGHT; 
  key->m_sCMSize = m_editorWidth;
  key->m_sx = 0;
  m_keys.append( key );

  int keyindex = 0;
  int x = 0;

  float currentalpha;
  float ratio = (float)m_editorWidth / (float)m_numColors;
  for ( int i = 4; i < num; i += 4 )
  {
    x = m_xColorCoord[(i/4) -1];
    float a = colors[i+3];

    AlphaKey* newkey = new AlphaKey();
    newkey->m_alpha = a;
    newkey->m_x = x; 
    newkey->m_y = a * IMAGEHEIGHT;
    newkey->m_sCMSize = m_editorWidth;
    newkey->m_sx = x;

    //detect change of slope
    float a1 = key->m_alpha;
    int x1 = ceil((float)key->m_x/ratio);
    float a2 = newkey->m_alpha;
    int x2 = ceil((float)newkey->m_x/ratio); 

    //y = alpha*x + beta
    float alpha = (a2 - a1)/(x2 - x1);

    if ( keyindex < 2 ) //init
    {
      currentalpha = (a2 - a1)/(x2 - x1);
      keyindex++;
    }
    else
    {
      if ( (fabs(alpha - currentalpha) > 1e-4) && (m_keys.size() <= m_keysNum) )
      {
        m_keys.append( key );
        keyindex++;
        currentalpha = alpha;
      }
      else
        delete key;
    }
    key = newkey;
  }
  //enter the last key before ending
  x = m_xColorCoord[m_numColors-1];
  AlphaKey* lastkey = new AlphaKey();
  lastkey->m_alpha = key->m_alpha;
  lastkey->m_x = x; 
  lastkey->m_y = key->m_alpha * IMAGEHEIGHT; 
  lastkey->m_sCMSize = m_editorWidth;
  lastkey->m_sx = lastkey->m_x;
  m_keys.append( lastkey );
}

//------------------------------------------------------------------------------
AlphaKey* 
Ivt_AlphaCurveEditor::getPickedKey( int x, int y )
{
  x -= WIDGETBORDER;
  y -= WIDGETBORDER;
  AlphaKey* key;
  m_leftKey = NULL;
  m_rightKey = NULL;
  int tolerance = KEYSIZE;
  for( int i = 0; i < m_keys.size(); i++ )
  {
    key = m_keys[i];
    if( (x >= key->m_x - tolerance) && 
        (x <= key->m_x + tolerance) && 
        (y >= IMAGEHEIGHT - key->m_y - tolerance) && 
        (y <= IMAGEHEIGHT - key->m_y + tolerance) )
    {
      if( i > 0 )
        m_leftKey = m_keys[i - 1];
      if( i + 1 < m_keys.size() )
        m_rightKey = m_keys[i + 1];
      return key;
    }
  }
  return NULL;
}

//------------------------------------------------------------------------------
void 
Ivt_AlphaCurveEditor::updatePickedKey( int x, int y )
{
  if ( m_pickedKey )
  {
    x -= WIDGETBORDER;
    y -= WIDGETBORDER;
    checkKeyPosition( x, y );

    //update alpha values
    updateAlphaValues();

    //update image
    createColormapImage();

    m_container->repaint();
  }
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::checkKeyPosition( int x, int y )
{
  if ( m_pickedKey )
  {
    if ( m_rightKey && m_leftKey )
    {
      if ( x <= m_leftKey->m_x + WIDGETBORDER )
        x = m_leftKey->m_x;
      if ( x >= m_rightKey->m_x + WIDGETBORDER )
        x = m_rightKey->m_x;
      m_pickedKey->m_x = x;
    }
    if ( y <= 0 )
      y = 0;
    if ( y > IMAGEHEIGHT )
      y = IMAGEHEIGHT;
    if ( x > m_editorWidth )
      m_pickedKey->m_x = m_editorWidth;
    m_pickedKey->m_y = IMAGEHEIGHT - y;
    m_pickedKey->m_alpha = (float)m_pickedKey->m_y / (float)IMAGEHEIGHT;
  }
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::updateAlphaValues()
{
  //Actually edit transfer function
  if ( m_pickedKey )
  {
    //linear interpolation of alpha values
    float* colors = m_transferFunction->actualColorMap.startEditing();
    float ratio = (float)m_editorWidth / (float)m_numColors;
    if ( m_leftKey )
    {
      float a1 = m_leftKey->m_alpha;
      int x1 = ceil((float)m_leftKey->m_x/ratio);
      float a2 = m_pickedKey->m_alpha;
      int x2 = ceil((float)m_pickedKey->m_x/ratio);
      if (x2 != x1)
      {
        float factor = (a2 - a1)/(x2 - x1);
        for ( int x = x1; x < x2; ++x )
        {
          float newalpha = factor * (x - x1) + a1;
          colors[x*4 + 3] = newalpha;
        }
      }
    }

    if ( m_rightKey )
    {
      float a1 = m_pickedKey->m_alpha;
      int x1 = ceil((float)m_pickedKey->m_x/ratio);
      float a2 = m_rightKey->m_alpha;
      int x2 = ceil((float)m_rightKey->m_x/ratio);
      if ( x2 != x1 )
      {
        float factor = (a2 - a1)/(x2 - x1);
        for ( int x = x1; x < x2; ++x )
        {
          float newalpha = factor * (x - x1) + a1;
          colors[x*4 + 3] = newalpha;
        }
      }
    }
    m_transferFunction->actualColorMap.finishEditing();
  }
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::addKey( int x, int y )
{
  x -= WIDGETBORDER;
  y -= WIDGETBORDER;
  y = m_colormapImage.height() - y;

  if ( x < 0 || y < 0 || x > m_colormapImage.width() || y > m_colormapImage.height() )
    return;
  else
  {
    //find insertion place:
    int index = 0;
    AlphaKey* key;
    Q_FOREACH( key, m_keys )
    {
      if ( key->m_x < x )
        index++;
    }
    //create new key 
    AlphaKey* newKey = new AlphaKey;
    newKey->m_x = x;
    newKey->m_y = y;
    newKey->m_alpha = y/m_colormapImage.height();

    //check if it is not too close of another one
    bool closed = false;
    Q_FOREACH( key, m_keys )
    {
      if( (fabs((double)(key->m_x - x)) <= (KEYSIZE/2)) && 
          (fabs((double)(key->m_y - y)) <= (KEYSIZE/2)) )
      {
        closed = true;
        break;
      }
    }
    if ( !closed )
    {
      //insert it
      newKey->m_sCMSize = m_editorWidth;
      newKey->m_sx = newKey->m_x;
      m_keys.insert( index , newKey );
      m_keysNum++;
    }
    m_container->repaint();
  }
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::removeKey( int x, int y )
{
  x -= WIDGETBORDER;
  y -= WIDGETBORDER;
  y = m_colormapImage.height() - y;
  if ( x < 0 || y < 0 || x > m_colormapImage.width() || y > m_colormapImage.height() )
    return;
  else
  {

    //find insertion place:
    QVector<AlphaKey*>::iterator it;
    for ( it = m_keys.begin(); it != m_keys.end(); ++it )
    {
      if ( (*it) == m_pickedKey )
        break;
    }
    if ( !m_leftKey || !m_rightKey )
      return;
    else
    {
      //update picked alpha so that alpha values are modified accordingly
      float a1 = m_leftKey->m_alpha;
      int x1 = m_leftKey->m_x/m_penWidth;
      float a2 = m_rightKey->m_alpha;
      int x2 = m_rightKey->m_x/m_penWidth;
      if ( x2 != x1 )
      {
        float factor = (a2 - a1)/(x2 - x1);
        m_pickedKey->m_alpha = factor * (m_pickedKey->m_x/m_penWidth - x1) + a1;
      }
    }

    //delete key 
    m_keys.erase(it);
    m_container->repaint();
  }
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::updateKeysPos()
{
    for ( int i = 0; i < m_keys.size(); i++ )
  {
    AlphaKey* key = m_keys[i];
    float ratio = ((float)m_editorWidth * (float)key->m_sx) / (float)key->m_sCMSize;
    key->m_x = (int)ratio;
  }
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::paintEvent( QPaintEvent* /*event*/ )
{
  QPixmap editor( WIDGETBORDER*2 + m_colormapImage.size().width(), WIDGETBORDER*2 + m_colormapImage.size().height() );
  QPainter painter( &editor );
#ifndef WIN32 // UNIX
  painter.eraseRect( 0, 0, WIDGETBORDER*2 + m_colormapImage.size().width(), WIDGETBORDER*2 + m_colormapImage.size().height() );
#endif

  //draw colormap
  painter.drawImage( WIDGETBORDER, WIDGETBORDER, m_colormapImage );

  //draw alpha keys
  QBrush brush( Qt::black );
  QBrush gbrush( Qt::green );
  QPen pen( Qt::black );
  pen.setWidth( 2 );
  painter.setPen( pen );
  painter.setBrush( brush );
  AlphaKey* key;
  QVector<QLine> lines;
  int i = 0;
  QPoint lastPoint;
  Q_FOREACH( key, m_keys )
  {
    i++;
    painter.drawRect( key->m_x + WIDGETBORDER - (KEYSIZE/2), IMAGEHEIGHT + WIDGETBORDER - key->m_y - (KEYSIZE/2), KEYSIZE, KEYSIZE );
    if ( i == 2 )
    {
      lines.append( QLine(lastPoint, QPoint(key->m_x + WIDGETBORDER, IMAGEHEIGHT + WIDGETBORDER - key->m_y)) );
      i = 1;
    }
    lastPoint = QPoint( key->m_x + WIDGETBORDER, IMAGEHEIGHT + WIDGETBORDER - key->m_y );
  }
  painter.drawLines( lines );

  //draw picked key:
  if ( m_pickedKey )
  {
    painter.setBrush( gbrush );
    int keysize = KEYSIZE + 4;
    painter.drawRect( m_pickedKey->m_x + WIDGETBORDER - (keysize/2), 
                      IMAGEHEIGHT + WIDGETBORDER - m_pickedKey->m_y - (keysize/2), keysize, keysize );
  }
  painter.end();

  m_colormapContainer->setPixmap( editor );
  m_colormapContainer->resize( editor.size() );
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::resizeEvent( QResizeEvent* /*event*/ )
{
  if (m_transferFunction && m_numColors)
  {
  // Update image
  createColormapImage();

  // Update alpha keys
  updateKeysPos();

  // Update alpha values
  updateAlphaValues();
  
  m_container->repaint();
  }
}

//------------------------------------------------------------------------------
bool 
Ivt_AlphaCurveEditor::eventFilter( QObject* obj, QEvent* event )
{
  if ( obj == m_colormapContainer )
  {
    if ( event->type() == QEvent::MouseButtonDblClick )
    {
      QMouseEvent* dblc = static_cast<QMouseEvent*>(event);
      addKey( dblc->x(), dblc->y() );
      m_pickedKey = getPickedKey( dblc->x(), dblc->y() );

      return true;
    }
    else if ( event->type() == QEvent::MouseButtonPress )
    {
      QMouseEvent* press = static_cast<QMouseEvent*>(event);
      m_pickedKey = getPickedKey( press->x(), press->y() );

      return true;
    }
    else if ( event->type() == QEvent::MouseButtonRelease )
    {
      QMouseEvent* release = static_cast<QMouseEvent*>(event);
      if ( release->button() == Qt::LeftButton )
        updatePickedKey( release->x(), release->y() ); 
      else if ( release->button() == Qt::RightButton )
      {
        removeKey( release->x(), release->y() );
        //update alpha values
        updateAlphaValues();

        //update image
        createColormapImage();
      }
      m_pickedKey = NULL;
      m_container->repaint();

      return true;
    }
    else if ( event->type() == QEvent::MouseMove )
    {
      QMouseEvent* move = static_cast<QMouseEvent*>(event);
      updatePickedKey( move->x(), move->y() );

      return true;
    }
    else
      return false;
  }
  else if ( obj == m_container )
  {
    if ( event->type() == QEvent::Resize )
      resizeEvent( (QResizeEvent*)event );
    else if ( event->type() == QEvent::Paint )
      paintEvent( (QPaintEvent*)event );

    return true;
  }
  else
    return QObject::eventFilter( obj, event );
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::setUp()
{
  m_transferFunction = (SoTransferFunction*) m_editedObject;

  m_numComponents = 0;
  if ( m_transferFunction->colorMapType.getValue() == SoTransferFunction::RGBA )
    m_numComponents = 4;
  if ( m_transferFunction->colorMapType.getValue() == SoTransferFunction::LUM_ALPHA )
    m_numComponents = 2;
  if  (m_transferFunction->colorMapType.getValue() == SoTransferFunction::ALPHA )
    m_numComponents = 1;

  assert( m_numComponents == 4 ); //for now assume it's RGBA

  m_numColors = m_transferFunction->actualColorMap.getNum() / m_numComponents;

  if (m_numColors == 0) return;

#ifdef WIN32
  m_container->resize( m_numColors*m_penWidth + 4*WIDGETBORDER, IMAGEHEIGHT + 4*WIDGETBORDER );
#else // UNIX
  m_container->resize( m_numColors*m_penWidth + 4*WIDGETBORDER, IMAGEHEIGHT + 6*WIDGETBORDER );
#endif

  createColormapImage();
  computeAlphaKeys();

  m_container->setMaximumSize( m_numColors * PENWIDTH + 4* WIDGETBORDER, m_container->height() );
  m_container->setMinimumSize( m_numColors + 4* WIDGETBORDER, m_container->height() );
}

//------------------------------------------------------------------------------
void
Ivt_AlphaCurveEditor::activate()
{  
  IvtShell::getInstance()->registerViewMenu( this ); // Register this editor to the view menu of IvTune

  setUp(); // Initialize this editor regarding the selected node
  
  IvtEditor::activate();

  show();  // Show this editor.
}

//------------------------------------------------------------------------------
void 
Ivt_AlphaCurveEditor::deactivate()
{
  IvtEditor::deactivate(); // Call it first

  IvtShell::getInstance()->unregisterViewMenu( this ); // Unregister this editor from the view menu of IvTune
 
  hide(); // Hide this editor
}

//------------------------------------------------------------------------------
void 
Ivt_AlphaCurveEditor::hide()
{
  IvtEditor::hide(); // Call it first

  // Hide the GUI elements of this editor
  m_container->hide();
}

//------------------------------------------------------------------------------
void 
Ivt_AlphaCurveEditor::load()
{
  // GUI instanciation
  m_container = new QFrame();
  m_widget = m_container;
  m_sca = new QScrollArea( m_container );
  m_colormapContainer = new QLabel( m_sca );
  m_sca->setWidget( m_colormapContainer );

  QHBoxLayout* layout = new QHBoxLayout( m_container );
  layout->addWidget( m_sca );
  m_colormapContainer->installEventFilter( this );
  m_container->installEventFilter( this );

  m_penWidth = PENWIDTH;
  m_xColorCoord = NULL;

  m_keysNum = 2;

  hide(); // Hide this editor as it will be shown when it will be invoked
}

//------------------------------------------------------------------------------
void 
Ivt_AlphaCurveEditor::show()
{
  IvtEditor::show();

  // Show the GUI elements of this editor
  m_container->show();
}

//------------------------------------------------------------------------------
void 
Ivt_AlphaCurveEditor::unload()
{
  // Delete instanciated objects in load()
  delete m_container;
}

#ifdef _WIN32
#pragma warning( pop )
#endif
