/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T.MEHAMLI (Dec 2010)
**=======================================================================*/

#include <Ivt_LDMControlParameters.h>

#include <IvTune/IvTuneExtender/IvtShell.h>
#include <IvTune/IvTuneExtender/IvtServices.h>

#include <Inventor/fields/SoField.h>
#include <Inventor/nodes/SoNode.h>

#include <LDM/fields/SoSFLDMResourceParameters.h>
#include <LDM/SoLDMGlobalResourceParameters.h>
#include <LDM/nodes/SoLDMResourceParameters.h>
#include <LDM/nodes/SoDataSet.h>

#include <VolumeViz/nodes/SoVolumeData.h>

#include <QDockWidget>
#include <QFrame>

//------------------------------------------------------------------------------
Ivt_LDMControlParameters::Ivt_LDMControlParameters()
: IvtEditor()
{

  SoVolumeRendering::init();

  m_info.name = "SFLDMResourceParameters";
  m_info.author ="Thermo Fisher Scientific";
  m_info.description = "An editor for LDM resource parameters.";
  m_info.version = "1.0.0";

  m_handledType = SoSFLDMResourceParameters::getClassTypeId();
}

//------------------------------------------------------------------------------
Ivt_LDMControlParameters::~Ivt_LDMControlParameters()
{
}

//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::setUp()
{
  m_ldmRPNode = ((SoSFLDMResourceParameters*)m_editedObject)->getValue();

  // Set the editor default values
  // Visual Feedback
  vfDrawTilesCB->setChecked( SoLDMGlobalResourceParameters::getVisualFeedbackParam(SoLDMGlobalResourceParameters::DRAW_TILES) );
  vfTextureFrontCB->setChecked( SoLDMGlobalResourceParameters::getVisualFeedbackParam(SoLDMGlobalResourceParameters::DRAW_TILE_OUTLINE) );

  // Options
  optVCCB->setChecked( SoLDMGlobalResourceParameters::getViewCulling() );
  optSRCB->setChecked(SoLDMGlobalResourceParameters::getScreenResolutionCulling());
  optVPRCB->setChecked( SoLDMGlobalResourceParameters::getViewpointRefinement() );
  optSERCB->setChecked( SoLDMGlobalResourceParameters::getSliceEqualResolution() );

  if ( !optNLTCB->isChecked() )
    optNLTCB->toggle();
  optNLTSlider->setValue( SoLDMGlobalResourceParameters::getNumIO() );
  optTSComboBox->setCurrentIndex( 1 );
  optSSComboBox->setCurrentIndex( 1 );
  optLPComboBox->setCurrentIndex( 0 );
  optMRTComboBox->setCurrentIndex( 1 );
  optMaRTComboBox->setCurrentIndex( 0 );
  optRRcomboBox->setCurrentIndex( 1 );
  // Main Memory Control parameters
  if ( !mmcpAmountCB->isChecked() )
    mmcpAmountCB->toggle();
  int value = m_ldmRPNode->maxMainMemory.getValue();
  if ( value > mmcpAmountSlider->maximum() )
    mmcpAmountSlider->setMaximum( value );
  mmcpAmountSlider->setValue( value );
  if ( !mmcpAmountCB->isChecked() )
    mmcpAmountCB->toggle();
  if ( !mmcpNotificationCB->isChecked() )
    mmcpNotificationCB->toggle();
  mmcpNotificationSlider->setValue( m_ldmRPNode->loadNotificationRate.getValue() );
  // Control parameters for Volume Texturing
  if ( !cpvtAmountCB->isChecked() )
    cpvtAmountCB->toggle();
  value = m_ldmRPNode->maxTexMemory.getValue();
  if ( value > cpvtAmountSlider->maximum() )
    cpvtAmountSlider->setMaximum( value );
  cpvtAmountSlider->setValue( value );

  if ( !cpvtTLRCB->isChecked() )
    cpvtTLRCB->toggle();
  value = m_ldmRPNode->tex3LoadRate.getValue();
  if ( value > cpvtTLRSlider->maximum() )
    cpvtTLRSlider->setMaximum( value );
  cpvtTLRSlider->setValue( value );
  // Control parameters for Slice Texturing
  if ( !cpstTLRCB->isChecked() )
    cpstTLRCB->toggle();
  value = m_ldmRPNode->tex2LoadRate.getValue();
  if ( value > cpstTLRSlider->maximum() )
    cpstTLRSlider->setMaximum( value );
  cpstTLRSlider->setValue( value );
}

//------------------------------------------------------------------------------
// Visual Feedback group slots
//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::setDrawTiles( bool enabled )
{
  SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::DRAW_TILES, enabled );
}

//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::setSliceTextures( bool enabled )
{
  SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::DRAW_SLICES_TEX, enabled );
}

//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::setFakeData( bool enabled )
{
  SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::USE_FAKE_DATA, enabled );
}

//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::setLoadUnloadTiles( bool enabled )
{
  SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::SHOW_LOAD_UNLOAD_TILES, enabled );
}

//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::setTextureFrontOutline( bool enabled )
{
  SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::DRAW_TILE_OUTLINE, enabled );
}

//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::setDataFrontOutline( bool enabled )
{
  SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::DRAW_TOPOLOGY, enabled );
}

//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::setValuationOutline( bool enabled )
{
  SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::SHOW_TILES_VALUATED, enabled );
}

//------------------------------------------------------------------------------
// Options group slots
//------------------------------------------------------------------------------
//Check boxes
void
Ivt_LDMControlParameters::setViewCulling( bool enabled )
{
  SoLDMGlobalResourceParameters::setViewCulling( enabled );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setScreenResolution( bool enabled )
{
  SoLDMGlobalResourceParameters::setScreenResolutionCulling( enabled );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setViewPointRefinement( bool enabled )
{
  SoLDMGlobalResourceParameters::setViewpointRefinement( enabled );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setSliceEqualResolution( bool enabled )
{
  SoLDMGlobalResourceParameters::setSliceEqualResolution( enabled );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setIgnoreFullyTransparentTiles( bool enabled )
{
  SoLDMGlobalResourceParameters::setIgnoreFullyTransparentTiles( enabled );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setOptNLTSliderInt( bool enabled )
{
  optNLTSlider->setTracking( enabled );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setFixedResolutionMode( bool enabled )
{
  m_ldmRPNode->fixedResolution = enabled;
}

//------------------------------------------------------------------------------
// Combo boxes
void 
Ivt_LDMControlParameters::tileSizeCB( int index )
{
  int tileSize = 1 << (index + 5);
  m_ldmRPNode->tileDimension = SbVec3i32( tileSize, tileSize, tileSize );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::subtilesSizeCB( int index )
{
  int subTileSize = 1 << (index + 3);
  m_ldmRPNode->subTileDimension = SbVec3i32( subTileSize, subTileSize, subTileSize );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::loadingPolicyCB( int index )
{
  switch (index)
  {
  case 0 :
    m_ldmRPNode->loadPolicy = SoVolumeData::SoLDMResourceParameters::NO_USER_INTERACTION;
    break;
  case 1 :
    m_ldmRPNode->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;
    break;
  case 2 :
    m_ldmRPNode->loadPolicy = SoVolumeData::SoLDMResourceParameters::NEVER;
    break;
  default:
    break;
  }
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::minResThresholdCB( int index )
{
  m_ldmRPNode->minResolutionThreshold = index;
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::maxResThresholdCB( int index )
{
  m_ldmRPNode->maxResolutionThreshold = index;
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::reqResCB( int index )
{
  m_ldmRPNode->resolution = index;
}

//------------------------------------------------------------------------------
// Line edit
void
Ivt_LDMControlParameters::optNLTLECB()
{
  optNLTSlider->setValue( optNLTLE->text().toInt() );
}

//------------------------------------------------------------------------------
// Slider
void
Ivt_LDMControlParameters::optNLTSliderCB( int value )
{
  optNLTLE->setText( QString::number(value) );

  SoLDMGlobalResourceParameters::setNumIO( (unsigned int)value );
}

//------------------------------------------------------------------------------
// Main Memory Control Parameters
//------------------------------------------------------------------------------
// Amount
void 
Ivt_LDMControlParameters::mmcpAmountSliderCB( int value )
{
  mmcpAmountLE->setText( QString::number(value) );
  m_ldmRPNode->maxMainMemory = value;

  updateSliders();
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::mmcpAmountLECB()
{
  mmcpAmountSlider->setValue( mmcpAmountLE->text().toInt() );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setMmcpAmountSliderInt( bool enabled )
{
  mmcpAmountSlider->setTracking( enabled );
}

//------------------------------------------------------------------------------
// Notification rate
void 
Ivt_LDMControlParameters::mmcpNRSliderCB( int value )
{
  mmcpNotificationLE->setText( QString::number(value) );
  m_ldmRPNode->loadNotificationRate = value;
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::mmcpNRLECB()
{
  mmcpNotificationSlider->setValue( mmcpNotificationLE->text().toInt() );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setMmcpNRSliderInt( bool enabled )
{
  mmcpNotificationSlider->setTracking( enabled );
}

//------------------------------------------------------------------------------
// Control Parameters for Volume Texturing
//------------------------------------------------------------------------------
// Amount
void 
Ivt_LDMControlParameters::cpvtAmountSliderCB( int value )
{
  cpvtAmountLE->setText( QString::number(value) );
  m_ldmRPNode->maxTexMemory = value;

  updateSliders();
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::cpvtAmountLECB()
{
  cpvtAmountSlider->setValue( cpvtAmountLE->text().toInt() );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setCpvtAmountSliderInt( bool enabled )
{
  cpvtAmountSlider->setTracking( enabled );
}

//------------------------------------------------------------------------------
// Texture load rate
void 
Ivt_LDMControlParameters::cpvtTLRSliderCB( int value )
{
  cpvtTLRLE->setText( QString::number(value) );
  m_ldmRPNode->tex3LoadRate = value;

  updateSliders();
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::cpvtTLRLECB()
{
  cpvtTLRSlider->setValue( cpvtTLRLE->text().toInt() );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setCpvtTLRSliderInt( bool enabled )
{
  cpvtTLRSlider->setTracking( enabled );
}

//------------------------------------------------------------------------------
// Control Parameters for Slice Texturing
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Texture load rate for slices
void 
Ivt_LDMControlParameters::cpstTLRSliderCB( int value )
{
  cpstTLRLE->setText( QString::number(value) );
  m_ldmRPNode->tex2LoadRate = value;

  updateSliders();
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::cpstTLRLECB()
{
  cpstTLRSlider->setValue( cpstTLRLE->text().toInt() );
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::setCpstTLRSliderInt( bool enabled )
{
  cpstTLRSlider->setTracking( enabled );
}

//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::updateSliders()
{
  int value = 0;

  // Main Memory Control Parameters
  value = m_ldmRPNode->maxMainMemory.getValue();
  mmcpAmountSlider->setSliderPosition( value );
  value = m_ldmRPNode->maxTexMemory.getValue();
  cpvtAmountSlider->setSliderPosition( value );
  value = m_ldmRPNode->tex3LoadRate.getValue();
  cpvtTLRSlider->setSliderPosition( value );
  value = m_ldmRPNode->tex2LoadRate.getValue();
  cpstTLRSlider->setSliderPosition( value );
}

//------------------------------------------------------------------------------
// Methods redifined from IvtExtension, IvtEditor and IvtFieldEditor
//------------------------------------------------------------------------------
void
Ivt_LDMControlParameters::activate()
{
  IvtShell::getInstance()->registerViewMenu( this ); // Register this editor to the view menu of IvTune

  IvtEditor::activate();

  show(); // Show this editor.

  setUp(); // Initialize this editor regarding the selected node
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::deactivate()
{
  IvtEditor::deactivate();

  IvtShell::getInstance()->unregisterViewMenu( this ); // Unregister this editor from the view menu of IvTune

  hide(); // Hide this editor
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::hide()
{
  IvtEditor::hide(); // Call it first

  // Hide the GUI elements of this editor
  m_container->hide();
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::load()
{
  // Containers instanciation
  m_container = new QFrame();
  m_widget = m_container;

  m_ldmRPNode = NULL;

  // Instanciation of the editor
  setupUi( m_container );

  // Create connections
  // Visual Feedback
  connect( vfDrawTilesCB, SIGNAL(toggled(bool)), this, SLOT(setDrawTiles(bool)) );
  connect( vfTextureFrontCB, SIGNAL(toggled(bool)), this, SLOT(setTextureFrontOutline(bool)) );
  connect( vfSliceTexturesCB, SIGNAL(toggled(bool)), this, SLOT(setSliceTextures(bool)) );
  connect( vfDataFrontCB, SIGNAL(toggled(bool)), this, SLOT(setDataFrontOutline(bool)) );
  connect( vfFakeDataCB, SIGNAL(toggled(bool)), this, SLOT(setFakeData(bool)) );
  connect( vfValuationCB, SIGNAL(toggled(bool)), this, SLOT(setValuationOutline(bool)) );
  connect( vfLulTilesCB, SIGNAL(toggled(bool)), this, SLOT(setLoadUnloadTiles(bool)) );
  // Options
  connect( optVCCB, SIGNAL(toggled(bool)), this, SLOT(setViewCulling(bool)) );
  connect( optSRCB, SIGNAL(toggled(bool)), this, SLOT(setScreenResolution(bool)) );
  connect( optVPRCB, SIGNAL(toggled(bool)), this, SLOT(setViewPointRefinement(bool)) );
  connect( optSERCB, SIGNAL(toggled(bool)), this, SLOT(setSliceEqualResolution(bool)) );
  connect( optIFTTCB, SIGNAL(toggled(bool)), this, SLOT(setIgnoreFullyTransparentTiles(bool)) );
  connect( optNLTCB, SIGNAL(toggled(bool)), this, SLOT(setOptNLTSliderInt(bool)) );
  connect( optNLTSlider, SIGNAL(valueChanged(int)), this, SLOT(optNLTSliderCB(int)) );
  connect( optNLTLE, SIGNAL(editingFinished()), this, SLOT(optNLTLECB()) );
  connect( optTSComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(tileSizeCB(int)) );
  connect( optSSComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(subtilesSizeCB(int)) );
  connect( optLPComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(loadingPolicyCB(int)) );
  connect( optMRTComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(minResThresholdCB(int)) );
  connect( optMaRTComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(maxResThresholdCB(int)) );
  connect( optFRMCB, SIGNAL(toggled(bool)), this, SLOT(setFixedResolutionMode(bool)) );
  connect( optRRcomboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(reqResCB(int)) );
  // Main Memory Control Parameters
  connect( mmcpAmountCB, SIGNAL(toggled(bool)), this, SLOT(setMmcpAmountSliderInt(bool)) );
  connect( mmcpAmountSlider, SIGNAL(valueChanged(int)), this, SLOT(mmcpAmountSliderCB(int)) );
  connect( mmcpAmountLE, SIGNAL(editingFinished()), this, SLOT(mmcpAmountLECB()) );
  connect( mmcpNotificationCB, SIGNAL(toggled(bool)), this, SLOT(setMmcpNRSliderInt(bool)) );
  connect( mmcpNotificationSlider, SIGNAL(valueChanged(int)), this, SLOT(mmcpNRSliderCB(int)) );
  connect( mmcpNotificationLE, SIGNAL(editingFinished()), this, SLOT(mmcpNRLECB()) );
  // Control Parameters for Volume Texturing
  connect( cpvtAmountCB, SIGNAL(toggled(bool)), this, SLOT(setCpvtAmountSliderInt(bool)) );
  connect( cpvtAmountSlider, SIGNAL(valueChanged(int)), this, SLOT(cpvtAmountSliderCB(int)) );
  connect( cpvtAmountLE, SIGNAL(editingFinished()), this, SLOT(cpvtAmountLECB()) );
  connect( cpvtTLRCB, SIGNAL(toggled(bool)), this, SLOT(setCpvtTLRSliderInt(bool)) );
  connect( cpvtTLRSlider, SIGNAL(valueChanged(int)), this, SLOT(cpvtTLRSliderCB(int)) );
  connect( cpvtTLRLE, SIGNAL(editingFinished()), this, SLOT(cpvtTLRLECB()) );
  // Control Parameters for Slice Texturing
  connect( cpstTLRCB, SIGNAL(toggled(bool)), this, SLOT(setCpstTLRSliderInt(bool)) );
  connect( cpstTLRSlider, SIGNAL(valueChanged(int)), this, SLOT(cpstTLRSliderCB(int)) );
  connect( cpstTLRLE, SIGNAL(editingFinished()), this, SLOT(cpstTLRLECB()) );

  hide(); // Hide this editor as it will be shown when it will be invoked
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::show()
{
  IvtEditor::show();

  // Show the GUI elements of this editor
  m_container->show();
}

//------------------------------------------------------------------------------
void 
Ivt_LDMControlParameters::unload()
{
  // Delete instanced objects in load()
  delete m_container;
}
