///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

// Simple Marching Cubes (isosurface extraction) for VolumeViz.
//
//
// This class contains a simple implementation of the classic Marching Cubes
// algorithm that computes the geometry (triangles) of an isosurface from a
// VolumeViz data set. It returns an SoTriangleSet node with an attached
// SoVertexProperty node containing the vertices and normal vectors.  The
// geometry can then be exported using the usual Open Inventor tools, for
// example SoSTLWriteAction will write an STL format file.
//
// The classic Marching Cubes algorithm operates on each cell of the mesh
// independently (see marchCube() method).  We process each "tile" of the
// VolumeViz data set independently (see marchTile() method).  This allows
// efficient data access because the SoLDMDataAccess class can give us a
// read-only reference to one tile of the volume without copying any data.
//
// This is great if the volume fits in one tile (recommended when possible
// for medical data sets).  If not, we have to make another pass to compute
// the triangles between tiles.
//
// TODO:
//   - Compute normal vectors.
//     This info is not critical. OIV normal computation can generate smooth
//     normals automatically. But normals computed using the actual volume
//     gradient should be more correct.
//
//   - Handle RegionOfInterest.
//     Extraction will be much faster if the application knows in advance that
//     only voxels in the ROI need to be considered.
//
//   - Lower resolution data.
//     Might be useful to extract isosurface from low-res data.
//     That would be much faster for large volumes and could be a kind of
//     preview, but it's not clear this is actually useful since the GPU
//     isosurface feature also provides a preview.
//
// Notes:
//   - This is a simple implementation of classic Marching Cubes.
//     The implementation itself could be optimized.
//     There are variations of Marching Cubes that have better behavior
//     such as fewer triangles, better shaped triangles, indexed vertices, etc.
//     And there are alternate algorithms such as Marching Tetrahedrons.
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// The marching cubes code used here was adapted from:
//
// Marching Cubes Example Program
// by Cory Bloyd (corysama@yahoo.com)
//
// A simple, portable and complete implementation of the Marching Cubes
// and Marching Tetrahedrons algorithms in a single source file.
// There are many ways that this code could be made faster, but the
// intent is for the code to be easy to understand.
//
// For a description of the algorithm go to
// http://astronomy.swin.edu.au/pbourke/modelling/polygonise/
//
// This code is public domain.
///////////////////////////////////////////////////////////////////////////////

#include <Inventor/nodes/SoTriangleSet.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <Inventor/errors/SoDebugError.h>

#include <VolumeViz/nodes/SoVolumeData.h>

#include <LDM/SoLDMDataAccess.h>

#include <Medical/helpers/VolumeMarchingCubes.h>

#include <vector>

///////////////////////////////////////////////////////////////////////////////
VolumeMarchingCubes::VolumeMarchingCubes()
: m_volData( NULL ) // Only exists during computation of isosurface.
, m_dataInfo( new SoLDMDataAccess::DataInfo() ) // Initialize data info to "invalid"
, m_isoValue(0)
, m_resolution( 0 ) // Eventually may allow extraction on lower resolution data. Not clear if this is useful or not...
, m_numTriangles(0)
{
  // This object is returned by LDM getData() and used in getVoxelValue().
  m_dataInfo->tileData = NULL;
}

///////////////////////////////////////////////////////////////////////////////
VolumeMarchingCubes::~VolumeMarchingCubes()
{
  delete m_dataInfo;
}

///////////////////////////////////////////////////////////////////////////////
size_t
VolumeMarchingCubes::getNumTriangles() const
{
  return m_numTriangles;
}

///////////////////////////////////////////////////////////////////////////////
float
VolumeMarchingCubes::getIsovalue() const
{
  return m_isoValue;
}

///////////////////////////////////////////////////////////////////////////////
SoTriangleSet*
VolumeMarchingCubes::getIsosurface( const SoVolumeData& _volume, float isovalue )
{
  // Not a meaningful request for RGBA volumes...
  if (_volume.dataRGBA.getValue() == TRUE)
    return NULL;

  m_isoValue     = isovalue;
  m_numTriangles = 0;
  m_vertices.clear();

  // This is annoying... but some query methods don't work on a const ref object. :-(
  SoVolumeData& volume = const_cast<SoVolumeData&>( _volume );

  // Volume properties
  // We don't currently need all this information, but it might be useful.
  m_voldim = (SbVec3i32)volume.data.getSize(); // Cast away const
  m_volext = volume.extent.getValue();
  m_tiledim = volume.ldmResourceParameters.getValue()->tileDimension.getValue();
  SbVec3f m_volMin = m_volext.getMin(); // This is more convenient than the SbBox
  SbVec3f m_volMax = m_volext.getMax();
  m_voxelBytes = volume.getDatumSize();
  m_voxelType  = volume.getDataType();

  // Voxel size in XYZ coords
  m_voxelSize[0] = ( m_volMax[0] - m_volMin[0] ) / (float)m_voldim[0];
  m_voxelSize[1] = ( m_volMax[1] - m_volMin[1] ) / (float)m_voldim[1];
  m_voxelSize[2] = ( m_volMax[2] - m_volMin[2] ) / (float)m_voldim[2];
  SbVec3f halfVoxel = m_voxelSize * 0.5;

  // Tile size in XYZ coords
  m_tileSize[0] = m_tiledim[0] * m_voxelSize[0];
  m_tileSize[1] = m_tiledim[1] * m_voxelSize[1];
  m_tileSize[2] = m_tiledim[2] * m_voxelSize[2];

  // How many tiles on each axis?
  int nTilesI = m_voldim[0] / m_tiledim[0];
  int nTilesJ = m_voldim[1] / m_tiledim[1];
  int nTilesK = m_voldim[2] / m_tiledim[2];
  if (nTilesI * m_tiledim[0] < m_voldim[0]) {
    nTilesI++;
  }
  if (nTilesJ * m_tiledim[1] < m_voldim[1]) {
    nTilesJ++;
  }
  if (nTilesK * m_tiledim[2] < m_voldim[2]) {
    nTilesK++;
  }
  int nTiles = nTilesI + nTilesJ + nTilesK;

#ifdef XTRA_DEBUG
  unsigned int totalTiles = nTilesI * nTilesJ * nTilesK;
  unsigned int curTile = 0;
  std::cout << "  Total tiles: " << totalTiles
            << " ( " << nTilesI << " x " << nTilesJ << " x " << nTilesK << ")\n";
#endif

  // Check for multiple tiles.
  if (nTiles > 1) {
    //-------------------------------- Multiple Tiles -------------------------
    // This is the simplest and most straightforward implementation.  March
    // across the volume querying voxel values as we need them.  Effectively we
    // ignore tiles by hiding tile specific code in the getVoxelValue() method.
    // There are faster ways to do this, but it's valuable to have a simple,
    // stable implementation for reference.

    // getVoxelValue method needs to access the volumeData node.
    m_volData = &volume;

    // March across the volume
    marchVolume();

    // Release any tiles locked by the getVoxelValue method.
    if (m_dataInfo->tileData != NULL) {
      m_volData->getLdmDataAccess().releaseData( m_dataInfo->tileID );
      m_dataInfo->tileData = NULL;
    }
    m_volData = NULL;
  }
  else {
    //-------------------------------- Single Tile ------------------------------
    // The entire volume fits in a single tile, therefore we just need to get a
    // pointer to the tile data and we have a pointer to the entirety of the
    // volume data as a contiguous block of memory.  For smaller volumes this is
    // the fastest way to iterate over all the voxels.
    // Note that an application can set the tile size when loading data from
    // non-LDM formats like DICOM.
    //
    // Eventually we could use this for a hybrid solution where we call marchTile()
    // for each tile in the volume, then fill in the "gaps" using getVoxelValue().
    //
    // For each tile we need the IJK and XYZ bounds, taking into account that:
    // -- The volume may not fill the tile completely.
    //    E.g. volume I dimension is 192, but the tile dimension is 256;
    // -- The corners of the mesh cells (cubes) are the voxel centers.
    //    So the 3D extent of the mesh is the volume extent minus the voxel
    //    size and starts at volmin + 1/2 the voxel size.

    // Current tile properties.
    SbVec3i32 tileMinIJK( 0, 0, 0 );
    SbVec3f   tileMinXYZ( 0, 0, 0 );
    SbVec3f   tileMaxXYZ( 0, 0, 0 );
    SbVec3i32 curTileDim;
    tileMinIJK[0] = 0;  // Abs IJK of first voxel in tile
    tileMinXYZ[0] = m_volMin[0] + halfVoxel[0];
    tileMaxXYZ[0] = tileMinXYZ[0] + (( m_voldim[0] - 1) * m_voxelSize[0]);

    tileMinIJK[1] = 0;  // Abs IJK of first voxel in tile
    tileMinXYZ[1] = m_volMin[1] + halfVoxel[1];
    tileMaxXYZ[1] = tileMinXYZ[1] + ((m_voldim[1] - 1) * m_voxelSize[1]);

    tileMinIJK[2] = 0;  // Abs IJK of first voxel in tile
    tileMinXYZ[2] = m_volMin[2] + halfVoxel[2];
    tileMaxXYZ[2] = tileMinXYZ[2] + ((m_voldim[2] - 1) * m_voxelSize[2]);

    // Get pointer to tile data.
    // Note this locks the tile in memory until we release it.
    SoLDMDataAccess::DataInfo info = volume.getLdmDataAccess().getData( m_resolution, tileMinIJK );
    if (info.errorFlag != SoLDMDataAccess::CORRECT) {
      SbString str;
      str.sprintf( "Unable to access tile data at IJK %d,%d,%d", tileMinIJK[0], tileMinIJK[1], tileMinIJK[2] ); 
      SoDebugError::post( "VolumeMarchingCubes", "%s", str.getString() );
      return NULL;
    }
    SoLDMTileID tileId   = info.tileID;            // Need this to release tile data
    void*       tileData = info.tileData;

    // Extract isosurface from a tile
    marchTile( curTileDim, tileMinXYZ, tileMaxXYZ, tileData );

    // Release the tile data
    volume.getLdmDataAccess().releaseData( tileId );
  }

  // Remember how many triangles we computed.
  size_t nVert = m_vertices.size();
  size_t nTris = nVert / 3;
  m_numTriangles = nTris;
#ifdef XTRA_DEBUG
  std::cout << nVert << " total vertices, " << nTris << " total triangles\n";
#endif

  // Create geometry nodes to return.
  SoVertexProperty* vprop = new SoVertexProperty();
  vprop->vertex.setValues( 0, (int)m_vertices.size(), (const SbVec3f*)&m_vertices[0] );

  SoTriangleSet* geometry = new SoTriangleSet();
  geometry->vertexProperty = vprop;

  // TODO: Should we free the memory after each extraction? More efficient to not free...
  // Free memory associated with vector (clear does not actually release memory).
  //m_vertices.clear();
  //std::vector<SbVec3f>().swap( m_vertices );

  return geometry;
}

///////////////////////////////////////////////////////////////////////////////
// Convert a voxel value to float for computation.

static float convertData( const unsigned char* data, SoDataSet::DataType type )
{
  switch (type) {
    default:
    case SoDataSet::UNSIGNED_BYTE:
      return (float)*data;
    case SoDataSet::UNSIGNED_SHORT:
      return (float)*((unsigned short*)data);
    case SoDataSet::UNSIGNED_INT32:
      return (float)*((unsigned int*)data);
    case SoDataSet::SIGNED_BYTE:
      return (float)*((char*)data);
    case SoDataSet::SIGNED_SHORT:
      return (float)*((short*)data);
    case SoDataSet::SIGNED_INT32:
      return (float)*((int*)data);
    case SoDataSet::FLOAT:
      return *((float*)data);
  }
}

///////////////////////////////////////////////////////////////////////////////
// Apply marching cubes to entire volume.
//
// This method is used when there are multiple tiles.
// It's less efficient than the single tile method, but it automatically
// handles cells that cross the border between tiles.

void VolumeMarchingCubes::marchVolume( )
{
  const SbVec3f volMin = m_volext.getMin();
  const int nI = m_voldim[0];
  const int nJ = m_voldim[1];
  const int nK = m_voldim[2];
  const float voxelSizeX = m_voxelSize[0];
  const float voxelSizeY = m_voxelSize[1];
  const float voxelSizeZ = m_voxelSize[2];
  const float volMinX = volMin[0] + 0.5f * voxelSizeX;
  const float volMinY = volMin[1] + 0.5f * voxelSizeY;
  const float volMinZ = volMin[2] + 0.5f * voxelSizeZ;

  SbVec3f corner;     // XYZ position of lower left back corner of each cube.
  SbVec3f vertex[15]; // max of 5 triangles per cube
  float   values[8];  // values at corners of cube

  // Loop over "cubes" in volume. Each block of 8 voxels is 1 cube.
  // Loop over K axis slices of volume
  for (int ik = 0; ik < nK-1; ik++) {
    corner[2] = volMinZ + ik * voxelSizeZ;

    // Loop over J axis rows of slice
    for (int ij = 0; ij < nJ-1; ij++) {
      corner[1] = volMinY + ij * voxelSizeY;

      // Loop over I axis voxels of row
      for (int ii = 0; ii < nI-1; ii++) {
        corner[0] = volMinX + ii * voxelSizeX;
        // Get data values at corners of this cube.
        // Order is 0: lower-left-back , 1: lower-right-back , 2: upper-right-back , 3: upper-left-back,
        //          4: lower-left-front, 5: lower-right-front, 6: upper-right-front, 7: upper-left-front.
        //
        // Calling getVoxelValue() is convenient, but not the most efficient way
        // to access the data (has to check for the correct tile on each call).
        // A simple optimization is to re-use the "right-hand side" values from
        // the previous cube as the left-hand side values of the current cube.
        // It's also slightly more efficient to fetch all "ii" values first since
        // they will all be in the same tile most of the time, then fetch the
        // "ii+1" values.
        if (ii == 0) {
          // Beginning of row - fetch the left-hand side values.
          values[0] = getVoxelValue( ii  , ij  , ik   );  // Lower-left-back
          values[4] = getVoxelValue( ii  , ij  , ik+1 );  // Lower-left-front
          values[3] = getVoxelValue( ii  , ij+1, ik   );  // Upper-left-back
          values[7] = getVoxelValue( ii  , ij+1, ik+1 );  // Upper-left-front
        }
        else {
          // Copy previously fetched right-hand side values.
          values[0] = values[1];
          values[3] = values[2];
          values[4] = values[5];
          values[7] = values[6];
        }

        // Fetch the new right-hand side values
        values[1] = getVoxelValue( ii+1, ij  , ik   ); // Lower-right-back
        values[5] = getVoxelValue( ii+1, ij  , ik+1 ); // Lower-right-front
        values[2] = getVoxelValue( ii+1, ij+1, ik   ); // Upper-right-back
        values[6] = getVoxelValue( ii+1, ij+1, ik+1 ); // Upper-right-front

        // Ignore irrelevant cubes.
        if (values[0] < m_isoValue && 
            values[1] < m_isoValue &&
            values[2] < m_isoValue &&
            values[3] < m_isoValue &&
            values[4] < m_isoValue &&
            values[5] < m_isoValue &&
            values[6] < m_isoValue &&
            values[7] < m_isoValue   )
            continue;

        // Compute triangles for this cube and store vertices.
        int numTri = marchCube( corner, m_voxelSize, values, m_isoValue, vertex );
        if (numTri > 0) {
          m_vertices.insert( m_vertices.end(), vertex, vertex + (numTri * 3) );
        }
      }
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
// Apply marching cubes to one tile of data.
//
// This method is given a pointer to the data in the tile.

void VolumeMarchingCubes::marchTile( SbVec3i32 tileDim, SbVec3f tileMin, SbVec3f /*tileMax*/, void* tileData )
{
  // Loop over "cubes" in tile.
  // Each block of 8 voxels is 1 cube.

  SbVec3f corner;
  SbVec3f size = m_voxelSize;
  SbVec3f vertex[15]; // max of 5 triangles per cube
  float   values[8];  // values at corners of cube

  // The actual voxel values could be anything from unsigned byte to float.
  // Get a "byte" pointer to the data so we can compute offsets to values.
  // Compute offsets based on bytes per voxel.
  unsigned char* data = (unsigned char*)tileData;
  int oneVoxel = m_voxelBytes;
  int oneRow   = m_tiledim[0] * m_voxelBytes;
  int oneSlice = m_tiledim[0] * m_tiledim[1] * m_voxelBytes;

  // Loop over K axis slices of volume
  for (int ik = 0; ik < (tileDim[2] - 1); ik++) {
    corner[2] = tileMin[2] + ik * m_voxelSize[2];

    // Loop over J axis rows of slice
    for (int ij = 0; ij < (tileDim[1] - 1); ij++) {
      corner[1] = tileMin[1] + ij * m_voxelSize[1];

      // Loop over I axis voxels of row
      for (int ii = 0; ii < (tileDim[0] - 1); ii++) {
        corner[0] = tileMin[0] + ii * m_voxelSize[0];
        // Get data values at corners of this cube.
        // Order is 0: lower-left-back , 1: lower-right-back , 2: upper-right-back , 3: upper-left-back,
        //          4: lower-left-front, 5: lower-right-front, 6: upper-right-front, 7: upper-left-front.
        // Have to convert from actual data type to float.
        int offset = (ii * m_voxelBytes) + ( ij * oneRow ) + ( ik * oneSlice );
        unsigned char* voxptr = data + offset;
        if (ii == 0) {
          // Beginning of row - fetch the left-hand side values.
          values[0] = convertData( voxptr         , m_voxelType ); // Lower-left-back
          values[3] = convertData( voxptr + oneRow, m_voxelType ); // Upper-left-back
          voxptr += oneSlice;
          values[4] = convertData( voxptr         , m_voxelType ); // Lower-left-front
          values[7] = convertData( voxptr + oneRow, m_voxelType ); // Upper-left-front
        }
        else {
          // Copy previously fetched right-hand side values.
          values[0] = values[1];
          values[3] = values[2];
          values[4] = values[5];
          values[7] = values[6];
        }
        voxptr = data + offset + oneVoxel;
        values[1] = convertData( voxptr         , m_voxelType );  // Lower-right-back
        values[2] = convertData( voxptr + oneRow, m_voxelType );  // Upper-right-back
        voxptr += oneSlice;
        values[5] = convertData( voxptr         , m_voxelType );  // Lower-right-front
        values[6] = convertData( voxptr + oneRow, m_voxelType );  // Upper-right-front

        // Ignore irrelevant cell.
        if (values[0] < m_isoValue && 
            values[1] < m_isoValue &&
            values[2] < m_isoValue &&
            values[3] < m_isoValue &&
            values[4] < m_isoValue &&
            values[5] < m_isoValue &&
            values[6] < m_isoValue &&
            values[7] < m_isoValue   )
            continue;

        // Compute triangles for this cube and store vertices.
        int numTri = marchCube( corner, size, values, m_isoValue, vertex );
        if (numTri) {
          m_vertices.insert( m_vertices.end(), vertex, vertex + (numTri * 3) );
        }
      }
    }
  }
}

////////////////////////////////////////////////////////////////////////
//
// Return data value at specified location.
//
// A simpler version of the code in the MeshVizXLM VolumeMesh example.
// This is a general solution for "get the value of 1 voxel".  As a
// result it is not an efficient solution for getting the values of
// multiple voxels contained in the same tile.  For example, it must
// compute the required tile id every time it is called.  However it
// does keep one tile of data locked in memory, so the overhead is
// reduced when querying a voxel in the same tile as the previous
// query.
float
VolumeMarchingCubes::getVoxelValue( int i, int j, int k)
{
  float value = 0;

  // Requested voxel position
  SbVec3i32 voxelPos(i,j,k);

  // If we already have a tile of data loaded...
  if (m_dataInfo->tileData) {
    // Check if the voxel we need is inside the current tile.
    // Note there is a simpler test for this using:
    //    SoLDMTileID needTileID = m_topoOctree->getTileID( voxelPos, m_resolution );
    //    if (needTileID != m_dataInfo->tileID) {
    // But getTileID() does a lot more math than we actually need!
    //
    // If we need a different tile than the one we have...
    if (i < m_tileMin[0] || i > m_tileMax[0] ||
        j < m_tileMin[1] || j > m_tileMax[1] ||
        k < m_tileMin[2] || k > m_tileMax[2]   ) {
      // Release the tile we already have.
      m_volData->getLdmDataAccess().releaseData( m_dataInfo->tileID );
      m_dataInfo->tileData = NULL;
    }
  }

  // If we don't have a tile of data loaded...
  if (! m_dataInfo->tileData) {
    // getData will load the LDM tile that contains the specified voxel.
    // m_dataInfo contains a pointer to the tile data.
    *m_dataInfo = m_volData->getLdmDataAccess().getData( m_resolution, voxelPos );
    if (m_dataInfo->errorFlag != SoLDMDataAccess::CORRECT) {
      m_dataInfo->tileData = NULL;
      std::cerr << "Error in getVoxelValue: " << m_dataInfo->errorFlag << ", voxel: " << i << " " << j << " " << k << std::endl;
      return 0;
    }
    m_tileMin = m_dataInfo->tilePosition.getMin();
    m_tileMax = m_dataInfo->tilePosition.getMax();
  }

  // Compute offset to voxel location in this tile
  // voxelPos is abs IJK position in volume.
  // Need position relative to tile.
  SbVec3i32& tileDim = m_dataInfo->tileDimension;
  size_t offset = (k - m_tileMin[2])*(tileDim[0] * tileDim[1])
                + (j - m_tileMin[1])* tileDim[0]
                + (i - m_tileMin[0]); // offset in voxels
  unsigned char *pValue = ((unsigned char*)m_dataInfo->tileData) + offset*m_voxelBytes;

  // Convert the data value to float.
  value = convertData( pValue, m_voxelType );
  return value;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// Private data for the implementation

// These tables are used so that everything can be done in little loops that you can look at all at once
// rather than in pages and pages of unrolled code.

// a2fVertexOffset lists the positions, relative to vertex0, of each of the 8 vertices of a cube
static const float a2fVertexOffset[8][3] =
{
  {0.0, 0.0, 0.0}, {1.0, 0.0, 0.0}, {1.0, 1.0, 0.0}, {0.0, 1.0, 0.0},
  {0.0, 0.0, 1.0}, {1.0, 0.0, 1.0}, {1.0, 1.0, 1.0}, {0.0, 1.0, 1.0}
};

// a2iEdgeConnection lists the index of the endpoint vertices for each of the 12 edges of the cube
static const int a2iEdgeConnection[12][2] =
{
  {0, 1}, {1, 2}, {2, 3}, {3, 0},
  {4, 5}, {5, 6}, {6, 7}, {7, 4},
  {0, 4}, {1, 5}, {2, 6}, {3, 7}
};

// a2fEdgeDirection lists the direction vector (vertex1-vertex0) for each edge in the cube
static const float a2fEdgeDirection[12][3] =
{
  {1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {-1.0, 0.0, 0.0}, {0.0, -1.0, 0.0},
  {1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {-1.0, 0.0, 0.0}, {0.0, -1.0, 0.0},
  {0.0, 0.0, 1.0}, {0.0, 0.0, 1.0}, { 0.0, 0.0, 1.0}, {0.0,  0.0, 1.0}
};

// a2iTetrahedronEdgeConnection lists the index of the endpoint vertices for each of the 6 edges of the tetrahedron
static const int a2iTetrahedronEdgeConnection[6][2] =
{
  {0, 1},  {1, 2},  {2, 0},  {0, 3},  {1, 3},  {2, 3}
};

// a2iTetrahedronEdgeConnection lists the index of vertices from a cube
// that made up each of the six tetrahedrons within the cube
static const GLint a2iTetrahedronsInACube[6][4] =
{
  {0, 5, 1, 6},
  {0, 1, 2, 6},
  {0, 2, 3, 6},
  {0, 3, 7, 6},
  {0, 7, 4, 6},
  {0, 4, 5, 6},
};

// For any edge, if one vertex is inside of the surface and the other is outside of the surface
//  then the edge intersects the surface
// For each of the 4 vertices of the tetrahedron can be two possible states : either inside or outside of the surface
// For any tetrahedron the are 2^4=16 possible sets of vertex states
// This table lists the edges intersected by the surface for all 16 possible vertex states
// There are 6 edges.  For each entry in the table, if edge #n is intersected, then bit #n is set to 1
/*
static int aiTetrahedronEdgeFlags[16] =
{
  0x00, 0x0d, 0x13, 0x1e, 0x26, 0x2b, 0x35, 0x38, 0x38, 0x35, 0x2b, 0x26, 0x1e, 0x13, 0x0d, 0x00,
};
*/

// For each of the possible vertex states listed in aiTetrahedronEdgeFlags there is a specific triangulation
// of the edge intersection points.  a2iTetrahedronTriangles lists all of them in the form of
// 0-2 edge triples with the list terminated by the invalid value -1.
//
// I generated this table by hand
/*
static int a2iTetrahedronTriangles[16][7] =
{
  {-1, -1, -1, -1, -1, -1, -1},
  { 0,  3,  2, -1, -1, -1, -1},
  { 0,  1,  4, -1, -1, -1, -1},
  { 1,  4,  2,  2,  4,  3, -1},

  { 1,  2,  5, -1, -1, -1, -1},
  { 0,  3,  5,  0,  5,  1, -1},
  { 0,  2,  5,  0,  5,  4, -1},
  { 5,  4,  3, -1, -1, -1, -1},

  { 3,  4,  5, -1, -1, -1, -1},
  { 4,  5,  0,  5,  2,  0, -1},
  { 1,  5,  0,  5,  3,  0, -1},
  { 5,  2,  1, -1, -1, -1, -1},

  { 3,  4,  2,  2,  4,  1, -1},
  { 4,  1,  0, -1, -1, -1, -1},
  { 2,  3,  0, -1, -1, -1, -1},
  {-1, -1, -1, -1, -1, -1, -1},
};
*/

// For any edge, if one vertex is inside of the surface and the other is outside of the surface
//  then the edge intersects the surface
// For each of the 8 vertices of the cube can be two possible states : either inside or outside of the surface
// For any cube the are 2^8=256 possible sets of vertex states
// This table lists the edges intersected by the surface for all 256 possible vertex states
// There are 12 edges.  For each entry in the table, if edge #n is intersected, then bit #n is set to 1

static int aiCubeEdgeFlags[256] =
{
  0x000, 0x109, 0x203, 0x30a, 0x406, 0x50f, 0x605, 0x70c, 0x80c, 0x905, 0xa0f, 0xb06, 0xc0a, 0xd03, 0xe09, 0xf00,
  0x190, 0x099, 0x393, 0x29a, 0x596, 0x49f, 0x795, 0x69c, 0x99c, 0x895, 0xb9f, 0xa96, 0xd9a, 0xc93, 0xf99, 0xe90,
  0x230, 0x339, 0x033, 0x13a, 0x636, 0x73f, 0x435, 0x53c, 0xa3c, 0xb35, 0x83f, 0x936, 0xe3a, 0xf33, 0xc39, 0xd30,
  0x3a0, 0x2a9, 0x1a3, 0x0aa, 0x7a6, 0x6af, 0x5a5, 0x4ac, 0xbac, 0xaa5, 0x9af, 0x8a6, 0xfaa, 0xea3, 0xda9, 0xca0,
  0x460, 0x569, 0x663, 0x76a, 0x066, 0x16f, 0x265, 0x36c, 0xc6c, 0xd65, 0xe6f, 0xf66, 0x86a, 0x963, 0xa69, 0xb60,
  0x5f0, 0x4f9, 0x7f3, 0x6fa, 0x1f6, 0x0ff, 0x3f5, 0x2fc, 0xdfc, 0xcf5, 0xfff, 0xef6, 0x9fa, 0x8f3, 0xbf9, 0xaf0,
  0x650, 0x759, 0x453, 0x55a, 0x256, 0x35f, 0x055, 0x15c, 0xe5c, 0xf55, 0xc5f, 0xd56, 0xa5a, 0xb53, 0x859, 0x950,
  0x7c0, 0x6c9, 0x5c3, 0x4ca, 0x3c6, 0x2cf, 0x1c5, 0x0cc, 0xfcc, 0xec5, 0xdcf, 0xcc6, 0xbca, 0xac3, 0x9c9, 0x8c0,
  0x8c0, 0x9c9, 0xac3, 0xbca, 0xcc6, 0xdcf, 0xec5, 0xfcc, 0x0cc, 0x1c5, 0x2cf, 0x3c6, 0x4ca, 0x5c3, 0x6c9, 0x7c0,
  0x950, 0x859, 0xb53, 0xa5a, 0xd56, 0xc5f, 0xf55, 0xe5c, 0x15c, 0x055, 0x35f, 0x256, 0x55a, 0x453, 0x759, 0x650,
  0xaf0, 0xbf9, 0x8f3, 0x9fa, 0xef6, 0xfff, 0xcf5, 0xdfc, 0x2fc, 0x3f5, 0x0ff, 0x1f6, 0x6fa, 0x7f3, 0x4f9, 0x5f0,
  0xb60, 0xa69, 0x963, 0x86a, 0xf66, 0xe6f, 0xd65, 0xc6c, 0x36c, 0x265, 0x16f, 0x066, 0x76a, 0x663, 0x569, 0x460,
  0xca0, 0xda9, 0xea3, 0xfaa, 0x8a6, 0x9af, 0xaa5, 0xbac, 0x4ac, 0x5a5, 0x6af, 0x7a6, 0x0aa, 0x1a3, 0x2a9, 0x3a0,
  0xd30, 0xc39, 0xf33, 0xe3a, 0x936, 0x83f, 0xb35, 0xa3c, 0x53c, 0x435, 0x73f, 0x636, 0x13a, 0x033, 0x339, 0x230,
  0xe90, 0xf99, 0xc93, 0xd9a, 0xa96, 0xb9f, 0x895, 0x99c, 0x69c, 0x795, 0x49f, 0x596, 0x29a, 0x393, 0x099, 0x190,
  0xf00, 0xe09, 0xd03, 0xc0a, 0xb06, 0xa0f, 0x905, 0x80c, 0x70c, 0x605, 0x50f, 0x406, 0x30a, 0x203, 0x109, 0x000
};

//  For each of the possible vertex states listed in aiCubeEdgeFlags there is a specific triangulation
//  of the edge intersection points.  a2iTriangleConnectionTable lists all of them in the form of
//  0-5 edge triples with the list terminated by the invalid value -1.
//  For example: a2iTriangleConnectionTable[3] list the 2 triangles formed when corner[0]
//  and corner[1] are inside of the surface, but the rest of the cube is not.
//
//  I found this table in an example program someone wrote long ago.  It was probably generated by hand

static int a2iTriangleConnectionTable[256][16] =
{
  {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 1, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 8, 3, 9, 8, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 3, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {9, 2, 10, 0, 2, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {2, 8, 3, 2, 10, 8, 10, 9, 8, -1, -1, -1, -1, -1, -1, -1},
  {3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 11, 2, 8, 11, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 9, 0, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 11, 2, 1, 9, 11, 9, 8, 11, -1, -1, -1, -1, -1, -1, -1},
  {3, 10, 1, 11, 10, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 10, 1, 0, 8, 10, 8, 11, 10, -1, -1, -1, -1, -1, -1, -1},
  {3, 9, 0, 3, 11, 9, 11, 10, 9, -1, -1, -1, -1, -1, -1, -1},
  {9, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 3, 0, 7, 3, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 1, 9, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 1, 9, 4, 7, 1, 7, 3, 1, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 10, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {3, 4, 7, 3, 0, 4, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1},
  {9, 2, 10, 9, 0, 2, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1},
  {2, 10, 9, 2, 9, 7, 2, 7, 3, 7, 9, 4, -1, -1, -1, -1},
  {8, 4, 7, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {11, 4, 7, 11, 2, 4, 2, 0, 4, -1, -1, -1, -1, -1, -1, -1},
  {9, 0, 1, 8, 4, 7, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1},
  {4, 7, 11, 9, 4, 11, 9, 11, 2, 9, 2, 1, -1, -1, -1, -1},
  {3, 10, 1, 3, 11, 10, 7, 8, 4, -1, -1, -1, -1, -1, -1, -1},
  {1, 11, 10, 1, 4, 11, 1, 0, 4, 7, 11, 4, -1, -1, -1, -1},
  {4, 7, 8, 9, 0, 11, 9, 11, 10, 11, 0, 3, -1, -1, -1, -1},
  {4, 7, 11, 4, 11, 9, 9, 11, 10, -1, -1, -1, -1, -1, -1, -1},
  {9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {9, 5, 4, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 5, 4, 1, 5, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {8, 5, 4, 8, 3, 5, 3, 1, 5, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 10, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {3, 0, 8, 1, 2, 10, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1},
  {5, 2, 10, 5, 4, 2, 4, 0, 2, -1, -1, -1, -1, -1, -1, -1},
  {2, 10, 5, 3, 2, 5, 3, 5, 4, 3, 4, 8, -1, -1, -1, -1},
  {9, 5, 4, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 11, 2, 0, 8, 11, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1},
  {0, 5, 4, 0, 1, 5, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1},
  {2, 1, 5, 2, 5, 8, 2, 8, 11, 4, 8, 5, -1, -1, -1, -1},
  {10, 3, 11, 10, 1, 3, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1},
  {4, 9, 5, 0, 8, 1, 8, 10, 1, 8, 11, 10, -1, -1, -1, -1},
  {5, 4, 0, 5, 0, 11, 5, 11, 10, 11, 0, 3, -1, -1, -1, -1},
  {5, 4, 8, 5, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1},
  {9, 7, 8, 5, 7, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {9, 3, 0, 9, 5, 3, 5, 7, 3, -1, -1, -1, -1, -1, -1, -1},
  {0, 7, 8, 0, 1, 7, 1, 5, 7, -1, -1, -1, -1, -1, -1, -1},
  {1, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {9, 7, 8, 9, 5, 7, 10, 1, 2, -1, -1, -1, -1, -1, -1, -1},
  {10, 1, 2, 9, 5, 0, 5, 3, 0, 5, 7, 3, -1, -1, -1, -1},
  {8, 0, 2, 8, 2, 5, 8, 5, 7, 10, 5, 2, -1, -1, -1, -1},
  {2, 10, 5, 2, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1},
  {7, 9, 5, 7, 8, 9, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1},
  {9, 5, 7, 9, 7, 2, 9, 2, 0, 2, 7, 11, -1, -1, -1, -1},
  {2, 3, 11, 0, 1, 8, 1, 7, 8, 1, 5, 7, -1, -1, -1, -1},
  {11, 2, 1, 11, 1, 7, 7, 1, 5, -1, -1, -1, -1, -1, -1, -1},
  {9, 5, 8, 8, 5, 7, 10, 1, 3, 10, 3, 11, -1, -1, -1, -1},
  {5, 7, 0, 5, 0, 9, 7, 11, 0, 1, 0, 10, 11, 10, 0, -1},
  {11, 10, 0, 11, 0, 3, 10, 5, 0, 8, 0, 7, 5, 7, 0, -1},
  {11, 10, 5, 7, 11, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 3, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {9, 0, 1, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 8, 3, 1, 9, 8, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1},
  {1, 6, 5, 2, 6, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 6, 5, 1, 2, 6, 3, 0, 8, -1, -1, -1, -1, -1, -1, -1},
  {9, 6, 5, 9, 0, 6, 0, 2, 6, -1, -1, -1, -1, -1, -1, -1},
  {5, 9, 8, 5, 8, 2, 5, 2, 6, 3, 2, 8, -1, -1, -1, -1},
  {2, 3, 11, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {11, 0, 8, 11, 2, 0, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1},
  {0, 1, 9, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1},
  {5, 10, 6, 1, 9, 2, 9, 11, 2, 9, 8, 11, -1, -1, -1, -1},
  {6, 3, 11, 6, 5, 3, 5, 1, 3, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 11, 0, 11, 5, 0, 5, 1, 5, 11, 6, -1, -1, -1, -1},
  {3, 11, 6, 0, 3, 6, 0, 6, 5, 0, 5, 9, -1, -1, -1, -1},
  {6, 5, 9, 6, 9, 11, 11, 9, 8, -1, -1, -1, -1, -1, -1, -1},
  {5, 10, 6, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 3, 0, 4, 7, 3, 6, 5, 10, -1, -1, -1, -1, -1, -1, -1},
  {1, 9, 0, 5, 10, 6, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1},
  {10, 6, 5, 1, 9, 7, 1, 7, 3, 7, 9, 4, -1, -1, -1, -1},
  {6, 1, 2, 6, 5, 1, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 5, 5, 2, 6, 3, 0, 4, 3, 4, 7, -1, -1, -1, -1},
  {8, 4, 7, 9, 0, 5, 0, 6, 5, 0, 2, 6, -1, -1, -1, -1},
  {7, 3, 9, 7, 9, 4, 3, 2, 9, 5, 9, 6, 2, 6, 9, -1},
  {3, 11, 2, 7, 8, 4, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1},
  {5, 10, 6, 4, 7, 2, 4, 2, 0, 2, 7, 11, -1, -1, -1, -1},
  {0, 1, 9, 4, 7, 8, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1},
  {9, 2, 1, 9, 11, 2, 9, 4, 11, 7, 11, 4, 5, 10, 6, -1},
  {8, 4, 7, 3, 11, 5, 3, 5, 1, 5, 11, 6, -1, -1, -1, -1},
  {5, 1, 11, 5, 11, 6, 1, 0, 11, 7, 11, 4, 0, 4, 11, -1},
  {0, 5, 9, 0, 6, 5, 0, 3, 6, 11, 6, 3, 8, 4, 7, -1},
  {6, 5, 9, 6, 9, 11, 4, 7, 9, 7, 11, 9, -1, -1, -1, -1},
  {10, 4, 9, 6, 4, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 10, 6, 4, 9, 10, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1},
  {10, 0, 1, 10, 6, 0, 6, 4, 0, -1, -1, -1, -1, -1, -1, -1},
  {8, 3, 1, 8, 1, 6, 8, 6, 4, 6, 1, 10, -1, -1, -1, -1},
  {1, 4, 9, 1, 2, 4, 2, 6, 4, -1, -1, -1, -1, -1, -1, -1},
  {3, 0, 8, 1, 2, 9, 2, 4, 9, 2, 6, 4, -1, -1, -1, -1},
  {0, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {8, 3, 2, 8, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1},
  {10, 4, 9, 10, 6, 4, 11, 2, 3, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 2, 2, 8, 11, 4, 9, 10, 4, 10, 6, -1, -1, -1, -1},
  {3, 11, 2, 0, 1, 6, 0, 6, 4, 6, 1, 10, -1, -1, -1, -1},
  {6, 4, 1, 6, 1, 10, 4, 8, 1, 2, 1, 11, 8, 11, 1, -1},
  {9, 6, 4, 9, 3, 6, 9, 1, 3, 11, 6, 3, -1, -1, -1, -1},
  {8, 11, 1, 8, 1, 0, 11, 6, 1, 9, 1, 4, 6, 4, 1, -1},
  {3, 11, 6, 3, 6, 0, 0, 6, 4, -1, -1, -1, -1, -1, -1, -1},
  {6, 4, 8, 11, 6, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {7, 10, 6, 7, 8, 10, 8, 9, 10, -1, -1, -1, -1, -1, -1, -1},
  {0, 7, 3, 0, 10, 7, 0, 9, 10, 6, 7, 10, -1, -1, -1, -1},
  {10, 6, 7, 1, 10, 7, 1, 7, 8, 1, 8, 0, -1, -1, -1, -1},
  {10, 6, 7, 10, 7, 1, 1, 7, 3, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 6, 1, 6, 8, 1, 8, 9, 8, 6, 7, -1, -1, -1, -1},
  {2, 6, 9, 2, 9, 1, 6, 7, 9, 0, 9, 3, 7, 3, 9, -1},
  {7, 8, 0, 7, 0, 6, 6, 0, 2, -1, -1, -1, -1, -1, -1, -1},
  {7, 3, 2, 6, 7, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {2, 3, 11, 10, 6, 8, 10, 8, 9, 8, 6, 7, -1, -1, -1, -1},
  {2, 0, 7, 2, 7, 11, 0, 9, 7, 6, 7, 10, 9, 10, 7, -1},
  {1, 8, 0, 1, 7, 8, 1, 10, 7, 6, 7, 10, 2, 3, 11, -1},
  {11, 2, 1, 11, 1, 7, 10, 6, 1, 6, 7, 1, -1, -1, -1, -1},
  {8, 9, 6, 8, 6, 7, 9, 1, 6, 11, 6, 3, 1, 3, 6, -1},
  {0, 9, 1, 11, 6, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {7, 8, 0, 7, 0, 6, 3, 11, 0, 11, 6, 0, -1, -1, -1, -1},
  {7, 11, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {3, 0, 8, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 1, 9, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {8, 1, 9, 8, 3, 1, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1},
  {10, 1, 2, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 10, 3, 0, 8, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1},
  {2, 9, 0, 2, 10, 9, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1},
  {6, 11, 7, 2, 10, 3, 10, 8, 3, 10, 9, 8, -1, -1, -1, -1},
  {7, 2, 3, 6, 2, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {7, 0, 8, 7, 6, 0, 6, 2, 0, -1, -1, -1, -1, -1, -1, -1},
  {2, 7, 6, 2, 3, 7, 0, 1, 9, -1, -1, -1, -1, -1, -1, -1},
  {1, 6, 2, 1, 8, 6, 1, 9, 8, 8, 7, 6, -1, -1, -1, -1},
  {10, 7, 6, 10, 1, 7, 1, 3, 7, -1, -1, -1, -1, -1, -1, -1},
  {10, 7, 6, 1, 7, 10, 1, 8, 7, 1, 0, 8, -1, -1, -1, -1},
  {0, 3, 7, 0, 7, 10, 0, 10, 9, 6, 10, 7, -1, -1, -1, -1},
  {7, 6, 10, 7, 10, 8, 8, 10, 9, -1, -1, -1, -1, -1, -1, -1},
  {6, 8, 4, 11, 8, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {3, 6, 11, 3, 0, 6, 0, 4, 6, -1, -1, -1, -1, -1, -1, -1},
  {8, 6, 11, 8, 4, 6, 9, 0, 1, -1, -1, -1, -1, -1, -1, -1},
  {9, 4, 6, 9, 6, 3, 9, 3, 1, 11, 3, 6, -1, -1, -1, -1},
  {6, 8, 4, 6, 11, 8, 2, 10, 1, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 10, 3, 0, 11, 0, 6, 11, 0, 4, 6, -1, -1, -1, -1},
  {4, 11, 8, 4, 6, 11, 0, 2, 9, 2, 10, 9, -1, -1, -1, -1},
  {10, 9, 3, 10, 3, 2, 9, 4, 3, 11, 3, 6, 4, 6, 3, -1},
  {8, 2, 3, 8, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1},
  {0, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 9, 0, 2, 3, 4, 2, 4, 6, 4, 3, 8, -1, -1, -1, -1},
  {1, 9, 4, 1, 4, 2, 2, 4, 6, -1, -1, -1, -1, -1, -1, -1},
  {8, 1, 3, 8, 6, 1, 8, 4, 6, 6, 10, 1, -1, -1, -1, -1},
  {10, 1, 0, 10, 0, 6, 6, 0, 4, -1, -1, -1, -1, -1, -1, -1},
  {4, 6, 3, 4, 3, 8, 6, 10, 3, 0, 3, 9, 10, 9, 3, -1},
  {10, 9, 4, 6, 10, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 9, 5, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 3, 4, 9, 5, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1},
  {5, 0, 1, 5, 4, 0, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1},
  {11, 7, 6, 8, 3, 4, 3, 5, 4, 3, 1, 5, -1, -1, -1, -1},
  {9, 5, 4, 10, 1, 2, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1},
  {6, 11, 7, 1, 2, 10, 0, 8, 3, 4, 9, 5, -1, -1, -1, -1},
  {7, 6, 11, 5, 4, 10, 4, 2, 10, 4, 0, 2, -1, -1, -1, -1},
  {3, 4, 8, 3, 5, 4, 3, 2, 5, 10, 5, 2, 11, 7, 6, -1},
  {7, 2, 3, 7, 6, 2, 5, 4, 9, -1, -1, -1, -1, -1, -1, -1},
  {9, 5, 4, 0, 8, 6, 0, 6, 2, 6, 8, 7, -1, -1, -1, -1},
  {3, 6, 2, 3, 7, 6, 1, 5, 0, 5, 4, 0, -1, -1, -1, -1},
  {6, 2, 8, 6, 8, 7, 2, 1, 8, 4, 8, 5, 1, 5, 8, -1},
  {9, 5, 4, 10, 1, 6, 1, 7, 6, 1, 3, 7, -1, -1, -1, -1},
  {1, 6, 10, 1, 7, 6, 1, 0, 7, 8, 7, 0, 9, 5, 4, -1},
  {4, 0, 10, 4, 10, 5, 0, 3, 10, 6, 10, 7, 3, 7, 10, -1},
  {7, 6, 10, 7, 10, 8, 5, 4, 10, 4, 8, 10, -1, -1, -1, -1},
  {6, 9, 5, 6, 11, 9, 11, 8, 9, -1, -1, -1, -1, -1, -1, -1},
  {3, 6, 11, 0, 6, 3, 0, 5, 6, 0, 9, 5, -1, -1, -1, -1},
  {0, 11, 8, 0, 5, 11, 0, 1, 5, 5, 6, 11, -1, -1, -1, -1},
  {6, 11, 3, 6, 3, 5, 5, 3, 1, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 10, 9, 5, 11, 9, 11, 8, 11, 5, 6, -1, -1, -1, -1},
  {0, 11, 3, 0, 6, 11, 0, 9, 6, 5, 6, 9, 1, 2, 10, -1},
  {11, 8, 5, 11, 5, 6, 8, 0, 5, 10, 5, 2, 0, 2, 5, -1},
  {6, 11, 3, 6, 3, 5, 2, 10, 3, 10, 5, 3, -1, -1, -1, -1},
  {5, 8, 9, 5, 2, 8, 5, 6, 2, 3, 8, 2, -1, -1, -1, -1},
  {9, 5, 6, 9, 6, 0, 0, 6, 2, -1, -1, -1, -1, -1, -1, -1},
  {1, 5, 8, 1, 8, 0, 5, 6, 8, 3, 8, 2, 6, 2, 8, -1},
  {1, 5, 6, 2, 1, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 3, 6, 1, 6, 10, 3, 8, 6, 5, 6, 9, 8, 9, 6, -1},
  {10, 1, 0, 10, 0, 6, 9, 5, 0, 5, 6, 0, -1, -1, -1, -1},
  {0, 3, 8, 5, 6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {10, 5, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {11, 5, 10, 7, 5, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {11, 5, 10, 11, 7, 5, 8, 3, 0, -1, -1, -1, -1, -1, -1, -1},
  {5, 11, 7, 5, 10, 11, 1, 9, 0, -1, -1, -1, -1, -1, -1, -1},
  {10, 7, 5, 10, 11, 7, 9, 8, 1, 8, 3, 1, -1, -1, -1, -1},
  {11, 1, 2, 11, 7, 1, 7, 5, 1, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 3, 1, 2, 7, 1, 7, 5, 7, 2, 11, -1, -1, -1, -1},
  {9, 7, 5, 9, 2, 7, 9, 0, 2, 2, 11, 7, -1, -1, -1, -1},
  {7, 5, 2, 7, 2, 11, 5, 9, 2, 3, 2, 8, 9, 8, 2, -1},
  {2, 5, 10, 2, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1},
  {8, 2, 0, 8, 5, 2, 8, 7, 5, 10, 2, 5, -1, -1, -1, -1},
  {9, 0, 1, 5, 10, 3, 5, 3, 7, 3, 10, 2, -1, -1, -1, -1},
  {9, 8, 2, 9, 2, 1, 8, 7, 2, 10, 2, 5, 7, 5, 2, -1},
  {1, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 7, 0, 7, 1, 1, 7, 5, -1, -1, -1, -1, -1, -1, -1},
  {9, 0, 3, 9, 3, 5, 5, 3, 7, -1, -1, -1, -1, -1, -1, -1},
  {9, 8, 7, 5, 9, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {5, 8, 4, 5, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1},
  {5, 0, 4, 5, 11, 0, 5, 10, 11, 11, 3, 0, -1, -1, -1, -1},
  {0, 1, 9, 8, 4, 10, 8, 10, 11, 10, 4, 5, -1, -1, -1, -1},
  {10, 11, 4, 10, 4, 5, 11, 3, 4, 9, 4, 1, 3, 1, 4, -1},
  {2, 5, 1, 2, 8, 5, 2, 11, 8, 4, 5, 8, -1, -1, -1, -1},
  {0, 4, 11, 0, 11, 3, 4, 5, 11, 2, 11, 1, 5, 1, 11, -1},
  {0, 2, 5, 0, 5, 9, 2, 11, 5, 4, 5, 8, 11, 8, 5, -1},
  {9, 4, 5, 2, 11, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {2, 5, 10, 3, 5, 2, 3, 4, 5, 3, 8, 4, -1, -1, -1, -1},
  {5, 10, 2, 5, 2, 4, 4, 2, 0, -1, -1, -1, -1, -1, -1, -1},
  {3, 10, 2, 3, 5, 10, 3, 8, 5, 4, 5, 8, 0, 1, 9, -1},
  {5, 10, 2, 5, 2, 4, 1, 9, 2, 9, 4, 2, -1, -1, -1, -1},
  {8, 4, 5, 8, 5, 3, 3, 5, 1, -1, -1, -1, -1, -1, -1, -1},
  {0, 4, 5, 1, 0, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {8, 4, 5, 8, 5, 3, 9, 0, 5, 0, 3, 5, -1, -1, -1, -1},
  {9, 4, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 11, 7, 4, 9, 11, 9, 10, 11, -1, -1, -1, -1, -1, -1, -1},
  {0, 8, 3, 4, 9, 7, 9, 11, 7, 9, 10, 11, -1, -1, -1, -1},
  {1, 10, 11, 1, 11, 4, 1, 4, 0, 7, 4, 11, -1, -1, -1, -1},
  {3, 1, 4, 3, 4, 8, 1, 10, 4, 7, 4, 11, 10, 11, 4, -1},
  {4, 11, 7, 9, 11, 4, 9, 2, 11, 9, 1, 2, -1, -1, -1, -1},
  {9, 7, 4, 9, 11, 7, 9, 1, 11, 2, 11, 1, 0, 8, 3, -1},
  {11, 7, 4, 11, 4, 2, 2, 4, 0, -1, -1, -1, -1, -1, -1, -1},
  {11, 7, 4, 11, 4, 2, 8, 3, 4, 3, 2, 4, -1, -1, -1, -1},
  {2, 9, 10, 2, 7, 9, 2, 3, 7, 7, 4, 9, -1, -1, -1, -1},
  {9, 10, 7, 9, 7, 4, 10, 2, 7, 8, 7, 0, 2, 0, 7, -1},
  {3, 7, 10, 3, 10, 2, 7, 4, 10, 1, 10, 0, 4, 0, 10, -1},
  {1, 10, 2, 8, 7, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 9, 1, 4, 1, 7, 7, 1, 3, -1, -1, -1, -1, -1, -1, -1},
  {4, 9, 1, 4, 1, 7, 0, 8, 1, 8, 7, 1, -1, -1, -1, -1},
  {4, 0, 3, 7, 4, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {4, 8, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {9, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {3, 0, 9, 3, 9, 11, 11, 9, 10, -1, -1, -1, -1, -1, -1, -1},
  {0, 1, 10, 0, 10, 8, 8, 10, 11, -1, -1, -1, -1, -1, -1, -1},
  {3, 1, 10, 11, 3, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 2, 11, 1, 11, 9, 9, 11, 8, -1, -1, -1, -1, -1, -1, -1},
  {3, 0, 9, 3, 9, 11, 1, 2, 9, 2, 11, 9, -1, -1, -1, -1},
  {0, 2, 11, 8, 0, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {3, 2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {2, 3, 8, 2, 8, 10, 10, 8, 9, -1, -1, -1, -1, -1, -1, -1},
  {9, 10, 2, 0, 9, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {2, 3, 8, 2, 8, 10, 0, 1, 8, 1, 10, 8, -1, -1, -1, -1},
  {1, 10, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {1, 3, 8, 9, 1, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 9, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {0, 3, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
  {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}
};

///////////////////////////////////////////////////////////////////////////////
// fGetOffset finds the approximate point of intersection of the surface
// between two points with the values fValue1 and fValue2
static float fGetOffset( float fValue1, float fValue2, float fValueDesired )
{
  double fDelta = fValue2 - fValue1;
  if (fDelta == 0.0) {
    return 0.5;
  }
  return (float)(( fValueDesired - fValue1 ) / fDelta );
}

///////////////////////////////////////////////////////////////////////////////
// vGetNormal() finds the gradient of the scalar field at a point.
// This gradient can be used as a very accurate vertx normal for lighting calculations
// TODO...
/*
static void vGetNormal( SbVec3f& rfNormal, float fX, float fY, float fZ )
{
  //rfNormal.fX = fSample(fX-0.01f, fY, fZ) - fSample(fX+0.01f, fY, fZ);
  //rfNormal.fY = fSample(fX, fY-0.01f, fZ) - fSample(fX, fY+0.01f, fZ);
  //rfNormal.fZ = fSample(fX, fY, fZ-0.01f) - fSample(fX, fY, fZ+0.01f);
  //vNormalizeVector(rfNormal, rfNormal);
}
*/

///////////////////////////////////////////////////////////////////////////////
/** Apply marching cubes to one cube.
  *
  *  Returns: number of triangles created (maximum 5, minimum 0).
  *
  *  corner: xyz position of cube lower left back corner.
  *  size  : xyz size of cube (voxel size).
  *  values: value at each corner of cube (voxel values).
  *  isoval: isovalue
  *  verts : xyz vertices of triangles (if any).
  *
  * Values array order is:
  *   0: lower-left-back , 1: lower-right-back , 2: upper-right-back , 3: upper-left-back,
  *   4: lower-left-front, 5: lower-right-front, 6: upper-right-front, 7: upper-left-front.
  */
int
VolumeMarchingCubes::marchCube( SbVec3f& corner, SbVec3f& size, float values[8], float isovalue, SbVec3f vertex[15] )
{
  int numVert = 0;
  int numTri  = 0;

  SbVec3f asEdgeVertex[12];

  // Find which vertices are inside of the surface and which are outside
  int iFlagIndex = 0;
  for (int i = 0; i < 8; i++) {
    if (values[i] <= isovalue) {
      iFlagIndex |= 1 << i;
    }
  }

  // Find which edges are intersected by the surface
  int iEdgeFlags = aiCubeEdgeFlags[iFlagIndex];

  // If the cube is entirely inside or outside of the surface, there will be no intersections
  if (iEdgeFlags == 0) {
    return 0;
  }

  // Find the point of intersection of the surface with each edge
  // Then find the normal to the surface at those points
  for (int iEdge = 0; iEdge < 12; iEdge++) {
    // if there is an intersection on this edge
    if (iEdgeFlags & ( 1 << iEdge )) {
      float fOffset = fGetOffset( values[ a2iEdgeConnection[iEdge][0] ],
                                  values[ a2iEdgeConnection[iEdge][1] ], isovalue );

      asEdgeVertex[iEdge][0] = corner[0] + ( a2fVertexOffset[ a2iEdgeConnection[iEdge][0] ][0]  +  fOffset * a2fEdgeDirection[iEdge][0] ) * size[0];
      asEdgeVertex[iEdge][1] = corner[1] + ( a2fVertexOffset[ a2iEdgeConnection[iEdge][0] ][1]  +  fOffset * a2fEdgeDirection[iEdge][1] ) * size[1];
      asEdgeVertex[iEdge][2] = corner[2] + ( a2fVertexOffset[ a2iEdgeConnection[iEdge][0] ][2]  +  fOffset * a2fEdgeDirection[iEdge][2] ) * size[2];
      //vGetNormal(asEdgeNorm[iEdge], asEdgeVertex[iEdge].fX, asEdgeVertex[iEdge].fY, asEdgeVertex[iEdge].fZ);
    }
  }

  // Store the triangles that were found.  There can be up to five per cube
  for (int iTriangle = 0; iTriangle < 5; iTriangle++) {
    if (a2iTriangleConnectionTable[iFlagIndex][3 * iTriangle] < 0) {
      break;
    }
    numTri++;

    for (int iCorner = 0; iCorner < 3; iCorner++) {
      int iVertex = a2iTriangleConnectionTable[iFlagIndex][3 * iTriangle + iCorner];
      //m_vertices.push_back( asEdgeVertex[iVertex] );
      vertex[numVert] = asEdgeVertex[iVertex];
      numVert++;
    }
  }
  return numTri;
}
