///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
//
// ObliqueSliceBorder
//
/////////////////////////////////////////////////////////////////////

#include <Inventor/actions/SoGLRenderAction.h>

#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoPolygonOffset.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <VolumeViz/elements/SoVolumeDataElement.h>

#include <Medical/nodes/PlaneBoxIntersection.h>
#include <Medical/nodes/ObliqueSliceBorder.h>


SO_NODE_SOURCE(ObliqueSliceBorder);

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
ObliqueSliceBorder::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

  // Initialize type id variables
  SO_NODE_INIT_CLASS(ObliqueSliceBorder, SoObliqueSlice, "ObliqueSlice");
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
ObliqueSliceBorder::exitClass()
{
  SO__NODE_EXIT_CLASS(ObliqueSliceBorder);
}

////////////////////////////////////////////////////////////////////////
// Constructor
ObliqueSliceBorder::ObliqueSliceBorder()
{
  // Setup fields
  SO_NODE_CONSTRUCTOR(ObliqueSliceBorder);
  SO_NODE_ADD_FIELD(border     , (TRUE));
  SO_NODE_ADD_FIELD(borderColor, (SbColor(0.84f, 0.43f, 0.02f))); // Orange luminance 55%

  // Build the internal scene graph.
  // Ref the Separator to avoid accidental deletion.
  m_scene = new SoSeparator();

  // Disable lighting for the slice border (easier to see).
  SoLightModel* lighting = new SoLightModel();
    lighting->model = SoLightModel::BASE_COLOR;
    m_scene->addChild( lighting );

  // Make the border wide for visibility.
  SoDrawStyle* style = new SoDrawStyle();
    style->lineWidth = 2;
    style->style.setIgnored( TRUE );
    style->linePattern.setIgnored( TRUE );
    style->linePatternScaleFactor.setIgnored( TRUE );
    m_scene->addChild( style );

  // The geometry.
  // Important to set the initial color in the vertex property, otherwise the
  // field is empty and trying to fetch the first value could hang.
  m_geometry = new PlaneBoxIntersection();
    unsigned int color = this->borderColor.getValue().getPackedValue();
    SoVertexProperty* vprop = (SoVertexProperty*)m_geometry->vertexProperty.getValue();
    vprop->orderedRGBA = color;
    m_scene->addChild( m_geometry.ptr() );

  // Connect the slice's plane to the boxIntersection plane.
  m_geometry->plane.connectFrom( &(this->plane) );
}

////////////////////////////////////////////////////////////////////////
// Destructor
ObliqueSliceBorder::~ObliqueSliceBorder()
{
}

////////////////////////////////////////////////////////////////////////
void
ObliqueSliceBorder::GLRender(SoGLRenderAction* action)
{
  // Do parent rendering
  SoObliqueSlice::GLRender( action );

  // If border is turned off, nothing else to do.
  if (this->border.getValue() == FALSE)
    return;

  // Get current volume data in the traversal state.
  // This should also create a dependency on the volumeDataElement so this
  // method is called again if there is any change to the volume data node.
  SoVolumeData* volume;
  SoVolumeDataElement::get( action->getState(), volume );

  // If there is volume data...
  if (volume != NULL) {
    // Make sure volume extent is current.
    const SbBox3f& volExt = volume->extent.getValue();
    if (volExt != m_geometry->box.getValue()) {
      m_geometry->box = volExt;
    }

    // Make sure color is current.
    SoVertexProperty* vprop = (SoVertexProperty*)m_geometry->vertexProperty.getValue();
    unsigned int color = this->borderColor.getValue().getPackedValue();
    if (vprop->orderedRGBA.getNum() == 0 || vprop->orderedRGBA[0] != color) {
      vprop->orderedRGBA = color;
    }

    // Render the border (traverse the internal scene graph)
#if SO_INVENTOR_VERSION >= 9620
    action->forwardTraversal( m_scene.ptr() );
#else
    action->switchToNodeTraversal( m_scene.ptr() );
#endif
  }
}
