///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Example class for computing the intersection of geometry with a plane.
//
// Notes:
// This class implements a "brute force" solution for computing the intersection
// of any Open Inventor geometry with a plane (usually a clip plane).
//
// We use an SoCallbackAction to traverse the specified scene graph. This
// action can convert (almost) any Open Inventor geometry into triangles.
// A triangle callback computes the intersection of each triangle with the
// plane and stores the resulting line segment (if any) in a list. We check
// for (and eliminate) duplicate line segments that occur for example when
// an edge is co-planar and shared by two triangles.  The resulting list of
// line segments can be rendered directly by an SoLineSet node and visually
// appears to be the intersection curves. It is not, of course, a connected
// polyline but this allows the intersection computation to be interactive
// even for fairly large shapes.  The intersection curves are useful to see
// exactly how the plane intersects the geometry.
//
// In "polygon mode" we do a post-processing step to connect all the line
// line segments into continuous boundaries that can be rendered as polygons.
// This operation is inherently unreliable because of floating precision.
// See "strategy" below.  If the end result is multiple boundaries we set
// the shape hints 'windingType' to ODD_TYPE so Open Inventor will
// triangulate the boundaries as 'holes' and/or 'islands'.  This is another
// complex problem but the algorithm is fairly robust. In order for
// triangulation to work correctly the boundaries must have consistent
// orientation (either clockwise or counterclockwise).  We compute a
// normal vector for each boundary and, if necessary, reverse the order of
// the indices defining that boundary.  The normal vector for the computed
// polygon is set overall to the opposite of the clip plane's normal (this
// is because the clip plane normal points toward the unclipped half space
// and therefore the opposite direction is toward the exterior of the shape).
//
// Tolerance (aka epsilon):
// Comparing floating point values and floating point coordinates is a very
// hard problem.  See the extensive literature.  We use "abs(diff) < epsilon"
// for floating point values and "diff.lengthSquared() < epsilon" for floating
// point coordinates.  These have reasonably good behavior in most cases. The
// next challenge is to choose an appropriate value for epsilon depending on
// the coordinate range.  We currently use the average line segment length
// divided by 10000.  Typically this comes out to the typical 1e-6 value
// anyway, but in theory it allows us to use a larger or smaller tolerance
// based on how far apart the vertices are.
//
// Re-connection strategy:
// Our goal is to connect all the line segments into one or more boundaries.
// In theory this should be possible, but in practice it is a challenging
// task. The main problem is limited floating point precision.  This forces
// us to use a "tolerance" to compare coordinates.  If the tolerance is too
// small we fail to make some valid connections.  If the tolerance is too
// big we risk "false positives" that result in segments being connected to
// the wrong segment.  If this happens, some other segments will be lost
// (not connected).  In fact this is one of the biggest problems.  To avoid
// this problem as much as possible we take an extremely conservative (i.e.
// slow) approach.  Each segment is checked against every other unused
// segment to find the "best" possible connection.  Thus the performance is
// O(N squared).  But reliability is more important than performance.
//
// Notes:
//   - Memory
//     For efficiency, the vertices and indices remain in their respective
//     std::vector and the OIV nodes just have a ptr to that memory.  This
//     avoids copying the vertices and indices (again).
//
// TODO:
//   - Important: In polygon mode, when we're done trying to connect the
//     line segments, check if we actually have closed faces or not.
//     If not, the unclosed faces should be rendered as line sets.
//
//   - Is it actually useful to sort line segments?  How can we use this?
//
//   - When the result is multiple boundaries, check if they are disjoint and
//     can be rendered as separate polygons instead of using 'windingType'.
//
//   - Add an option to allow application to free the internal memory?
//
//   - Add an option to condense the vertex list in polygon mode?
//     (i.e. remove unused vertices to save memory)
//
//   - During traversal, apply current transform to geometry?
//     If the action is applied to a scene graph containing multiple shapes
//     with transforms this is mandatory to get the correct result.
//     Low priority for now because the usual use case will be to apply
//     this node to a single shape.
//
//   - During traversal, check bounding box of Separator nodes?
//     Low priority because the usual use case will be to apply this node
//     to a single shape.
//
// Future:
//   - Exporting clipped geometry to STL.
//     Currently you can clip geometry, but it's not possible to export the
//     clipped geometry to STL because SoSTLWriteAction does not consider
//     clip planes. The code in this class could easily be adapted to generate
//     a clipped set of triangles from the original geometry.
 
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI SAS                           ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/

#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/nodes/SoIndexedFaceSet.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/SoPrimitiveVertex.h>

#include <Inventor/STL/algorithm>
#include <Inventor/STL/vector>

#include <Medical/nodes/PlaneGeometryIntersection.h>

SO_NODE_SOURCE(PlaneGeometryIntersection);

// TODO: Is sorting a feature or not?  Possibly enables some optimizations...
static bool m_sort = true;

// TODO: Is it safe to reject "trivial" line segments or is it a tolerance problem...
static bool m_rejectTrivialSegments = false;

///////////////////////////////////////////////////////////////////////////////
// Data structure for storing intersections of triangles with the plane.
struct LineSegment {
  SbVec3f v0;
  SbVec3f v1;

  LineSegment( SbVec3f& _v0, SbVec3f& _v1 ) {
    // Pre-sort end points so we can later sort segments by first vertex X value.
    if (_v0[0] < _v1[0]) {
      v0 = _v0;
      v1 = _v1;
    }
    else {
      v1 = _v0;
      v0 = _v1;
    }
  }
  // Comparison operator to enable STL sort ascending by first vertex X value.
  bool operator< (LineSegment other) const {
    return v0[0] < other.v0[0];
  }
};

///////////////////////////////////////////////////////////////////////////////
// Internal details of implementation
// (so we don't have to expose STL stuff in the public header)
class PlaneGeometryIntersectionImpl
{
public:
  /** Constructor */
  PlaneGeometryIntersectionImpl()
  {
    // Need a callback action to compute the triangle intersections.
    m_cbAction = new SoCallbackAction();
    m_cbAction->addTriangleCallback( SoShape::getClassTypeId(),
                                     PlaneGeometryIntersectionImpl::triangleCB,
                                     (void*)this );
  }
  /** Destructor */
  ~PlaneGeometryIntersectionImpl()
  {
    delete m_cbAction;
  }

  /** Triangle callback for computing triangle intersections. */
  static void triangleCB( void *userData,
                SoCallbackAction *action,
                const SoPrimitiveVertex *v1, 
                const SoPrimitiveVertex *v2, 
                const SoPrimitiveVertex *v3 );

  /** Recompute intersection boundaries */
  int recompute( const SoNode* scene, const SbPlane& plane, bool polyMode, float epsilon );

  /** Intersect triangle with plane.
   *  Sets p0/p1 to end points of line segment if there is a valid intersection.
   *  Check return code to see if there was an intersection or possible duplicate line segment.
   */
  int intersectTri( const SbVec3f& v0, const SbVec3f& v1, const SbVec3f& v2, SbVec3f& p0, SbVec3f& p1 );

  /** Compute normal vector from three vertices somewhere in this index range. */
  SbVec3f computeNormal( int index0, int index1, SbVec3f& defaultNormal );

  /** Compute normal vector from three vertices. */
  SbVec3f computeNormal( SbVec3f& p0, SbVec3f& p1, SbVec3f& p2 );

  /** Compare float values using tolerance */
  inline bool equals( float a, float b, float epsilon )
  {
    return (fabs(a - b) < epsilon);
  }

  SoNode* m_scene;
  SbPlane m_plane;
  bool    m_makePolylines;
  float   m_epsilon;

  SoCallbackAction* m_cbAction;

  // Nodes we need to modify during recomputation
  SoIndexedFaceSet* m_faceSet;    // Render polygon boundaries
  SoIndexedLineSet* m_lineSet2;   // Render unclosed boundaries
  SoLineSet*        m_lineSet;    // Render unordered line segments
  SoShapeHints*     m_shapeHints;
  SoSwitch*         m_switch;

  int m_numSegs;
  int m_numTris;
  int m_numFaces;
  double m_sumOfEdges;  // Used to compute average edge length

  // Segments/vertices and indices
  std::vector<LineSegment> m_segments;
  std::vector<LineSegment> m_dupSegments;
  std::vector<int>         m_indices;
  std::vector<int>         m_faces;
};

///////////////////////////////////////////////////////////////////////////////
void
PlaneGeometryIntersection::initClass()
{
    getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );
    SO_NODE_INIT_CLASS(PlaneGeometryIntersection, SoSeparator, "Separator");
}

///////////////////////////////////////////////////////////////////////////////
void
PlaneGeometryIntersection::exitClass()
{
  SO__NODE_EXIT_CLASS(PlaneGeometryIntersection);
}

///////////////////////////////////////////////////////////////////////////////
PlaneGeometryIntersection::PlaneGeometryIntersection()
{
  // Initialize fields
    SO_NODE_CONSTRUCTOR(PlaneGeometryIntersection);
  SO_NODE_ADD_FIELD( scene    , (NULL)  );
  SO_NODE_ADD_FIELD( plane    , (SbPlane(SbVec3f(1, 0, 0), 0)));
  SO_NODE_ADD_FIELD( polygon  , (FALSE));
  SO_NODE_ADD_FIELD( automatic, (TRUE) );
  SO_NODE_ADD_FIELD( tolerance, (1e-6f) );

  // Hide the inherited Separator fields from IvTune (distracting).
  boundingBoxCaching.setFieldType( SoField::PRIVATE_FIELD );
  renderCulling.setFieldType     ( SoField::PRIVATE_FIELD );
  pickCulling.setFieldType       ( SoField::PRIVATE_FIELD );
  //directVizCaching.setFieldType  ( SoField::PRIVATE_FIELD );
  fastEditing.setFieldType       ( SoField::PRIVATE_FIELD );
  renderUnitId.setFieldType      ( SoField::PRIVATE_FIELD );

  // Sensors to detect when one of our fields has changed.
  m_sceneSensor = new SoFieldSensor( fieldSensorCB, (void*)this );
  m_sceneSensor->attach( &scene );
  m_planeSensor  = new SoFieldSensor( fieldSensorCB, (void*)this );
  m_planeSensor->attach( &plane );
  m_polygonSensor  = new SoFieldSensor( fieldSensorCB, (void*)this );
  m_polygonSensor->attach( &polygon );

  // Internal implementation details.
  m_impl = new PlaneGeometryIntersectionImpl();
  m_impl->m_plane = plane.getValue(); // Used to check for redundant value setting

  // Build internal scene graph.
  //
  // Switch controls whether lineset or faceset is rendered.
  m_impl->m_switch = new SoSwitch();
    m_impl->m_switch->whichChild = 0;
    addChild( m_impl->m_switch );
  
  // This node holds the vertices and is ref'd by lineset and faceset.
  SoVertexProperty* vprop = new SoVertexProperty();

  // LineSet nodes
  // Lighting is disabled for the LineSet to make it easier to see.
  SoSeparator* lineSep = new SoSeparator();
    lineSep->setName( "PGI_Curve" );
    m_impl->m_switch->addChild( lineSep );

  SoLightModel* model = new SoLightModel();
    model->model = SoLightModel::BASE_COLOR;
    lineSep->addChild( model );

  m_impl->m_lineSet = new SoLineSet();
    m_impl->m_lineSet->vertexProperty = vprop;
    lineSep->addChild( m_impl->m_lineSet );

  // FaceSet nodes
  // We need a ShapeHints node because:
  //   - The polygon is probably not convex (needs to be triangulated), and
  //   - The polygon may have "holes" and/or "islands".
  //
  // We need an IndexedFaceSet to render the closed boundaries (actual polygons).
  // We need an IndexedLineSet to render unclosed boundaries.
  SoSeparator* faceSep = new SoSeparator();
    faceSep->setName( "PGI_Polygon" );
    m_impl->m_switch->addChild( faceSep );

  m_impl->m_shapeHints = new SoShapeHints();
    m_impl->m_shapeHints->vertexOrdering = SoShapeHints::UNKNOWN_ORDERING;
    m_impl->m_shapeHints->faceType       = SoShapeHints::UNKNOWN_FACE_TYPE;
    m_impl->m_shapeHints->windingType    = SoShapeHints::ODD_TYPE;
    faceSep->addChild( m_impl->m_shapeHints );

  m_impl->m_faceSet = new SoIndexedFaceSet();
    m_impl->m_faceSet->vertexProperty = vprop;
    faceSep->addChild( m_impl->m_faceSet );

  m_impl->m_lineSet2 = new SoIndexedLineSet();
    m_impl->m_lineSet2->vertexProperty = vprop;
    faceSep->addChild( m_impl->m_lineSet2 );
}

///////////////////////////////////////////////////////////////////////////////
PlaneGeometryIntersection::~PlaneGeometryIntersection()
{
  delete m_sceneSensor;
  delete m_planeSensor;
  delete m_impl;
}

///////////////////////////////////////////////////////////////////////////////
void
PlaneGeometryIntersection::resetGeometry()
{
  // Typically called because there is no intersection.
  // For now just set the switch to not traverse either geometry node.
  // Currently allocated memory stays allocated to use in next computation.
  m_impl->m_switch->whichChild = -1;
}

///////////////////////////////////////////////////////////////////////////////
// Called when one of our fields changed.
void
PlaneGeometryIntersection::fieldSensorCB(void* data, SoSensor* sensor)
{
    PlaneGeometryIntersection* self = (PlaneGeometryIntersection*)data;

  // Which field changed?
  const SoFieldSensor* fsensor = (SoFieldSensor*)sensor;
  const SoField* field = fsensor->getAttachedField();

  // Problem: When our plane field is connected from the clip plane manip's
  // plane field, our plane field gets modified frequently with exactly the
  // same value.  This is useless and could be a bug, but we just have to
  // work around it for now. :-(
  if (field == &(self->plane)) {
    // Plane field has changed.
    const SbPlane& plane = self->plane.getValue();
    if (plane == self->m_impl->m_plane) // Redundant change to plane field
      return;
  }
  else if (field == &(self->scene)) {
    // Scene field has changed
    if (self->scene.getValue() == NULL) { // Allow application to set the scene to "no scene".
      self->resetGeometry();
      return;
    }
  }

  // If automatic mode, recompute intersections and (if requested) boundaries.
  if (self->automatic.getValue() == TRUE)
    self->recompute();
}

///////////////////////////////////////////////////////////////////////////////
// Recompute intersections and (if requested) boundaries.
int
PlaneGeometryIntersection::recompute()
{
  int rc = 1; // No geometry generated.

  const SoNode* theScene = scene.getValue();
  if (theScene == NULL) { // Nothing to do
    resetGeometry();
    return rc;
  }

  const SbPlane& thePlane = plane.getValue();
  const bool     polyMode = (polygon.getValue() == TRUE);
  const float    epsilon  = tolerance.getValue();

  rc = m_impl->recompute( theScene, thePlane, polyMode, epsilon );
  if (rc > 0) {  // No intersections found.
    resetGeometry();
  }
  return rc;
}

///////////////////////////////////////////////////////////////////////////////
// Return shape node so application can get the actual coordinates.
const SoShape*
PlaneGeometryIntersection::getShape() const
{
  if (polygon.getValue() == TRUE) {
    return m_impl->m_faceSet;
  }
  else {
    return m_impl->m_lineSet;
  }
  return NULL;
}

///////////////////////////////////////////////////////////////////////////////
/**
 * Intersect a triangle with a plane.
 * If there is a valid intersection, parameters p0 and p1 define a line
 * segment. No guarantees about the ordering of the coordinates.
 * Return value is intersection status:
 *   -  0 Yes: Two intersection points (one might be a vertex).
 *   -  2 Yes: But... intersection points are both vertices.
 *   -  1 No : Triangle is in front of the plane.
 *   - -1 No : Triangle is behind the plane
 *   - -2 No : Triangle is *in* the plane.
 *   - -3 No : Exactly one vertex is in the plane.
 *
 * Notes:
 *   - Caller should carefully handle the +2 return value.  If the
 *     intersection points are both vertices, then the line segment
 *     is an *edge* of the triangle and it is possible that one or
 *     more triangles share this edge.  In that case there will be
 *     duplicate line segments. You can ignore this if you only need
 *     to draw the clip line, but you will need to eliminate
 *     duplicates in order to construct correct polylines/polygons.
 *
 *   - We do not consider the one-vertex exactly in plane case an
 *     intersection but the caller can check the return value.
 *
 *   - We do not consider the triangle-in-plane case an intersection
 *     but the caller can check the return value.
 *
 *   - Caller may want to test for and reject degenerate segments,
 *     i.e. when end points are effectively the same coordinate.
 *     Such segments can cause segment connection to fail.
 */
int
PlaneGeometryIntersectionImpl::intersectTri(
  const SbVec3f& v0, const SbVec3f& v1, const SbVec3f& v2, SbVec3f& p0, SbVec3f& p1 )
{
  // A triangle has 3 vertices that construct 3 line segments
  int cntFront=0, cntBack=0;
  float distance[3];
  int   zeroIndex[3] = {-1, -1, -1};
  int   zeroCount = 0;
  const SbVec3f* v[3] = {&v0, &v1, &v2};

  for (int i = 0; i < 3; ++i) {
    distance[i] = m_plane.getDistance( *v[i] );
    if (equals(distance[i], 0, m_epsilon)) { // Vertex is "in" the plane
      distance[i] = 0;
      zeroIndex[zeroCount] = i;
      zeroCount++;
    }
    else if (distance[i] < 0) ++cntBack;    // Vertex is behind the plane
    else                      ++cntFront;   // Vertex is in front of the plane
  }
  if (cntBack == 3) {
    return -1;              // No intersection - All vertices behind plane
  }
  else if (cntFront == 3) { 
    return  1;              // No intersection - All vertices in front of plane
  }
  else if (zeroCount == 3) { 
    return -2;              // No intersection - All vertices are *in* the plane
  }
  else if (zeroCount > 1) { // Special case: Two points are co-planar
    p0 = *v[zeroIndex[0]]; // Line segment is exactly that edge.
    p1 = *v[zeroIndex[1]]; // NOTE: This means there may be a duplicate edge!
    return 2;
  }

  int lines[] = {0,1,1,2,2,0}; // CCW Triangle
  std::vector<SbVec3f> intersectPoints;
  int firstIndex = -1;

  for (int i = 0; i < 3; ++i) {
    const int index0 = lines[i*2+0];
    const int index1 = lines[i*2+1];
    const SbVec3f &a = *v[index0];
    const SbVec3f &b = *v[index1];
    const float da = distance[index0];
    const float db = distance[index1];
    if (da * db < 0) {
      const float s = da / (da-db); // intersection factor (between 0 and 1)
      SbVec3f bMinusa = b - a;
      intersectPoints.push_back(a + bMinusa * s);
    }
    else if (da == 0) { // plane falls exactly on one of the three Triangle vertices
      if (intersectPoints.size() < 2 && firstIndex != index0) {
        intersectPoints.push_back( a );
        firstIndex = index0;
      }
    }
    else if (db == 0) { // plane falls exactly on one of the three Triangle vertices
      if (intersectPoints.size() < 2 && firstIndex != index1) {
        intersectPoints.push_back( b );
        firstIndex = index1;
      }
    }
  }
  if (intersectPoints.size() == 2) {
    // General case: Output the intersecting line segment object
    p0 = intersectPoints[0];
    p1 = intersectPoints[1];
    return 0;
  }
  return -3;  // No intersection - plane just touches one vertex
}

///////////////////////////////////////////////////////////////////////////////
// Callback called for each triangle in the specified scene graph.
// Compute the intersection line segment.  If there is one, store it for later
// but discard duplicate segments (can only happen if the line segment is an
// edge of the triangle that might be shared with another triangle).
void
PlaneGeometryIntersectionImpl::triangleCB(
                void *userData, SoCallbackAction* /*action*/,
                const SoPrimitiveVertex *pv0, 
                const SoPrimitiveVertex *pv1, 
                const SoPrimitiveVertex *pv2 )
{
  PlaneGeometryIntersectionImpl* self = (PlaneGeometryIntersectionImpl*)userData;

  const SbVec3f& v0 = pv0->getPoint();
  const SbVec3f& v1 = pv1->getPoint();
  const SbVec3f& v2 = pv2->getPoint();
  SbVec3f p0, p1;
  float epsilon = self->m_epsilon;

  int rc = self->intersectTri( v0, v1, v2, p0, p1 );
  if (rc == 0) {
    if (m_rejectTrivialSegments) {
      if (p0.equals(p1,epsilon)) return;
    }
    // This is a typical intersection line segment.
    self->m_segments.push_back( LineSegment(p0,p1) );
    self->m_sumOfEdges += (p1 - p0).lengthSquared();
  }
  else if (rc == 2) {
    if (m_rejectTrivialSegments) {
      if (p0.equals(p1,epsilon)) return;
    }
    // This is an edge of a triangle and possibly a duplicate!
    bool foundDup = false; // We don't need to search if numDupSeg is 0.
    int numDupSeg = (int)self->m_dupSegments.size(); // size() is actually number of verts!
    if (numDupSeg > 0) {
      // Check if we already have this line segment (in any order).
      for (int i = 0; i < numDupSeg; ++i) {
        if (p0.equals(self->m_dupSegments[i].v0,epsilon)) {
          if (p1.equals(self->m_dupSegments[i].v1,epsilon)) {
            foundDup = true;
            break;
          }
        }
        else if (p0.equals(self->m_dupSegments[i].v1,epsilon)) {
          if (p1.equals(self->m_dupSegments[i].v0,epsilon)) {
            foundDup = true;
            break;
          }
        }
      }
    }
    if (! foundDup) {
      // So far this one is unique, so remember it.
      self->m_dupSegments.push_back( LineSegment(p0,p1) );
      self->m_sumOfEdges += (p1 - p0).lengthSquared();
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
// Compute a normal vector from three points using cross-product.
// There is a chance of getting a bogus normal because the first three
// points are co-linear.  In that case try the next three points, but
// don't try to use more points than we actually have. :)
// If all else fails, return the default value and move on.
SbVec3f
PlaneGeometryIntersectionImpl::computeNormal( SbVec3f& p0, SbVec3f& p1, SbVec3f& p2 )
{
  SbVec3f vec0 = p0 - p1;
  SbVec3f vec1 = p2 - p1;
  return vec1.cross( vec0 );
}
SbVec3f
PlaneGeometryIntersectionImpl::computeNormal( int index0, int index1, SbVec3f& defaultNormal )
{
  const SbVec3f zeroVec(0,0,0);
  SbVec3f normal = defaultNormal;
  SbVec3f* verts = (SbVec3f*)&m_segments[0]; // Cast because it's actually an array of LineSegment.

  for (int i0 = index0; i0 <= index1; ++i0) {
    // If we're at the end of the index range, "wrap around" to beginning of range.
    int i1, i2;
    if (i0 == index1 - 1) {
      i1 = index1;
      i2 = index0;
    }
    else if (i0 == index1) {
      i1 = index0;
      i2 = i1 + 1;
    }
    else {
      i1 = i0 + 1;
      i2 = i0 + 2;
    }
    SbVec3f& p0 = verts[m_indices[i0]];
    SbVec3f& p1 = verts[m_indices[i1]];
    SbVec3f& p2 = verts[m_indices[i2]];
    SbVec3f testNormal = computeNormal( p0, p1, p2 );
    if (! testNormal.equals(zeroVec,m_epsilon)) {
      normal = testNormal;
      break;
    }
  }
  return normal;
}

///////////////////////////////////////////////////////////////////////////////
// Recompute intersections and (if requested) boundaries.
int
PlaneGeometryIntersectionImpl::recompute(
  const SoNode* scene, const SbPlane& plane, bool polyMode, float epsilon )
{
  if (scene == NULL)
    return -1;

  // Store internally for triangle callback to use.
  m_scene = const_cast<SoNode*>(scene);
  m_plane = plane;
  m_makePolylines = polyMode;
  m_epsilon = epsilon;
  m_sumOfEdges = 0;

  // Compute all the intersection line segments.
  //   - m_segments contains all the line segments that pass through the
  //     interior of a triangle.
  //   - m_dupSegments contains all the line segments that lie on an edge
  //     of a triangle. There is special processing for this list to avoid
  //     duplicate line segments caused by two triangles sharing a common edge.
  m_numTris  = 0;
  m_numFaces = 0;
  m_segments.clear();
  m_dupSegments.clear();
  m_indices.clear();
  m_faces.clear();

  m_lineSet->numVertices.setNum( 0 );
  m_faceSet->coordIndex.setNum( 0 );
  m_lineSet2->coordIndex.setNum( 0 );

  // TODO: Try this to work around crashes during apply() of the callback action.
  //m_cbAction->apply( m_scene );
  SoCallbackAction cba;
  cba.addTriangleCallback( SoShape::getClassTypeId(), PlaneGeometryIntersectionImpl::triangleCB, (void*)this );
  cba.apply( m_scene );

  // Combine the two lists of segments.
#ifdef _DEBUG
  int numSegDup    = (int)m_dupSegments.size();
#endif
  m_segments.insert( m_segments.end(), m_dupSegments.begin(), m_dupSegments.end() );
  m_dupSegments.clear();
  m_numSegs  = (int)m_segments.size();
  
  // Cleanup and leave if there were no intersections.
  if (m_numSegs == 0) {
    return 1;
  }
  else if (polyMode && m_numSegs < 3) { // Or not enough segments to form a triangle
    return 1;
  }

  // Calculate an epsilon value based on the average edge length.
  double avgEdge = m_sumOfEdges / m_numSegs;
  float newEpsilon = (float)(avgEdge / 10000);
  //std::cout << "Edges: " << LineSegment::m_numEdges << " " << LineSegment::m_minEdgeLen << " "
  //          << LineSegment::m_maxEdgeLen << " " << LineSegment::m_sumOfEdges << " "
  //          << avgEdge << " neweps: " << newEpsilon << " old: " << m_epsilon << std::endl;
  m_epsilon = newEpsilon;

  // Store the vertices and normal vector.
  // The m_segments vector is effectively an array of SbVec3f.
  // Note that we do NOT make a copy of the vertices, just point to the vector.
  // Negate normalVector because the clip plane's normal vector points toward
  // the unclipped half space, which is effectively the interior of the shape.
  const int numVerts     = m_numSegs * 2;
  SbVec3f   normalVector = plane.getNormal();
  normalVector.negate();
  normalVector.normalize();

  SoVertexProperty* vprop = (SoVertexProperty*)m_lineSet->vertexProperty.getValue();
    vprop->vertex.setValuesPointer( numVerts, (SbVec3f*)&m_segments[0] );
    vprop->normal.set1Value( 0, normalVector );
    vprop->normalBinding = SoVertexProperty::OVERALL;

  if (! polyMode) {
    // We already have all the line segments in the intersection.

    // Make the line set be rendered
    m_switch->whichChild = 0;
    
    // It's inefficient for rendering, but if we only need to render
    // the intersection as a "line", then it minimizes compute time to
    // just brute force render all the segments.
    //
    // We just need to set numVertices = 2 for every segment.
    //
    // TODO: Use an std::vector so we can its fast fill method.

    m_lineSet->numVertices.setNum( m_numSegs );
    int* p = m_lineSet->numVertices.startEditing();
    for (int i = 0; i < m_numSegs; ++i) {
      *p++ = 2;
    }
    m_lineSet->numVertices.finishEditing();
    return 0;
  }

  // --------------------------------------------------------------------------
  // Now for the tricky part...
  // Make polygons by connecting line segments.

  // Status of which segments have already been added to the boundarys.
  std::vector<bool> isSegUsed;
  isSegUsed.resize( m_numSegs, false ); // Fill with false values

  // Allow access to the segment list as a list of vertices.
  SbVec3f* vertList = (SbVec3f*)&m_segments[0];

  // We can do some optimizations (later) if we know the segments are sorted in X.
  // Note that std::sort will reallocate memory so we need to reset the vertexProperty node.
  if (m_sort) {
    std::sort( m_segments.begin(), m_segments.end() );
    vertList = (SbVec3f*)&m_segments[0];
    vprop->vertex.setValuesPointer( numVerts, vertList );
  }

  // Setup
  // If sorted, the first segment automatically has the smallest X value.
  // So we are looking for a connection to the 2nd vert of the 1st segment.
  // We assume that all segments can be used.
  int numSegUnused     = m_numSegs - 1;  // How many segments are NOT yet used.
  int prevNumSegUnused = numSegUnused;   // Used to check if we made progress on each pass
  int numIndicesAdded = 2;
  m_faces.push_back( 0 ); // First face obviously starts with the first coord index.

  // If not sorting, find line segment with smallest X value as starting point.
  int   imin = 0;
  if (! m_sort) {
    float xmin = m_segments[0].v0[0];
    for (int i = 1; i < m_numSegs; ++i) {
      float x = m_segments[i].v0[0];
      if (x < xmin) {
        xmin = x;
        imin = i;
      }
    }
  }
  m_indices.push_back( imin * 2 );          // Convert segment index to coordinate index.
  m_indices.push_back( imin * 2 + 1 );
  isSegUsed[imin] = true;
  SbVec3f firstVert = m_segments[imin].v0;  // If valid boundary, should connect back to this one.
  SbVec3f curVert   = m_segments[imin].v1;  // This is the one we currently want to connect to.
  int     firstSeg  = imin;
  int     firstUnusedSeg = 0;

#ifdef _DEBUG
  std::cout << "Poly: " << m_numSegs << " total segments " << numSegDup << " had duplicates.\n";
#endif

  static bool xtraDebug = false; // HACK (remove eventually)
  std::vector<int> segIndices;  // List of segments connected (in order)
  std::vector<bool> faceClosed; // Is boundary[i] closed (forms a proper polygon).
  if (xtraDebug) segIndices.push_back( imin );
#ifdef _DEBUG
    std::cout << "  Starting at seg " << imin << " " << curVert << std::endl;
#endif

  // Outer Loop -----------------------------------------------------
  // Continue in this loop as long as there are unused line segments (or other exit criteria).
  while (numSegUnused > 0) {
    // Inner Loop -----------------------------------------
    // Loop over the line segments.
    // Each time we're given a coordinate and must find the segment that is
    // the "best" match.  Match means that either one of the coordinates
    // in the candidate line segment is approximately equal to the given
    // coordinate.  Best match means the segment whose coordinate is closest
    // to the given coordinate.  So we have to loop over all the segments
    // every time.  Of course we don't consider segments that are already
    // used and we can start at the first unused segment in the list.

    //loopAgain:  // TODO: Find a cleaner solution :-)

    float bestDiff = m_epsilon; // Can we find a match better than this?
    int   bestSeg  = -1;
    int   bestVert = -1;

    for (int i = firstUnusedSeg; i < m_numSegs; ++i) {
      if (isSegUsed[i]) continue; // Ignore segments we've already used.
      // Consider each coordinate of the candidate segment because we
      // don't know what "direction" we're going around the boundary.
      SbVec3f& v0 = m_segments[i].v0;
      float diff = (curVert - v0).lengthSquared();
      if (diff <= bestDiff) { // Match!
        bestDiff = diff;
        bestSeg  = i;
        bestVert = 0; // first one
      }
      SbVec3f& v1 = m_segments[i].v1;
      diff = (curVert - v1).lengthSquared();
      if (diff <= bestDiff) { // Match!
        bestDiff = diff;
        bestSeg  = i;
        bestVert = 1;
      }
    } // End inner loop -----------------------------------

    // Are we really done?
    if (bestSeg >= 0) {
      // We found a matching segment, so extend the boundary.
      isSegUsed[bestSeg] = true;
      if (xtraDebug) segIndices.push_back( bestSeg );
      int pushIndex = (bestSeg * 2) + (1 - bestVert); // Push index of vertex we didn't match
      m_indices.push_back( pushIndex );
      numIndicesAdded++;
      curVert = vertList[ pushIndex ];
      numSegUnused--;
      if (numSegUnused > 0) { // If there are more segments, go around again.
        continue; //goto loopAgain;
      }
    }

    // Figure out why we stopped and what we've got at this point...
    // Note we have to check numIndicesAdded to avoid the case where the coordinates
    // in the first segment are approximately equal.
    if (curVert.equals(firstVert,m_epsilon) && numIndicesAdded > 2) {
      // Somehow we miraculously made it back to our starting vertex.
      faceClosed.push_back( true );

      if (numSegUnused == 0) {
        // Total success!  We looped back to the beginning and have no segments left over.
#ifdef _DEBUG
        std::cout << "    connected to first vertex. Done - All segs used!" << std::endl;
#endif
        break; // Leave the outer/while loop
      }
      else if (numSegUnused < 3) {
        // Effectively we're done...
        // It's not an ideal result because we have segments left over,
        // but we can't make a polygon with less than 3 segments...
#ifdef _DEBUG
        std::cout << "    connected to first vertex. Done - Less than 3 segs unused" << std::endl;
#endif
        // Cleanup
        if (numIndicesAdded < 3) {
          // BUT... It can't be a polygon if there are only 2 vertices...
          // Throw away those indices and the trailing -1 (added back later).
          m_indices.pop_back();
          m_indices.pop_back();
          if (m_indices.size() > 0 && m_indices.back() == -1) {
            m_indices.pop_back();
          }
        }
        break; // Leave the while loop
      }
      else if (numIndicesAdded < 3) {
        // Done with this piece of the boundary.
#ifdef _DEBUG
        std::cout << "    connected to first vertex. Reject face < 3 indices. numSegsUnused: " << numSegUnused << std::endl;
#endif
        // BUT... It can't be a polygon if there are only 2 vertices...
        // Throw away those vertices (indices actually).
        m_indices.pop_back();
        m_indices.pop_back();
        if (m_indices.size() > 0 && m_indices.back() != -1) {
          m_indices.push_back( -1 );
        }
      }
      else {
        // Another good case!
        // We looped back to the beginning and there are enough segments
        // left over that we probably can make another polyline.
#ifdef _DEBUG
        std::cout << "    connected back to first vertex. New face. numSegsUnused: " << numSegUnused << std::endl;
#endif
        m_indices.push_back( -1 );                  // Starting new boundary
        m_faces.push_back( (int)m_indices.size() ); // New indices start one *past* the -1 value
        numIndicesAdded = 0;
      }
    } // endif curVert == firstVert
    else {
      // We stopped for some reason other connecting back to the first vertex.
#ifdef _DEBUG
      std::cout << "    stopped without finding first vertex. numSegsUnused: " << numSegUnused << std::endl;
#endif
      if (numIndicesAdded < 3) {
        // The current boundary can't be a polygon if there are only 2 vertices...
        // Throw away those indices and the trailing -1 (added back later).
        m_indices.pop_back();
        m_indices.pop_back();
        if (m_indices.size() > 0 && m_indices.back() == -1) {
          m_indices.pop_back();
        }
      }
      else {
        // Else we have an unclosed boundary.  We'll deal with that later.
        // For now, finish it off and start on the next boundary.
        faceClosed.push_back( false );
        m_indices.push_back( -1 );                  // Starting new boundary
        m_faces.push_back( (int)m_indices.size() ); // New indices start one *past* the -1 value
        numIndicesAdded = 0;
      }
    }
    if (numSegUnused < 3)
      break;  // Something's wrong but we can't make a triangle with less then 3 segments...

    // Find new starting segment (smallest X) from all unused segments
    imin = 0;
    if (m_sort) {
      // Smallest X coord will be first unused segment
      for (int i = 0; i < m_numSegs; ++i) {
        if (! isSegUsed[i]) {
          imin = i;
          firstUnusedSeg = i + 1;
          break;
        }
      }
    }
    else {
      // Search for smallest X coord
      float xmin = FLT_MAX;
      for (int i = 0; i < m_numSegs; ++i) {
        if (! isSegUsed[i]) {
          float x = m_segments[i].v0[0];
          if (x < xmin) {
            xmin = x;
            imin = i;
          }
        }
      }
    }
    isSegUsed[imin] = true;
    numSegUnused--;
    firstSeg  = imin;
    firstVert = m_segments[imin].v0;
    curVert   = m_segments[imin].v1;
    m_indices.push_back( imin * 2 );
    m_indices.push_back( imin * 2 + 1 );
    numIndicesAdded = 2;
 
      // Check for endless loop
    if (numSegUnused > 0 && numSegUnused == prevNumSegUnused) {
      // We have unused segments but didn't find ANY connections on the last pass.
      // Bad!  This shouldn't happen but we have to check to avoid
      // any possibility of an endless loop where we couldn't find
      // any match for the current end vertex.
#ifdef _DEBUG
      std::cout << "  uh-oh. Last pass did not reduce numSegsUnused " << numSegUnused << std::endl;
#endif
      break;
    }
    prevNumSegUnused = numSegUnused;
#ifdef _DEBUG
    std::cout << "  Starting again at seg " << imin << " " << curVert << std::endl;
#endif
  } // End while loop -----------------------------------------------

  // Process the boundaries
  // Step 1: For closed boundaries (polygons) - check the orientation.
  //    Since we are given the triangles (more or less) randomly, we can't
  //    make any assumptions about the orientation of the boundaries.
  //    Our goal is for all the boundaries to be consistently oriented
  //    and with the normal vector pointing "out" from the geometry.
  //    We just need to check that the normal vector points in the opposite
  //    direction from the clip plane normal.
  //
  // Step 2: For unclosed boundaries - copy indices to indexedLineSet.
  //    If a boundary is not closed, it's not a polgon.

  int     numFaces   = (int)m_faces.size();
  int     numIndices = (int)m_indices.size();
  if (m_faces[numFaces-1] >= numIndices)
    numFaces--; // There was an invalid face at the end which we threw away.

  // Loop over closed boundaries
  for (int iface = 0; iface < numFaces; ++iface) {
    if (! faceClosed[iface])
      continue;

    // Index of first and last coordinate in boundary
    int index0 = m_faces[iface];
    int index1;
    if (iface == numFaces - 1) {
      index1 = numIndices - 1;
    }
    else {
      index1 = m_faces[iface+1] - 2; // Skip back over the -1 index at end of each face.
    }
    // We have a polygon. Compute normal and compare with desired normal.
    SbVec3f normal = computeNormal( index0, index1, normalVector );
    float d = normal.dot( normalVector ); // Dot product tells us if two vectors point in the same half-space.
    if (d < 0) {
      // Orientation of this boundary is reversed.
      // We can fix that by reversing the order of the indices.
      // The reverse() function expects ptr to start of values and ptr to *one past* end of values.
#ifdef _DEBUG
      std::cout << "  Reversing indices for boundary " << iface << std::endl;
#endif
      // Remember STL expects one past the actual last value.
      std::reverse( (int*)&m_indices[0] + index0, (int*)&m_indices[0] + index1 + 1 );
    }
  }

  // Loop over unclosed boundaries.
  // We loop from end to start so that removing indices does not invalidate
  // the remaining indices in m_faces.
  for (int iface = numFaces - 1; iface >= 0; --iface) {
    if (faceClosed[iface])
      continue;
    // Index of first and last coordinate in boundary
    int index0 = m_faces[iface];
    int index1;
    if (iface == numFaces - 1) {
      index1 = numIndices - 1;
    }
    else {
      index1 = m_faces[iface+1] - 1;
    }
    // Unclosed boundaries cannot be rendered as polygons.
    int position = m_lineSet2->coordIndex.getNum();
    // Copy the indices because we're going to change the m_indices vector.
    int num = index1 - index0 + 1;
    m_lineSet2->coordIndex.setValues( position, num, (int*)&m_indices[index0] );
    // Remove indices from the polygon vector.
    // Remember STL expects one past the actual last value.
    m_indices.erase( m_indices.begin() + index0, m_indices.begin() + index1 + 1 );
  }

  // Set the faceSet coordinate indices.
  // First check if there are any left (all boundaries might be unclosed).
  numIndices = (int)m_indices.size();
  if (numIndices > 0) {
    if (m_indices.back() != -1) {
      m_indices.push_back( -1 );
      numIndices++;
    }
    m_faceSet->coordIndex.setValuesPointer( numIndices, (int*)&m_indices[0] );
  }
  else {
    m_faceSet->coordIndex.setNum( 0 );
  }

  // Update the shape hints.
  if (numFaces > 1 && numIndices > 0) {
    m_shapeHints->windingType = SoShapeHints::ODD_TYPE;
  }
  else {
    m_shapeHints->windingType = SoShapeHints::NO_WINDING_TYPE;
  }

  // Make the face set be rendered
  m_switch->whichChild = 1;

  return 0;
}
