// Ruler utility functions
///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoTextProperty.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <Inventor/SoPickedPoint.h>
#include <Inventor/events/SoEvent.h>

#include <VolumeViz/details/SoOrthoSliceDetail.h>

#include <Medical/nodes/Ruler.h>

SO_NODE_SOURCE(Ruler)

///////////////////////////////////////////////////////////////////////////////
void
Ruler::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

   // Initialize type id variables
   SO__NODE_INIT_CLASS(Ruler, "Ruler", SoSeparator);
}

///////////////////////////////////////////////////////////////////////////////
void
Ruler::exitClass()
{
   SO__NODE_EXIT_CLASS(Ruler);
}

///////////////////////////////////////////////////////////////////////////////
Ruler::Ruler()
{
  // Init public fields
  SO_NODE_CONSTRUCTOR(Ruler);
  SO_NODE_ADD_FIELD(globalFactor, (1.));
  SO_NODE_ADD_FIELD(label       , ("mm"));

  m_initPos.setValue(0,0,0);
  m_text3D = false; // Use 2D text by default

  setName( "Ruler" );
  buildRuler();
}

///////////////////////////////////////////////////////////////////////////////
Ruler::~Ruler()
{
}

///////////////////////////////////////////////////////////////////////////////
SoFont*
Ruler::getFontNode() const
{
  return m_annotationFont;
}

///////////////////////////////////////////////////////////////////////////////
// Create the ruler scene graph
void
Ruler::buildRuler()
{
  // Ruler = 2 Cone + 1 lineset + 1 3D Text.

  // Draw Style
  m_dStyle = new SoDrawStyle();
    m_dStyle->style = SoDrawStyle::INVISIBLE;
    m_dStyle->lineWidth=2;
    addChild(m_dStyle);

  // Pick Style
  SoPickStyle* rulerPickStyle = new SoPickStyle();
    rulerPickStyle->style = SoPickStyle::UNPICKABLE ;
    addChild(rulerPickStyle);

  // Shape hints
  SoShapeHints* rulerShapeHints = new SoShapeHints();
    rulerShapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
    addChild( rulerShapeHints );

  // SoMaterial 
  SoMaterial *rulerAnnotationMat = new SoMaterial();
    rulerAnnotationMat->diffuseColor.setValue(SbColor(1,0,0));
    addChild(rulerAnnotationMat);

  //////////////////////////////////////////////////////////////////////////////
  // Left Cone
  SoSeparator *leftConeSep = new SoSeparator();
    addChild(leftConeSep);
  
    m_leftConeTransform = new SoTransform();
      leftConeSep->addChild( m_leftConeTransform );

    m_initialLeftTranslation = new SoTranslation();
      m_initialLeftTranslation->translation.setValue( SbVec3f(0, -globalFactor.getValue(), 0) );
      leftConeSep->addChild( m_initialLeftTranslation );

    m_leftCone = new SoCone();
      m_leftCone->bottomRadius.setValue( globalFactor.getValue() );
      m_leftCone->height.setValue( 2 * globalFactor.getValue() );
      leftConeSep->addChild( m_leftCone );

  //////////////////////////////////////////////////////////////////////////////
  // Right Cone
  SoSeparator *rightConeSep = new SoSeparator();
    addChild(rightConeSep);
  
    m_rightConeTransform = new SoTransform();
      rightConeSep->addChild( m_rightConeTransform );

    m_initialRightTranslation = new SoTranslation();
      m_initialRightTranslation->translation.setValue( SbVec3f(0, -globalFactor.getValue(), 0) );
      rightConeSep->addChild( m_initialRightTranslation );
    
    m_rightCone = new SoCone();
      m_rightCone->bottomRadius.setValue( globalFactor.getValue() );
      m_rightCone->height.setValue( 2 * globalFactor.getValue() );
      rightConeSep->addChild( m_rightCone );

  /////////////////////////////////////////////////////////////////////////////
  // Ruler line
  m_rulerVertexProp = new SoVertexProperty();
  m_ruler = new SoLineSet();
    m_ruler->vertexProperty = m_rulerVertexProp;
    addChild( m_ruler );

  /////////////////////////////////////////////////////////////////////////////
  // Ruler annotation
  SoAnnotation *rulerAnnotationSep = new SoAnnotation();
    addChild(rulerAnnotationSep);

    // Annotation position
    m_annotationTransform = new SoTransform();
      rulerAnnotationSep->addChild(m_annotationTransform);

    // Annotation appearance
    SoMaterial* annoMatl = new SoMaterial();
      annoMatl->diffuseColor.setValue( 0.1f, 0.1f, 0.1f );
      rulerAnnotationSep->addChild( annoMatl );

    // Annotation properties
    SoTextProperty* annoProp = new SoTextProperty();
      annoProp->alignmentV = SoTextProperty::BOTTOM;
      annoProp->style = SoTextProperty::BACK_FRAME;
      annoProp->styleColors.set1Value( (int)SoTextProperty::BACK_FRAME_COLOR, SbColorRGBA( 1, 1, 1, 0.5f ) );
      rulerAnnotationSep->addChild( annoProp );

    // Annotation font.  TODO: Should it be customisable
    m_annotationFont = new SoFont();
      if (m_text3D)
        m_annotationFont->size = 10 * globalFactor.getValue();
      else
        m_annotationFont->size = 17; // pixels
      rulerAnnotationSep->addChild( m_annotationFont );

    // Annotation initial offset
    SoTranslation *intialTranslation = new SoTranslation();
      intialTranslation->translation.setValue(SbVec3f(0, globalFactor.getValue(),0));
      rulerAnnotationSep->addChild(intialTranslation);

    // Text of the ruler
    if (m_text3D) {
      m_rulerAnnotation3D = new SoText3();
      rulerAnnotationSep->addChild(m_rulerAnnotation3D);
    }
    else {
      m_rulerAnnotation2D = new SoText2();
      rulerAnnotationSep->addChild(m_rulerAnnotation2D);
    }

    m_rulerVertexProp = new SoVertexProperty();
    m_ruler = new SoLineSet();
    m_ruler->vertexProperty = m_rulerVertexProp;
    addChild( m_ruler );
}

//////////////////////////////////////////////////////////////////////////////
// Manage field modification
void
Ruler::notify(SoNotList *list)
{
  if (list->getLastRec()->getType() == SoNotRec::CONTAINER)
  {
    if (list->getLastField() == &globalFactor)
    {
      updateGlobalFactor();
    }
  }
  // Forward other events to upper layer
  SoSeparator::notify(list);
}

//////////////////////////////////////////////////////////////////////////////
// Global Factor management
void
Ruler::updateGlobalFactor()
{
  // Update all node using globalFactor value
  m_initialLeftTranslation->translation.setValue( SbVec3f(0, -globalFactor.getValue(), 0) );
  m_initialRightTranslation->translation.setValue( SbVec3f(0, -globalFactor.getValue(), 0) );

  m_leftCone->bottomRadius.setValue( globalFactor.getValue() );
    m_leftCone->height.setValue( 2 * globalFactor.getValue() );

  m_rightCone->bottomRadius.setValue( globalFactor.getValue() );
    m_rightCone->height.setValue( 2 * globalFactor.getValue() );
}

/*********************************************************************************/
bool
Ruler::manageMouseMove( bool firstClick, const SoOrthoSliceDetail* detail )
{
  bool returnClickStatus = firstClick;

  SbVec3f newPos(0,0,0);
  if (detail != NULL)
    newPos = detail->getValueObjectPos();
  else
    newPos = m_initPos;

  if( firstClick ) 
  {
    returnClickStatus = false;
    m_initPos = newPos;
    m_rulerVertexProp->vertex.set1Value( 0, m_initPos );
    m_leftConeTransform->translation.setValue( m_initPos );
    m_dStyle->style = SoDrawStyle::INVISIBLE;

    m_ruler->numVertices = 1;
  } 
  else 
  {
    m_rulerVertexProp->vertex.set1Value( 1, newPos );
    m_rightConeTransform->translation.setValue( newPos );
    m_ruler->numVertices = 2;
    m_dStyle->style = SoDrawStyle::FILLED;
  }

  // Test to orient cones in function of the distance between 2 selected points. To avoid cone overllaping.
  SbVec3f delta = m_initPos - detail->getValueObjectPos();
  float dist    = delta.length();
  if ( dist < globalFactor.getValue() * 5 ) 
  {
    m_leftConeTransform->rotation.setValue ( SbRotation( SbVec3f(0, 1,0), -delta) );
    m_rightConeTransform->rotation.setValue( SbRotation( SbVec3f(0,-1,0), -delta) );
  } 
  else 
  {
    m_leftConeTransform->rotation.setValue ( SbRotation( SbVec3f(0, 1,0), delta) );
    m_rightConeTransform->rotation.setValue( SbRotation( SbVec3f(0,-1,0), delta) );
  }

  SbString labelString = label.getValue();
  SbString annotString;
  if (labelString.isEmpty())
    annotString.sprintf( "%.2f", dist );
  else
    annotString.sprintf( "%.2f %s", dist, labelString.toLatin1() );
  if (m_text3D) {
    m_rulerAnnotation3D->string.setValue( annotString );
    m_annotationTransform->rotation.setValue( SbRotation( SbVec3f(-1,0,0), delta) );
  }
  else
    m_rulerAnnotation2D->string.setValue( annotString );
  m_annotationTransform->translation.setValue(m_initPos - (delta/3));

  return returnClickStatus;
}
