// View utility class

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

// Notes:
//   - About whether the event is inside our viewport...
//     At first it seems possible to check events in this node's HandleEvent method
//     and only handle the event if it is inside our viewport.  It's NOT that easy!
//     It would work in some cases, but not in others.
//       - Case 1: Draggers
//         Would work because they call HandleEventAction::setGrabber() on the mouse
//         down event. Then they get the mouse move events directly, even if the cursor
//         moves outside our viewport.
//       - Case 2: SceneExaminer
//         SceneExaminer does _not_ call setGrabber().  So the default behavior is that
//         you can only move the camera as long as the cursor stays inside our viewport.
//         That is very inconvenient for the user.  In theory we could fix that by
//         calling setGrabber() ourselves, but then we must avoid conflict with draggers
//         in the scene that need to call setGrabber(). So we would have to track the
//         interaction mode (viewing or selection) in the SceneExaminer, but we can't
//         assume there is a SceneExaminer in our sub-graph.
//     I conclude that it's better to manage event routing in a separate node, that is
//     above all the SceneView node in the scene graph.  See ViewManager.
//
//   - Internal scene graph
//     Separator : this
//       |- Callback : Always traversed to update parent viewport size
//       |- Switch : Master switch controlled by 'active' field
//          |- Viewport  : Sets the viewport in pixels
//          |- Separator : Contains the application's scene
//          |- Switch    : Border switch controlled by 'drawBorder' field

// TODO:
//   - Allow setting viewport in pixels.
//   - Remove test in callback() method when issue #55555 is fixed.
//   - ? Allow setting border color?
//   - ? Allow drawing a background?
//   - Snapshot: Could be useful to provide a getSnapShot() method.
//     Legacy viewers have saveSnapshot() but there are issues including:
//       - No way to get image in application.
//       - No way to get image with opaque background.

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoViewport.h>

#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/actions/SoRayPickAction.h>

#include <Inventor/elements/SoViewportRegionElement.h>
#include <Inventor/events/SoEvent.h>

#include <Inventor/sensors/SoNodeSensor.h>

#include <Inventor/STL/algorithm> // for std::min/max

#include <Medical/nodes/SceneView.h>

SO_NODE_SOURCE(SceneView);

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
SceneView::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

  // Initialize type id variables
  SO_NODE_INIT_CLASS(SceneView, SoSeparator, "Separator");
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
SceneView::exitClass()
{
  SO__NODE_EXIT_CLASS(SceneView);
}

///////////////////////////////////////////////////////////////////////////////
SceneView::SceneView()
{
  // Setup fields
  SO_NODE_CONSTRUCTOR(SceneView);
  SO_NODE_ADD_FIELD(viewportOrigin, (0,0));
  SO_NODE_ADD_FIELD(viewportSize  , (1,1));
  SO_NODE_ADD_FIELD(active        , (TRUE));
  SO_NODE_ADD_FIELD(drawBorder    , (TRUE));

  // Hide inherited fields from IvTune.
  // It's not necessary to do this, but the SoSeparator fields are not relevant,
  // so this makes it a little bit easier to observe and modify in IvTune.
#if SO_INVENTOR_VERSION >= 9100
  boundingBoxCaching.setFieldType( SoField::PRIVATE_FIELD );
  boundingBoxIgnoring.setFieldType( SoField::PRIVATE_FIELD );
  renderCulling.setFieldType( SoField::PRIVATE_FIELD );
  pickCulling.setFieldType( SoField::PRIVATE_FIELD );
  fastEditing.setFieldType( SoField::PRIVATE_FIELD );
  renderUnitId.setFieldType( SoField::PRIVATE_FIELD );
#if SO_INVENTOR_VERSION < 10000
  directVizCaching.setFieldType( SoField::PRIVATE_FIELD );
#endif
#endif

  // Internal state
  m_curViewport.setValue(0,0,0,0);
  m_viewport.setValue(0,0,0,0);
  m_vportRegion.setViewportPixels(0,0,0,0);
  m_viewportPixels.setValue(0,0,0,0);
  m_viewportOrig.setValue(0,0);
  m_viewportSize.setValue(0,0);
  m_parentOrig.setValue(0,0);
  m_parentSize.setValue(0,0);
  m_handleEvents = true;

  // --------------------------------------------------------------------------
  // Build internal scene graph
  //   Separator : this
  //     |- Callback : Always traversed to update parent viewport size
  //     |- Switch : Master switch controlled by 'active' field
  //          |- Viewport  : Sets the viewport in pixels
  //          |- Separator : Contains the application's scene
  //          |- Switch    : Border switch controlled by 'drawBorder' field
  //
  // Remember not to use our public addChild() method (it's prohibited).

  SoCallback* callbackNode = new SoCallback();
    callbackNode->setCallback( callback, (void*)this );
    SoSeparator::addChild( callbackNode );

  m_masterSwitch = new SoSwitch();
    m_masterSwitch->setName( "ViewActiveSwitch" );
    m_masterSwitch->whichChild = active.getValue() ? -3 : -1;
    SoSeparator::addChild( m_masterSwitch.ptr() );

  m_vportNode = new SoViewport();
    m_masterSwitch->addChild( m_vportNode.ptr() );

  m_sceneNode = new SoSeparator();
    m_masterSwitch->addChild( m_sceneNode.ptr() );

  m_borderSwitch = new SoSwitch();
    m_borderSwitch->setName( "ViewBorderSwitch" );
    m_borderSwitch->whichChild = drawBorder.getValue() ? -3 : -1;
    m_masterSwitch->addChild( m_borderSwitch.ptr() );

  // --------------------------------------------------------------------------
  // Build border sub-graph

  SoAnnotation* borderSep = new SoAnnotation();
    m_borderSwitch->addChild( borderSep );

  SoBBox* bbox = new SoBBox();
    bbox->mode = SoBBox::NO_BOUNDING_BOX;
    borderSep->addChild( bbox );

  SoLightModel* lmodel = new SoLightModel();
    lmodel->model = SoLightModel::BASE_COLOR;
    borderSep->addChild( lmodel );

  SoPickStyle* pstyle = new SoPickStyle();
    pstyle->style = SoPickStyle::UNPICKABLE;
    borderSep->addChild( pstyle );

  SoOrthographicCamera* camera = new SoOrthographicCamera();
    camera->viewportMapping = SoCamera::LEAVE_ALONE;
    borderSep->addChild( camera );

  SoVertexProperty* vprop = new SoVertexProperty();
    vprop->vertex.setNum( 5 );
    vprop->vertex.set1Value( 0, -0.998f, -0.998f, 0 );
    vprop->vertex.set1Value( 1,  0.998f, -0.998f, 0 );
    vprop->vertex.set1Value( 2,  0.998f,  0.998f, 0 );
    vprop->vertex.set1Value( 3, -0.998f,  0.998f, 0 );
    vprop->vertex.set1Value( 4, -0.998f, -0.998f, 0 );

  m_borderLine = new SoLineSet();  // Must be member variable so we can update it later.
    m_borderLine->vertexProperty = vprop;
    borderSep->addChild( m_borderLine.ptr() );

  // Detect changes to our fields.
  // Node sensor is a little more complicated, but less overhead than N field sensors.
  m_sensor = new SoNodeSensor( sensorCB, (void*)this );
  m_ignoreSensor = true;
  m_sensor->attach( this );
  m_sensor->setPriority( 0 );
  m_ignoreSensor = false;
}

///////////////////////////////////////////////////////////////////////////////
// Destructor
SceneView::~SceneView()
{
  delete m_sensor;
}

///////////////////////////////////////////////////////////////////////////////
// Enable handling events.
// Default is true. This method is used, for example, by the
// ViewManager node to temporarily disable event handling without
// modifying a field and triggering notification. 
void
SceneView::enableEventHandling( SbBool onoff )
{
  m_handleEvents = onoff ? true : false;
}

///////////////////////////////////////////////////////////////////////////////
// Called when our Callback node is traversed.
// 
// Primary task is to update the local viewport if using normalized coordinates
// and the parent viewport has changed.  This should happen at least once, on
// the first traversal in which the viewport element is correctly set.
void
SceneView::callback(void *userData, SoAction *action) 
{
  // Get current viewport from state.
  // Also creates a dependency so we should be traversed if it ever changes.
  SoState* state = action->getState();
  const SbViewportRegion& parentVP = SoViewportRegionElement::get( state );

  SceneView* self = (SceneView*)userData;

  // If this is a handleEvent action...
  if (action->isOfType(SoHandleEventAction::getClassTypeId())) {
    // If event handling is not allowed...
    if (self->m_handleEvents == false) {
      // Stop traversing in this branch of the scene graph.
      action->stopActionInBranch();
    }
  }

  self->viewportParentChanged( parentVP );
}

///////////////////////////////////////////////////////////////////////////////
// Called when any of our fields or children are modified.
void
SceneView::sensorCB( void* data, SoSensor* sensor )
{
  // Return if we're ignoring the sensor.
  SceneView* self = (SceneView*)data;
  if (self->m_ignoreSensor)
    return;

  // Return if we don't know which field triggered the sensor.
  // That will happen, for example, when the change that triggered the sensor
  // was a change below us in the scene graph.  We don't care about that.
  const SoNodeSensor* nodeSensor = (SoNodeSensor*)sensor;
  const SoField* field = nodeSensor->getTriggerField();
  if (field == NULL)
    return;

  // Viewport field modified -------------------------------------------------
  if (field == &self->viewportOrigin || field == &self->viewportSize) {
    self->viewportFieldChanged();
  }

  // 'active' field modified --------------------------------------------------
  else if (field == &self->active) {
    int newWhich = self->active.getValue() ? -3 : -1;
    if (self->m_masterSwitch->whichChild.getValue() != newWhich)
      self->m_masterSwitch->whichChild = newWhich;
  }

  // 'drawBorder' field modified ----------------------------------------------------
  else if (field == &self->drawBorder) {
    int newWhich = self->drawBorder.getValue() ? -3 : -1;
    if (self->m_borderSwitch->whichChild.getValue() != newWhich)
      self->m_borderSwitch->whichChild = newWhich;
  }
}

// Make sure application's viewport values are valid
#define CLAMP( value ) (std::min( std::max( value, 0.0f ), 1.0f ))

///////////////////////////////////////////////////////////////////////////////
// Called when the viewport origin or size field has changed.
//
// We need to recompute our viewport in pixels.
void
SceneView::viewportFieldChanged()
{
  // Keep specified values in valid range.
  const SbVec2f& vorig = viewportOrigin.getValue();
  const SbVec2f& vsize = viewportSize.getValue();
  SbVec4f vport( CLAMP(vorig[0]), CLAMP(vorig[1]), CLAMP(vsize[0]), CLAMP(vsize[1]) );

  // Return if no actual change
  if (vport == m_curViewport)
    return;

  // Special case: If application did not specify a size...
  if (vport[2] == 0 && vport[3] == 0) {
    // Use full size of parent viewport / render area
    vport.setValue(0,0,1,1);
  }

  // Remember new value
  m_curViewport = vport;

  // Return if we don't know our parent size yet
  if (m_parentSize[0] == 0 || m_parentSize[1] == 0)
    return;

  // Update viewport
  updateOurViewport();
}

///////////////////////////////////////////////////////////////////////////////
// Called when our parent viewport / render area size has changed.
//
// We need to recompute our viewport in pixels.
void
SceneView::viewportParentChanged( const SbViewportRegion& parentVP )
{
  const SbVec2i32& parentOrig = parentVP.getViewportOriginPixels_i32();
  const SbVec2i32& parentSize = parentVP.getViewportSizePixels_i32();

  // Return if we don't know our parent viewport / render area size yet.
  // This can happen if there is a traversal before the render area is displayed.
  if (parentSize[0] == 0 && parentSize[1] == 0)
    return;

  // Return if the parent viewport size hasn't really changed.
  if (parentOrig == m_parentOrig && parentSize == m_parentSize   )
      return;

  // Remember the new parent size
  m_parentOrig = parentOrig;
  m_parentSize = parentSize;

  // Update our viewport
  updateOurViewport();
  m_vportRegion.setWindowSize(parentVP.getWindowSize_i32());
}

///////////////////////////////////////////////////////////////////////////////
// Recompute and update our viewport.
// Called from viewportFieldChanged() and viewportParentChanged().
void
SceneView::updateOurViewport()
{
  // Recompute viewport in pixels from viewport in normalized coordinates
  // We can set float values in SoViewport, but ultimately they are integer
  // pixel coordinates when passed to OpenGL, so it's better to control the
  // "rounding off" ourselves and have more predictable results.
  m_viewport.setValue( m_curViewport[0] * (float)m_parentSize[0], m_curViewport[1] * (float)m_parentSize[1],
                       m_curViewport[2] * (float)m_parentSize[0], m_curViewport[3] * (float)m_parentSize[1] );
  SbVec2i32 viewportOrig( (int)(m_viewport[0]+0.5f), (int)(m_viewport[1]+0.5f) );
  SbVec2i32 viewportSize( (int)(m_viewport[2]+0.5f), (int)(m_viewport[3]+0.5f) );
  m_viewport[0] = (float)viewportOrig[0];
  m_viewport[1] = (float)viewportOrig[1];
  m_viewport[2] = (float)viewportSize[0];
  m_viewport[3] = (float)viewportSize[1];

  // The origin of our viewport (in pixels) is **relative** to our parent viewport.
  // This allows, for example, nested multi-view layouts.
  viewportOrig += m_parentOrig;

  // Update the viewport node if necessary.
  bool vportChanged = false;
  if (viewportOrig != m_viewportOrig) {
    m_vportNode->origin.setValue( (float)viewportOrig[0], (float)viewportOrig[1] );
    vportChanged = true;
  }
  if (viewportSize != m_viewportSize) {
    m_vportNode->size.setValue( (float)viewportSize[0], (float)viewportSize[1] );
    vportChanged = true;
  }

  // Update border geometry if necessary.
  if (vportChanged) {
    // Compute size of half a pixel in normalized coordinates.
    // In OpenGL, the center of the first pixel inside the viewport is at 0.5
    // in pixel coordinates, not 0. So using -1 and 1 in normalized coordinates
    // for the border is unsafe (could result in the border line being clipped).
    float halfPixelX = (float)(0.5 * (2 / m_viewport[2]));
    float halfPixelY = (float)(0.5 * (2 / m_viewport[3]));
    float l = -1 + halfPixelX;
    float r =  1 - halfPixelX;
    float b = -1 + halfPixelY;
    float t =  1 - halfPixelY;

    SoVertexProperty* vprop = (SoVertexProperty*)m_borderLine->vertexProperty.getValue();
    vprop->enableNotify( FALSE );
    vprop->vertex.set1Value( 0, l, b, 0 );
    vprop->vertex.set1Value( 1, r, b, 0 );
    vprop->vertex.set1Value( 2, r, t, 0 );
    vprop->vertex.set1Value( 3, l, t, 0 );
    vprop->vertex.set1Value( 4, l, b, 0 );
    vprop->enableNotify( TRUE );
  }

  // Remember what we did
  m_viewportOrig = viewportOrig;
  m_viewportSize = viewportSize;
  m_viewportPixels.setValue( m_viewportOrig[0], m_viewportOrig[1], m_viewportSize[0], m_viewportSize[1] );
  m_vportRegion.setViewportPixels( m_viewportOrig, m_viewportSize );
}

///////////////////////////////////////////////////////////////////////////////
// Convenience method to set viewport in normalized coordinates with one call.
// Modifies the 'viewportOrig' and 'viewportSize' fields.
void SceneView::setViewport( float origX, float origY, float sizeX, float sizeY )
{
  // Temporarily disable sensor then call touch so it only triggers once.
  m_ignoreSensor = true;
  viewportOrigin.setValue( origX, origY );
  viewportSize.setValue( sizeX, sizeY );
  m_ignoreSensor = false;
  viewportOrigin.touch();
}

///////////////////////////////////////////////////////////////////////////////
// Convenience method to set viewport in normalized coordinates with one call.
// Modifies the 'viewportOrig' and 'viewportSize' fields.
void SceneView::setViewport( const SbVec4f& viewport )
{
  setViewport( viewport[0], viewport[1], viewport[2], viewport[3] );
}

///////////////////////////////////////////////////////////////////////////////
// Returns the current viewport as a convenient single object. */
const SbVec4f&
SceneView::getViewport() const
{
  return m_curViewport;
}

///////////////////////////////////////////////////////////////////////////////
// Return our actual pixel viewport region for use with viewAll() etc.
const SbViewportRegion& SceneView::getViewportRegion() const
{
  return m_vportRegion;
}

///////////////////////////////////////////////////////////////////////////////
// Return our actual pixel viewport.
const SbVec4f& SceneView::getViewportPixels() const
{
  return m_viewport;
}

///////////////////////////////////////////////////////////////////////////////
// Override generic child methods to force use of our internal scene graph.

void SceneView::addChild( SoNode* child )
{
  m_sceneNode->addChild( child );
}
int
SceneView::findChild( const SoNode* child ) const
{
  return m_sceneNode->findChild( child );
}
SoNode*
SceneView::getChild( int index ) const
{
  return m_sceneNode->getChild( index );
}
void SceneView::insertChild( SoNode *child, int newChildIndex )
{
  m_sceneNode->insertChild( child, newChildIndex );
}
void SceneView::replaceChild( SoNode* oldChild, SoNode* newChild )
{
  m_sceneNode->replaceChild( oldChild, newChild );
}
void SceneView::removeChild( SoNode* child )
{
  m_sceneNode->removeChild( child );
}
void SceneView::removeChild( int index )
{
  m_sceneNode->removeChild( index );
}
void SceneView::removeAllChildren()
{
  m_sceneNode->removeAllChildren();
}
void SceneView::replaceChild( int index, SoNode* newChild )
{
  m_sceneNode->replaceChild( index, newChild );
}
int SceneView::getNumChildren() const
{
  return m_sceneNode->getNumChildren();
}
