// SliceOrientationMarkers utility class
//
// Slice orientation markers display
//   - Axial   : R - L and A - P (top to bottom)
//   - Coronal : R - L and S - I
//   - Sagittal: A - P and S - I

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTextProperty.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>

#include <Inventor/sensors/SoFieldSensor.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

#include <Medical/nodes/SliceOrientationMarkers.h>

SO_NODE_SOURCE(SliceOrientationMarkers);

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
SliceOrientationMarkers::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

  // Initialize type id variables
  SO_NODE_INIT_CLASS(SliceOrientationMarkers, SoAnnotation, "Annotation");
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
SliceOrientationMarkers::exitClass()
{
  SO__NODE_EXIT_CLASS(SliceOrientationMarkers);
}

///////////////////////////////////////////////////////////////////////////////
SliceOrientationMarkers::SliceOrientationMarkers()
{
  // Setup fields
  SO_NODE_CONSTRUCTOR(SliceOrientationMarkers);
  SO_NODE_ADD_FIELD(axis     , (MedicalHelper::AXIAL));
  SO_NODE_ADD_FIELD(offset   , (0.05f));
  SO_NODE_ADD_FIELD(fontName , ("Arial:Bold"));
  SO_NODE_ADD_FIELD(fontSize , (17));

  // Set up static info for enumerated type fields
  SO_NODE_DEFINE_ENUM_VALUE( MedicalHelper::Axis, MedicalHelper::AXIAL    );
  SO_NODE_DEFINE_ENUM_VALUE( MedicalHelper::Axis, MedicalHelper::CORONAL  );
  SO_NODE_DEFINE_ENUM_VALUE( MedicalHelper::Axis, MedicalHelper::SAGITTAL );

  // Set up info for enumerated type field
  SO_NODE_SET_SF_ENUM_TYPE( axis, MedicalHelper::Axis );

  // Hide inherited fields from IvTune.
  // It's not necessary to do this, but the SoSeparator fields are not
  // relevant to our "shape" node, so this makes it a little bit easier
  // to observe and modify in IvTune.
#if SO_INVENTOR_VERSION >= 9100
  boundingBoxCaching.setFieldType( SoField::PRIVATE_FIELD );
  renderCulling.setFieldType( SoField::PRIVATE_FIELD );
  pickCulling.setFieldType( SoField::PRIVATE_FIELD );
  fastEditing.setFieldType( SoField::PRIVATE_FIELD );
  renderUnitId.setFieldType( SoField::PRIVATE_FIELD );
#endif

  // Build internal scene graph
  //
  // Separator "AxisBoxRoot"
  // +- BBox
  // +- LightModel
  // +- PickStyle
  // +- OrthographicCamera
  // +- Font
  // +- Separator "Left"
  // |  +- TextProperty
  // |  +- Translation
  // |  +- Text2
  // +- Separator "Right"
  // |  +- TextProperty
  // |  +- Translation
  // |  +- Text2
  // +- Separator "Top"
  // |  +- TextProperty
  // |  +- Translation
  // |  +- Text2
  // +- Separator "Bottom"
  // |  +- TextProperty
  // |  +- Translation
  // |  +- Text2

  SoBBox* bbox = new SoBBox();
    bbox->mode = SoBBox::NO_BOUNDING_BOX;
    this->addChild( bbox );

  SoLightModel* lmodel = new SoLightModel();
    lmodel->model = SoLightModel::BASE_COLOR;
    this->addChild( lmodel );

  SoPickStyle* pstyle = new SoPickStyle();
    pstyle->style = SoPickStyle::UNPICKABLE;
    this->addChild( pstyle );

  SoOrthographicCamera* camera = new SoOrthographicCamera();
    camera->viewportMapping = SoCamera::LEAVE_ALONE;
    this->addChild( camera );

  m_fontNode = new SoFont();
    m_fontNode->size.connectFrom( &this->fontSize );
    m_fontNode->name.connectFrom( &this->fontName );
    m_fontNode->renderStyle = SoFont::TEXTURE;
    this->addChild( m_fontNode.ptr() );

  // Create text nodes.
  // Need separator for each to keep translations separate.
  // Each has its own position and alignment settings.
  for (int i = 0; i < 4; ++i) {
    SoSeparator* tempSep = new SoSeparator();
      this->addChild( tempSep );

    SoTextProperty* textProp = new SoTextProperty();
      // Allow app to control the fields that we don't care about.
      textProp->backFrameLineWidth.setIgnored( TRUE );
      textProp->characterSpacing.setIgnored( TRUE );
      textProp->kerning.setIgnored( TRUE );
      textProp->margin.setIgnored( TRUE );
      textProp->style.setIgnored( TRUE );
      textProp->styleColors.setIgnored( TRUE );
      tempSep->addChild( textProp );

      switch (i) {
      case 0: // Left
        textProp->alignmentH = SoTextProperty::LEFT;
        textProp->alignmentV = SoTextProperty::HALF;
        break;
      case 1: // Right
        textProp->alignmentH = SoTextProperty::RIGHT;
        textProp->alignmentV = SoTextProperty::HALF;
        break;
      case 2: // Top
        textProp->alignmentH = SoTextProperty::CENTER;
        textProp->alignmentV = SoTextProperty::TOP;
        break;
      case 3: // Bottom
        textProp->alignmentH = SoTextProperty::CENTER;
        textProp->alignmentV = SoTextProperty::BASE;
        break;
      }

    m_tranNodes[i] = new SoTranslation();
      tempSep->addChild( m_tranNodes[i] );

    m_textNodes[i] = new SoText2();
      m_textNodes[i]->justification = SoText2::INHERITED;
      tempSep->addChild( m_textNodes[i] );
  }

  // Initialize letters
  updateLetters();

  // Detect changes to our fields or children
  m_axisSensor = new SoFieldSensor( fieldSensorCB, (void*)this );
  m_axisSensor->setPriority( 0 );
  m_axisSensor->attach( &axis );

  m_offsetSensor = new SoFieldSensor( fieldSensorCB, ( void* )this );
  m_offsetSensor->setPriority( 0 );
  m_offsetSensor->attach( &offset );
}

///////////////////////////////////////////////////////////////////////////////
SliceOrientationMarkers::~SliceOrientationMarkers()
{
  delete m_axisSensor;
  delete m_offsetSensor;
  if ( m_fontNode.ptr() != NULL )
  {
    m_fontNode->size.disconnect( &this->fontSize );
    m_fontNode->name.disconnect( &this->fontName );
  }
}

///////////////////////////////////////////////////////////////////////////////
// Update letters when axis or offset changes.
void
SliceOrientationMarkers::updateLetters()
{
  // Text positions
  float offsetVal = offset.getValue();
  m_tranNodes[0]->translation.setValue( offsetVal - 1, 0, 0 ); // Left
  m_tranNodes[1]->translation.setValue( 1 - offsetVal, 0, 0 ); // Right
  m_tranNodes[2]->translation.setValue( 0, 1 - offsetVal, 0 ); // Top
  m_tranNodes[3]->translation.setValue( 0, offsetVal - 1, 0 ); // Bottom

  int which = axis.getValue();
  switch (which) {
    case MedicalHelper::AXIAL:
      m_textNodes[0]->string.set1Value( 0,  "R" ); // Left
      m_textNodes[1]->string.set1Value( 0,  "L" ); // Right
      m_textNodes[2]->string.set1Value( 0,  "A" ); // Top
      m_textNodes[3]->string.set1Value( 0,  "P" ); // Bottom
      break;
    case MedicalHelper::CORONAL:
      m_textNodes[0]->string.set1Value( 0,  "R" );
      m_textNodes[1]->string.set1Value( 0,  "L" );
      m_textNodes[2]->string.set1Value( 0,  "S" );
      m_textNodes[3]->string.set1Value( 0,  "I" );
      break;
    case MedicalHelper::SAGITTAL:
      m_textNodes[0]->string.set1Value( 0,  "A" );
      m_textNodes[1]->string.set1Value( 0,  "P" );
      m_textNodes[2]->string.set1Value( 0,  "S" );
      m_textNodes[3]->string.set1Value( 0,  "I" );
      break;
  }
}

///////////////////////////////////////////////////////////////////////////////
// Called when axis or offset fields are modified.
//
void
SliceOrientationMarkers::fieldSensorCB( void* data, SoSensor* /*sensor*/ )
{
  SliceOrientationMarkers* self = (SliceOrientationMarkers*)data;
  self->updateLetters();
}
