// TextBox utility class
//

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTextProperty.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/actions/SoSearchAction.h>

#include <Inventor/elements/SoViewportRegionElement.h>

#include <Inventor/sensors/SoNodeSensor.h>

#include <Inventor/STL/algorithm> // For std::min

#include <Medical/nodes/TextBox.h>

SO_NODE_SOURCE(TextBox);

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
TextBox::initClass()
{
  getClassRenderEngineMode().setRenderMode(SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING);

  // Initialize type id variables
  SO__NODE_INIT_CLASS(TextBox, "TextBox", SoAnnotation);
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
TextBox::exitClass()
{
  SO__NODE_EXIT_CLASS(TextBox);
}

///////////////////////////////////////////////////////////////////////////////
TextBox::TextBox()
{
  // Setup fields
  SO_NODE_CONSTRUCTOR(TextBox);
  SO_NODE_ADD_FIELD(position   , (0,0,0));
  SO_NODE_ADD_FIELD(alignmentH , (TextBox::LEFT));
  SO_NODE_ADD_FIELD(alignmentV , (TextBox::TOP));
  SO_NODE_ADD_FIELD(textAlignH , (TextBox::LEFT));
  SO_NODE_ADD_FIELD(fontName   , ("Arial:Bold"));
  SO_NODE_ADD_FIELD(fontSize   , (15));
  SO_NODE_ADD_FIELD(border     , (FALSE));
  SO_NODE_ADD_FIELD(borderColor, (SbColor(1,1,1)));

  // Set up static info for enumerated type fields
  SO_NODE_DEFINE_ENUM_VALUE( AlignmentH, LEFT      );
  SO_NODE_DEFINE_ENUM_VALUE( AlignmentH, RIGHT     );
  SO_NODE_DEFINE_ENUM_VALUE( AlignmentH, CENTER    );
  SO_NODE_DEFINE_ENUM_VALUE( AlignmentV, TOP       );
  SO_NODE_DEFINE_ENUM_VALUE( AlignmentV, MIDDLE    );
  SO_NODE_DEFINE_ENUM_VALUE( AlignmentV, BOTTOM    );

  // Set up info for enumerated type field
  SO_NODE_SET_SF_ENUM_TYPE( alignmentH, AlignmentH );
  SO_NODE_SET_SF_ENUM_TYPE( alignmentV, AlignmentV );
  SO_NODE_SET_SF_ENUM_TYPE( textAlignH, AlignmentH );

  // Hide inherited fields from IvTune.
  // It's not necessary to do this, but the SoSeparator fields are not
  // relevant to our "shape" node, so this makes it a little bit easier
  // to observe and modify in IvTune.
#if SO_INVENTOR_VERSION >= 9100
  boundingBoxCaching.setFieldType( SoField::PRIVATE_FIELD );
  renderCulling.setFieldType( SoField::PRIVATE_FIELD );
  pickCulling.setFieldType( SoField::PRIVATE_FIELD );
  fastEditing.setFieldType( SoField::PRIVATE_FIELD );
  renderUnitId.setFieldType( SoField::PRIVATE_FIELD );
#endif
  
  m_isModified = true;              // Update on next traversal
  m_curViewport.setWindowSize(0,0); // Initial viewport is unknown (get this during traversal)
  m_curBBox.makeEmpty();            // ditto

  // Build internal scene graph

  // Callback allows us to update our state at traversal time
  SoCallback* callbackNode = new SoCallback();
    callbackNode->setCallback( callbackNodeCB, (void*)this );
    this->addChild( callbackNode );

  // Set to NO_BOUNDING_BOX to prevent text box geometry from affecting
  // the bounding box of the application's scene.
  SoRef<SoBBox> bboxNode = new SoBBox();
    bboxNode->mode = SoBBox::NO_BOUNDING_BOX;
    this->addChild( bboxNode.ptr() );

  // Lighting off for visibility.
  SoLightModel* lmodel = new SoLightModel();
    lmodel->model = SoLightModel::BASE_COLOR;
    this->addChild( lmodel );

  // Avoid interfering with picking application geometry.
  SoPickStyle* pstyle = new SoPickStyle();
    pstyle->style = SoPickStyle::UNPICKABLE;
    this->addChild( pstyle );

  // Use normalized device coordinates.
  SoOrthographicCamera* camera = new SoOrthographicCamera();
    camera->viewportMapping = SoCamera::LEAVE_ALONE;
    this->addChild( camera );

  // Need to keep text translation from affecting border geometry.
  m_textSep = new SoSeparator();
    this->addChild( m_textSep.ptr() );

  // Note we can't simply connect from the position field.
  // Justification for multi-string text nodes doesn't work right (#43882),
  // so we have to adjust the text position to "fake it".
  // However translation and position are the same for default LEFT/TOP justification.
  m_tranNode = new SoTranslation();
    m_tranNode->translation = position.getValue();
    m_textSep->addChild( m_tranNode.ptr() );

  // Since OIV doesn't handle alignment of text blocks (multi-string nodes) the
  // way we want, force a predictable mode, so we can work around it.
  SoTextProperty* prop = new SoTextProperty();
    prop->alignmentH.connectFrom( &this->textAlignH );
    prop->alignmentV = SoTextProperty::TOP;
    m_textSep->addChild( prop );

  // Switch to control whether font properties are inherited or not.
  // TODO: How to expose this?
  m_fontSwitch = new SoSwitch();
    m_fontSwitch->whichChild = 0;
    m_textSep->addChild( m_fontSwitch.ptr() );

  m_fontNode = new SoFont();
    m_fontNode->size.connectFrom( &this->fontSize );
    m_fontNode->name.connectFrom( &this->fontName );
    m_fontNode->renderStyle = SoFont::TEXTURE;
    m_fontSwitch->addChild( m_fontNode.ptr() );

  m_textNode = new SoText2();
    m_textNode->string.setNum( 0 ); // Ensures that appendText works first time!
    m_textNode->spacing = 1.1f;
    m_textNode->justification = SoText2::INHERITED;
    m_textSep->addChild( m_textNode.ptr() );

  SoSeparator* geomSep = new SoSeparator();
    this->addChild( geomSep );

  // Switch to control visibility of the border.
  m_borderSwitch = new SoSwitch();
    m_borderSwitch->whichChild = border.getValue() ? 0 : -1;
    geomSep->addChild( m_borderSwitch.ptr() );

  // Border coordinates (to be filled in later!)
  SoVertexProperty* vprop = new SoVertexProperty();
    uint32_t packedColor = borderColor.getValue().getPackedValue();
    vprop->orderedRGBA.set1Value( 0, packedColor );

  // Border geometry
  m_borderGeom = new SoLineSet();
    m_borderGeom->vertexProperty = vprop;
    m_borderSwitch->addChild( m_borderGeom.ptr() );

  // Detect changes to our fields or children
  m_nodeSensor = new SoNodeSensor( nodeSensorCB, (void*)this );
    m_nodeSensor->setPriority( 0 );
    m_nodeSensor->attach( this );
}

///////////////////////////////////////////////////////////////////////////////
TextBox::~TextBox()
{
  removeAllChildren();
  delete m_nodeSensor;
}

///////////////////////////////////////////////////////////////////////////////
static 
void updateBorderGeom( SoVertexProperty* vprop, const SbBox3f& bbox )
{
  SbVec3f p0 = bbox.getMin();
  SbVec3f p1   = bbox.getMax();
  SbVec3f size = bbox.getSize();
  // Leave some slack
  p0[0] -= 0.02f * size[0];
  p1[0] += 0.02f * size[0];
  p1[1] += 0.02f * size[1];

  static const int numVertices = 5;
  const SbVec3f vertices[numVertices] = {
    p0,
    SbVec3f(p1[0], p0[1], p0[2]),
    p1,
    SbVec3f(p0[0], p1[1], p0[2]),
    p0
  };

  if (vprop->vertex.getNum() != numVertices || memcmp(vertices, vprop->vertex.getValues(0), numVertices * sizeof(SbVec3f)) != 0)
    vprop->vertex.setValues(0, numVertices, vertices);
}

///////////////////////////////////////////////////////////////////////////////
void
TextBox::updatePosAndGeom( SoState* state )
{
  // We may be called during traversal with a valid SoState.
  // That normally indicates the viewport has changed.
  // We may be called from a node/field sensor with no SoState.
  // That normally indicates a string has been added or deleted.
  // In that case we just use the last viewport we saved.
  //
  // Note we have to adjust the actual position of the text box to simulate
  // some alignment/justification modes. OIV does not currently justify a
  // multi-string text node as a "block" of text.

  SbViewportRegion vport = m_curViewport;
  if (state != NULL)
    vport = SoViewportRegionElement::get( state );

  // Update bounding box.
  // Note1: This node has "probably" been added to a group node, but... we could arrive
  //        here through the node sensor without ever ref'ing it. So avoid crashing.
  // Note2: SoText2 considers clipping at the window edge when it computes its bounding
  //        box. Sigh.  So we need to temporarily move the string to a "safe" location
  //        while we compute the actual bounding. (Even this will break if there are too
  //        many strings but that doesn't seem like a common use case.) Remember to back
  //        this out before updating border geometry...
  const SbVec3f SAFE_POS( -0.99f, 0.99f, 0 );

  const SbBool saveNotTrans = m_tranNode->translation.enableNotify( FALSE );
  const SbVec3f oldPos = m_tranNode->translation.getValue();
  m_tranNode->translation.setValue( SAFE_POS );

  this->ref();
  SoGetBoundingBoxAction gbba( vport );
  gbba.apply( m_textSep.ptr() );
  SbBox3f bbox = gbba.getBoundingBox();
  this->unrefNoDelete();

  // Update alignment
  int boxAlignH = alignmentH.getValue();
  int boxAlignV = alignmentV.getValue();
  int txtAlignH = textAlignH.getValue();
  SbVec3f pos  = position.getValue();   // App specified text box position
  SbVec3f size = bbox.getSize();
  SbVec3f delta( 0, 0, 0 );

  // Horizontal alignment adjustment
  if (boxAlignH == TextBox::LEFT) {
    if (txtAlignH == TextBox::CENTER)
      delta[0] = 0.5f * size[0];
    else if (txtAlignH == TextBox::RIGHT)
      delta[0] = size[0];
  }
  else if (boxAlignH == TextBox::CENTER) {
    if (txtAlignH == TextBox::LEFT)
      delta[0] = -0.5f * size[0];
    else if (txtAlignH == TextBox::RIGHT)
      delta[0] =  0.5f * size[0];
  }
  else if (boxAlignH == TextBox::RIGHT) {
    if (txtAlignH == TextBox::LEFT)
      delta[0] = -size[0];
    else if (txtAlignH == TextBox::CENTER)
      delta[0] = -0.5f * size[0];
  }

  // Vertical alignment adjustment
  if (boxAlignV != TextBox::TOP) {
    if (boxAlignV == TextBox::MIDDLE) {
      delta[1] = 0.5f * size[1];
    }
    else if (boxAlignV == TextBox::BOTTOM) {
      delta[1] = size[1];
    }
  }

  // Update our actual position
  pos += delta;
  m_tranNode->translation.setValue( pos );
  m_tranNode->translation.enableNotify( saveNotTrans );
  if ( pos != oldPos )
    // notify position changed
    m_tranNode->touch();

  // Update border geometry
  SbVec3f bmin = bbox.getMin() - SAFE_POS + pos;
  SbVec3f bmax = bbox.getMax() - SAFE_POS + pos;
  bbox.setBounds( bmin, bmax );
  SoVertexProperty* vprop = (SoVertexProperty*)m_borderGeom->vertexProperty.getValue();
  updateBorderGeom( vprop, bbox );

  // Remember current state
  m_curBBox = bbox;
  m_curViewport = vport;
  m_isModified  = false;
}

///////////////////////////////////////////////////////////////////////////////
// Called when any of our fields or children are modified.
//
// This is a "shotgun" approach, but we wanted to allow for the possibility of
// the application changing, for example, the font name or size.
void
TextBox::nodeSensorCB( void* data, SoSensor* /*sensor*/ )
{
  TextBox* self = (TextBox*)data;
  self->m_isModified = true;
  self->enableNotify( FALSE );

  // Border
  int which = self->border.getValue() ? 0 : -1;
  if (self->m_borderSwitch->whichChild.getValue() != which) {
    self->m_borderSwitch->whichChild = which;
  }

  // Border color
  uint32_t packedColor = self->borderColor.getValue().getPackedValue();
  SoVertexProperty* vprop = (SoVertexProperty*)self->m_borderGeom->vertexProperty.getValue();
  if (vprop->orderedRGBA[0] != packedColor) {
    vprop->orderedRGBA.set1Value( 0, packedColor );
  }
  self->enableNotify( TRUE );
}

///////////////////////////////////////////////////////////////////////////////
void
TextBox::callbackNodeCB( void* data, SoAction* action )
{
  TextBox* self = (TextBox*)data;

  // Ignore actions that (probably) don't require adjusting position.
  if (action->isOfType(SoHandleEventAction::getClassTypeId()) ||
      action->isOfType(SoSearchAction::getClassTypeId()) ||
      action->isOfType(SoGetBoundingBoxAction::getClassTypeId()))
  {
    return;
  }

  // We have to check the viewport because the bounding box of an SoText2
  // depends on the current window size.
  // Changes to actual fields should be handled by the node sensor.
  SoState* state = action->getState();
  SbViewportRegion vport = SoViewportRegionElement::get( state );
  if (self->m_isModified == false && vport == self->m_curViewport)
    return;

  // Something changed...
  self->m_curViewport = vport;
  self->updatePosAndGeom( state );
}

///////////////////////////////////////////////////////////////////////////////
SoFont*
TextBox::getFontNode()
{
  return (m_fontNode.ptr() ? m_fontNode.ptr() : NULL);
}

///////////////////////////////////////////////////////////////////////////////
SoText2*
TextBox::getTextNode()
{
  return (m_textNode.ptr() ? m_textNode.ptr() : NULL);
}

///////////////////////////////////////////////////////////////////////////////
void
TextBox::setLine( const SbString& text, int line )
{
  // Note: SoText2 may crash if one of the string is 'empty'
  //       (which is different from having 0 characters).
  if (text.isEmpty())
    m_textNode->string.set1Value( line, SbString("") );
  else
    m_textNode->string.set1Value( line, text );
}

///////////////////////////////////////////////////////////////////////////////
const SbString&
TextBox::getLine( int line )
{
  return m_textNode->string[line];
}

///////////////////////////////////////////////////////////////////////////////
void
TextBox::addLine( const SbString& text )
{
  // Note: SoText2 may crash if one of the string is 'empty'
  //       (which is different from having 0 characters).
  int n = m_textNode->string.getNum();
  if (text.isEmpty())
    m_textNode->string.set1Value( n, SbString("") );
  else
    m_textNode->string.set1Value( n, text );
}

///////////////////////////////////////////////////////////////////////////////
void
TextBox::deleteAll()
{
  m_textNode->string.deleteValues( 0 );
}

///////////////////////////////////////////////////////////////////////////////
void
TextBox::deleteLines( int start, int numToDelete )
{
  int numLines = m_textNode->string.getNum();
  if (start < 0 || start >= numLines || numToDelete == 0)
    return;

  if (numToDelete == -1)
    numToDelete = numLines - start;
  else
    numToDelete = std::min( numToDelete, (numLines - start) );
  m_textNode->string.deleteValues( start, numToDelete );
}

///////////////////////////////////////////////////////////////////////////////
int
TextBox::getNumLines() const
{
  return m_textNode->string.getNum();
}