// ViewManager utility class
//
///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

// NOTES:
//   - Goal:
//     1. Manage a list of "views", where a view consists of (at least)
//        - A viewport
//        - A scene to render (which includes a camera!)
//        - Flag that controls if the view is active (i.e. traverrsed).
//        - Flag to optionally draw a border
//
//     2. Route events to the "correct" view based on the position of the event,
//        the viewport of each view, the order of views, etc.
//        This is the easy part.
//
//   - MouseWheel events
//     Through version 9.7.0, the RemoteViz client does set 'position' in mouse
//     wheel events.  As a result they always appear to be at position (0,0) and
//     get handled by the wrong view.  We work around this by ignoring the position
//     in mouse wheel events and forcing MouseWheel events to be handled by
//     m_eventViewIndex, the view that contained the last known event position.
//     
//   - Organization of our data.
//        - We could have a separate list for each property of a view (see
//          above).  But that's ridiculous and the views would get out of sync
//          very easily, probably resulting in a crash.
//        - We could create a new class to encapsulate all the properties of a
//          view and keep a list of each classes.  This is a valid solution as
//          far as computer science, but fails an Open Inventor requirement.
//          We would not be able to write the scene graph to a file, reload
//          and know the layout of the viewports.
//        - So finally it seems that the best solution is to create a new
//          _node_ class that encapsulates all the properties of a view.
//          Given that, we might as well just store the SceneView nodes as
//          children.
//
// ISSUES:
//   - Should we try to enforce that children can only be SceneView nodes?
//     Status: Try it and see how it works.
//             We could ignore this and depend on applications to do the right
//             thing, but we risk a crash.
//
// TODO:
//   - Should be able to specify viewport in pixels.
//   - ? Could be useful to make maximizeView() a feature.


#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTextProperty.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/touch/events/SoTouchEvent.h>

#include <Inventor/gestures/events/SoRotateGestureEvent.h>
#include <Inventor/gestures/events/SoScaleGestureEvent.h>

#include <Inventor/sensors/SoNodeSensor.h>

#include <Inventor/errors/SoDebugError.h>

#include "Medical/nodes/ViewManager.h"
#include "Medical/nodes/SceneView.h"

SO_NODE_SOURCE(ViewManager);

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
ViewManager::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

  // Initialize type id variables
  SO_NODE_INIT_CLASS(ViewManager, SoGroup, "Group");
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
ViewManager::exitClass()
{
  SO__NODE_EXIT_CLASS(ViewManager);
}

///////////////////////////////////////////////////////////////////////////////
ViewManager::ViewManager()
{
  // Setup fields
  SO_NODE_CONSTRUCTOR(ViewManager);

  // Internal state
  m_numViews = 0;
  m_buttonsPressed   =  0;  // No buttons are pressed
  m_eventViewIndex   = -1;  // No events have been processed yet
  m_captureViewIndex = -1;  // No view owns the event stream
  m_eventPos.setValue(0,0);

  // Detect changes to our fields or children
  m_sensor = new SoNodeSensor( sensorCB, (void*)this );
  m_sensor->setPriority( 0 );
  m_sensor->attach( this );
}

///////////////////////////////////////////////////////////////////////////////
ViewManager::~ViewManager()
{
  delete m_sensor;
  m_numViews = 0;
}

///////////////////////////////////////////////////////////////////////////////
// Error msg
static void badChildError( const char* methodName )
{
  SbString str = SbString("ViewManager::") + methodName;
  SoDebugError::post( str.toLatin1(), "Invalid child - must be a 'SceneView' node." );
}

///////////////////////////////////////////////////////////////////////////////
// Add a view (convenience method).
// Adds the specified view to the end of the list.
void
ViewManager::addView( const SceneView* newView )
{
  if (newView->isOfType(SceneView::getClassTypeId()))
    SoGroup::addChild( (SoNode*)newView );
  else
    badChildError("addView");
}

///////////////////////////////////////////////////////////////////////////////
// Override generic child methods to enforce children are SceneView nodes.
void ViewManager::addChild( SoNode* child )
{
  if (child->isOfType(SceneView::getClassTypeId()))
    SoGroup::addChild( child );
  else
    badChildError("addChild");
}
void ViewManager::insertChild( SoNode *child, int newChildIndex )
{
  if (child->isOfType(SceneView::getClassTypeId()))
    SoGroup::insertChild( child, newChildIndex );
  else
    badChildError("insertChild");
}
void ViewManager::replaceChild( SoNode* oldChild, SoNode* newChild )
{
  if (newChild->isOfType(SceneView::getClassTypeId()))
    SoGroup::replaceChild( oldChild, newChild );
  else
    badChildError("replaceChild");
}
void ViewManager::replaceChild( int index, SoNode* newChild )
{
  if (newChild->isOfType(SceneView::getClassTypeId()))
    SoGroup::replaceChild( index, newChild );
  else
    badChildError("replaceChild");
}

///////////////////////////////////////////////////////////////////////////////
// Number of views
int
ViewManager::getNumViews() const
{
  return this->getNumChildren();
}

///////////////////////////////////////////////////////////////////////////////
// Returns the view with the specified index (convenience method).
// Returns null if index is not valid.
SceneView*
ViewManager::getView( int index ) const
{
  int numViews = getNumViews();
  if (index >= 0 && index < numViews)
    return (SceneView*)getChild( index );
  else
    return NULL;
}

///////////////////////////////////////////////////////////////////////////////
// Returns the first view containing the specified pixel coordinate.
// This is the view that would handle an event at this location.
// This is convenient, for example, to implement global hot keys that should
// only affect the view that contains the event.  Then it is not necessary to
// duplicate an SoEventCallback in each view for this purpose.
// Returns null if the location is not valid. */
SceneView*
ViewManager::getView( const SbVec2f& position ) const
{
  int index = getViewIndex( position );
  return getView( index );
}

///////////////////////////////////////////////////////////////////////////////
// Returns the index of the specified view.
// Returns -1 if the specified view does not exist.
int
ViewManager::getViewIndex( const SceneView* view ) const
{
  int numViews = getNumViews();
  for (int index = 0; index < numViews; ++index) {
    if ((SoNode*)view == getChild(index)) {
      return index;
    }
  }
  return -1;
}

///////////////////////////////////////////////////////////////////////////////
// Returns a view index for the specified window coordinate.
// This is the index of the view that would handle an event at this location.
// Returns -1 if the location is not valid.
int
ViewManager::getViewIndex( const SbVec2f& pos ) const
{
  int numViews = getNumViews();
  // Loop over views
  for (int i = 0; i < numViews; ++i) {
    // If view is valid and active...
    // Note that we try to enforce that all children are a SceneView node,
    // but it's safer to test and avoid a crash.
    const SceneView* view = dynamic_cast<SceneView*>(getChild(i));
    if (view != NULL && view->active.getValue() == TRUE) {

      // Then if event position is inside the view's viewport...
      const SbVec4f& vport = view->getViewportPixels();
      if (pos[0] >= vport[0] && pos[1] >= vport[1]) {
        if (pos[0] < (vport[0] + vport[2]) && pos[1] < (vport[1] + vport[3])) {
          // This is the one...
          return i;
        }
      }
    }
  }
  return -1;
}

///////////////////////////////////////////////////////////////////////////////
// Returns the index of the view that is handling the current event.
// This value is set when a HandleEventAction traverses the view manager and
// is valid for nodes in the scene graph below the view manager.
// This value is convenient for event handlers that should only affect the
// current view and need to access a node associated with that view.
// If event capture is active (a mouse button was pressed), then the index
// returned is the event capture view, not necessarily the same as the view
// currently containing the cursor.
// Returns -1 if not possible to answer the query.
int
ViewManager::getLastEventViewIndex() const
{
  return m_eventViewIndex;
}

///////////////////////////////////////////////////////////////////////////////
// Returns the position of the last event handled (in pixels).
// Generally Open Inventor sets the cursor position for all mouse and keyboard events.
// But this query could be useful if it is necessary to handle an event that does
// not have an associated position on screen. Returns (0,0) if no other info.
SbVec2f
ViewManager::getLastEventPosition() const
{
  return m_eventPos;
}

///////////////////////////////////////////////////////////////////////////////
// Convenience method to set viewport of a view.
void
ViewManager::setViewport( int index, SbVec4f& vport )
{
  int numViews = getNumViews();
  if (index >= 0 && index < numViews) {
    getView(index)->setViewport( vport );
  }
}

///////////////////////////////////////////////////////////////////////////////
// Convenience method to set viewport of a view.
void
ViewManager::setViewport( int index, float xorig, float yorig, float xsize, float ysize )
{
  int numViews = getNumViews();
  if (index >= 0 && index < numViews) {
    getView(index)->setViewport( xorig, yorig, xsize, ysize );
  }
}

///////////////////////////////////////////////////////////////////////////////
// Set event capture view.
// If viewIndex >= 0, calls enableEventHandling with true for that view and
// false for every other view.  If viewIndex < 0, sets true for all views.
// Invalid indices are ignored.  Used internally to manage event capture.
// Generally the application should not call this method.
void
ViewManager::setEventCaptureView( int viewIndex )
{
  int numViews = getNumViews();
  if (viewIndex < 0 || viewIndex >= numViews) {
    // Reset to all views accepting events
    for (int i = 0; i < numViews; ++i) {
      getView( i )->enableEventHandling( TRUE );
    }
  }
  else {
    // Only the event capture view will accept events
    for (int i = 0; i < numViews; ++i) {
      getView( i )->enableEventHandling( (i == viewIndex) ? TRUE : FALSE );
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
// Called when the node is modified.
// We have no fields so this means:
//   - Children were added/deleted/replaced.
//     This is useful because we want to enforce only have SceneView children.
//     or...
//   - Something below us in the scene graph was modified.
//     This is useless, but unavoidable in Open Inventor.
void
ViewManager::sensorCB( void* data, SoSensor* /*_sensor*/ )
{
  ViewManager* self = (ViewManager*)data;

  bool viewsChanged = false;
  int numViews = self->getNumViews();
  if (numViews != self->m_numViews) {
    // More or less views than we had before is a definite change!
    viewsChanged = true;
  }
  else {
    // Else we need to check if one of the views now points to a different node.
  }

  // TODO: If a new node was added, check that it's really a View node.
}

///////////////////////////////////////////////////////////////////////////////
// Handle traversal by SoHandleEventAction
//
// Which viewport should handle the event?
// Based on position of event in pixel coordinates and order of views.
// I.e. the first view found that contains the event handles it.
//
// Also need to support "capture" of event stream while a mouse button is pressed.

void ViewManager::handleEvent(SoHandleEventAction *action)
{
  // Get event.
  const SoEvent* evt = action->getEvent();

  // Update last known event position in pixels.
  // Note: If the events are coming from a RemoteViz client, then (through
  // OIV 9.7), the event position for MouseWheel and Keyboard events is not
  // valid.  Issues 56245 and 56223.  With certainty we can only trust mouse
  // button, mouse move (location2) and touch events to have a valid position.
  // In all other cases we use the last known event position "as is".
  if (evt->isOfType(SoMouseButtonEvent::getClassTypeId())
   || evt->isOfType(SoLocation2Event::getClassTypeId())
   || evt->isOfType(SoTouchEvent::getClassTypeId()))
  {
    m_eventPos = evt->getPositionFloat();
  }

  // Remember current state
  unsigned int prevButtonsPressed = m_buttonsPressed;

  // If this is a mouse button event...
  SoMouseButtonEvent* btnEvent = dynamic_cast<SoMouseButtonEvent*>((SoEvent*)evt); // Null if different event
  if (btnEvent != NULL) {
    unsigned int buttonNum = (unsigned int)btnEvent->getButton();
    // Update summary of button state
    if (btnEvent->getState() == SoButtonEvent::DOWN) {
      m_buttonsPressed |= (1 << buttonNum);
    }
    else { // Button released
      m_buttonsPressed ^= (1 << buttonNum);
    }
  }

  // If a button was pressed and no buttons were previously pressed...
  if (m_buttonsPressed != 0 && prevButtonsPressed == 0) {
    // Begin event capture
    m_captureViewIndex = getViewIndex( m_eventPos );
    setEventCaptureView( m_captureViewIndex );
  }

  // If events are captured by a specific view
  if (m_captureViewIndex >= 0) {
    m_eventViewIndex = m_captureViewIndex;

    // Send event to capture view
    const SceneView* view = getView(m_captureViewIndex);
    if (view != NULL)
      SoGroup::handleEvent( action );

    // If no buttons are currently pressed, end event capture.
    if (m_buttonsPressed == 0) {
      m_captureViewIndex = -1;
      setEventCaptureView( m_captureViewIndex );
    }

    // Nothing more to do.
    return;
  }

  // Else, loop over views and find where to send the event
  int numViews = getNumViews();
  for (int viewIndex = 0; viewIndex < numViews; ++viewIndex) {

    // If view is valid and active...
    const SceneView* view = dynamic_cast<SceneView*>(getChild(viewIndex));
    if (view != NULL && view->active.getValue() == TRUE) {

      // Then if event position is inside the view's viewport...
      const SbVec4f& vport = view->getViewportPixels();
      if (m_eventPos[0] >= vport[0] && m_eventPos[1] >= vport[1]) {
        if (m_eventPos[0] < (vport[0] + vport[2]) && m_eventPos[1] < (vport[1] + vport[3])) {

          // Send event to this view
          setEventCaptureView( viewIndex );
          m_eventViewIndex = viewIndex;
          SoGroup::handleEvent( action );
          setEventCaptureView( -1 );

          // If handled, then we're done
          if (action->isHandled())
            break;
        }
      }
    }
  }
}

