
if (CMAKE_VERSION VERSION_LESS 3.5)
    message(FATAL_ERROR "OpenInventor requires at least CMake version 3.5")
endif()

if (NOT OpenInventor_FIND_COMPONENTS)
    if(NOT OpenInventor_FIND_QUIETLY)
      message(STATUS "No components specified for OpenInventor, trying to find all of OpenInventor")
    endif()
    set(OpenInventor_FIND_COMPONENTS inventor)
    include(${CMAKE_CURRENT_LIST_DIR}/ComponentsList.cmake)
endif()

##############################################################
#
# OpenInventor cannot be used without OpenGL
# Make sure OpenGL is available
#
##############################################################
set(OpenGL_GL_PREFERENCE GLVND)
find_package(OpenGL ${OPENGL_COMPONENTS} REQUIRED)

##############################################################
#
# Set up dependencis of the libraries, so that when asking for
# e.g. volumeViz, all its dependencies are also found
# (i.e. ldm, inventor scaleViz inventor_computing_algorithms
# inventor_gl inventor_image inventor_base glew)
#
##############################################################
set(inventor_gl_dependencies inventor_base glew)
set(inventor_dependencies inventor_gl inventor_image)
set(inventor_ui_dependencies inventor scaleViz)
set(inventor_ui_qt_dependencies inventor_ui)
set(inventor_ui_xt_dependencies inventor_ui)
set(inventor_ui_win_dependencies inventor_ui)
set(ldm_dependencies inventor scaleViz inventor_computing_algorithms)
set(volumeViz_dependencies ldm)
set(inventor_ViewerComponents_dependencies inventor)
set(inventor_QtViewerComponents_dependencies inventor_ViewerComponents)
set(remoteViz_services_dependencies inventor)

function(resolve_dependencies module_list)
  foreach(module ${module_list})
    # if this module has dependencies, add them to the list of things to find
    if(${module}_dependencies)
      list(APPEND OpenInventor_FIND_COMPONENTS ${${module}_dependencies})
      resolve_dependencies(${${module}_dependencies})
    endif()
  endforeach()
  #propagate the changes to the parent scope
  set(OpenInventor_FIND_COMPONENTS ${OpenInventor_FIND_COMPONENTS} PARENT_SCOPE)
endfunction()

resolve_dependencies("${OpenInventor_FIND_COMPONENTS}")
list(REMOVE_DUPLICATES OpenInventor_FIND_COMPONENTS)

##############################################################
#
# set legacy names
# These are the names used by OIV < 10 for the components
# For compatibility reasons with old client CMakeFiles
# we need to provide this so we don't break existing CMakes
#
##############################################################

set(fei_inventor_legacy_name OpenInventor)
set(fei_inventor_base_legacy_name OpenInventorBase)
set(fei_inventor_gl_legacy_name OpenInventorGL)
set(fei_inventor_image_legacy_name OpenInventorImage)

set(fei_inventor_ui_legacy_name OpenInventorGui)
set(fei_inventor_ui_qt_legacy_name OpenInventorQt)
set(fei_inventor_ui_xt_legacy_name OpenInventorXt)
set(fei_inventor_ui_win_legacy_name OpenInventorWin)

set(fei_dialogViz_legacy_name DialogViz)
set(fei_dialogViz_qt_legacy_name DialogVizQt)

set(fei_meshViz_legacy_name MeshViz)
set(fei_meshVizXLM_extractor_legacy_name MeshVizExtractor)
set(fei_meshVizXLM_dataMapping_legacy_name MeshVizDataMapping)

set(fei_inventor_computing_algorithms_legacy_name OpenInventorAlgorithms)
set(fei_scaleViz_legacy_name ScaleViz)

set(fei_volumeViz_legacy_name VolumeViz)
set(fei_ldm_legacy_name LDM)

set(fei_hardCopy_legacy_name HardCopy)

set(fei_glew_legacy_name OpenInventorGLEW)

set(fei_ivTune_legacy_name IvTune)
set(fei_ivTune_extender_legacy_name IvtExtenderAPI)

set(fei_remoteViz_services_legacy_name RemoteVizServices) 
set(fei_remoteViz_client_legacy_name RemoteVizClient) 

# Inventor io plugin modules
set(fei_inventor_io_stl_legacy_name  OpenInventorIO_STL)
set(fei_inventor_io_cad_legacy_name  OpenInventorIO_CAD)
set(fei_inventor_io_vrml_legacy_name OpenInventorIO_VRML)
set(fei_inventor_io_dxf_legacy_name  OpenInventorIO_DXF)

# libraries in OIVHOME/source
set(fei_inventor_medical_legacy_name OpenInventorMedical)
set(fei_inventor_ViewerComponents_legacy_name OpenInventorViewerComponents)
set(fei_inventor_QtViewerComponents_legacy_name OpenInventorQtViewerComponents)
set(fei_inventor_WinViewerComponents_legacy_name OpenInventorWinViewerComponents)
set(fei_inventor_MFCViewerComponents_legacy_name OpenInventorMFCViewerComponents)

#set(_legacy_name )


# define OIVHOME for convenience
get_filename_component(OIVHOME "${CMAKE_CURRENT_LIST_DIR}/../../.." ABSOLUTE)

set(OIV_INCLUDE_DIR "${OIVHOME}/include")

##############################################################
#
# This defines necessary variables in order to find
# OIV Modules
#
# TODO this should be simplified/abstracted (into a CMake module perhaps)
# because it is also used in the OIV buildsystem
#
##############################################################
if(CMAKE_COMPILER_IS_GNUCXX OR APPLE)
  if(CMAKE_CXX_COMPILER_ID MATCHES "Clang") # Clang or AppleClang
    set(OIV_CXX_COMPILER_ID "clang")
    string(SUBSTRING ${CMAKE_CXX_COMPILER_VERSION} 0 1 major_version)
  elseif(CMAKE_CXX_COMPILER_ID STREQUAL "GNU")
    set(OIV_CXX_COMPILER_ID "gcc")
    if (CMAKE_CXX_COMPILER_VERSION VERSION_GREATER 9.9)
      string(SUBSTRING ${CMAKE_CXX_COMPILER_VERSION} 0 2 major_version)
    else()
      string(SUBSTRING ${CMAKE_CXX_COMPILER_VERSION} 0 1 major_version)
    endif()
  endif()
  set(OIV_CXX_COMPILER_VERSION ${major_version} CACHE STRING "GCC version used for prepacked")
else()
  string(TOLOWER ${CMAKE_CXX_COMPILER_ID} OIV_CXX_COMPILER_ID)

  # regex magic to extract the MSVC version from the generator string that looks like "Visual Studio 14 2015 Win64"
  string(REGEX REPLACE "Visual Studio ([0-9][0-9]) [0-9][0-9][0-9][0-9].*" "\\1" OIV_CXX_COMPILER_VERSION ${CMAKE_GENERATOR})
endif()

if(WIN32)
  if(CMAKE_CL_64)
    set(OIV_SYSTEM_PROCESSOR x86_64)
  else()
    set(OIV_SYSTEM_PROCESSOR x86)
  endif()
else()
  set(OIV_SYSTEM_PROCESSOR ${CMAKE_SYSTEM_PROCESSOR})
endif()

if(APPLE)
  set(OIV_CXX_PLATFORM_ID  "OSX")
else()
  set(OIV_CXX_PLATFORM_ID  ${CMAKE_CXX_PLATFORM_ID})
endif()

set(CMAKE_ARCHITECTURE "${OIV_CXX_PLATFORM_ID}-${OIV_SYSTEM_PROCESSOR}-${OIV_CXX_COMPILER_ID}${OIV_CXX_COMPILER_VERSION}")

# these variables are used by find_module_config to avoid rebuilding them for every module
if(DEFINED OIVARCH) # set as a cmake variable (e.g. -DOIVARCH=...)
  # nothing to do
elseif(DEFINED ENV{OIVARCH}) # set by the environment
  set(OIVARCH $ENV{OIVARCH})
else() # using current compiler arch
  set(OIVARCH arch-${CMAKE_ARCHITECTURE})
endif()


# check if OIVARCH format is appropriate for finding packages (i.e. no config, only arch)
# backup and restore the input value if needed
if (OIVARCH MATCHES "-(R|r)elease$")
  set(OIVARCH_BACKUP ${OIVARCH})
  string(REGEX REPLACE "-(R|r)elease$" "" OIVARCH "${OIVARCH}")
elseif (OIVARCH MATCHES "-(D|d)ebug$")
  set(OIVARCH_BACKUP ${OIVARCH})
  string(REGEX REPLACE "-(D|d)ebug$" "" OIVARCH "${OIVARCH}")
elseif (OIVARCH MATCHES "-..(C|c)onfiguration.$")
  set(OIVARCH_BACKUP ${OIVARCH})
  string(REGEX REPLACE "-..(C|c)onfiguration.$" "" OIVARCH "${OIVARCH}")
endif()

set(OIVARCH_DEBUG ${OIVARCH}-debug)
set(OIVARCH_RELEASE ${OIVARCH}-release)

##############################################################
#
# This macro helps finding a module for a specified configuration
#
##############################################################
macro(find_module_config module config)
  # Find the ${config} part of the current module
  find_library(${module}_LIBRARY_${config}
      NAMES ${module} ${module}D
      PATHS ${OIVHOME}/${OIVARCH_${config}}/bin/ ${OIVHOME}/${OIVARCH_${config}}/lib/
      NO_DEFAULT_PATH)
  if(${module}_LIBRARY_${config})
    # The module was found, set the location properly
    if(CMAKE_BUILD_TYPE)
      set_target_properties(${module} PROPERTIES IMPORTED_IMPLIB ${${module}_LIBRARY_${config}})
      set_target_properties(${module} PROPERTIES IMPORTED_LOCATION ${${module}_LIBRARY_${config}})
    else()
      set_target_properties(${module} PROPERTIES IMPORTED_IMPLIB_${config} ${${module}_LIBRARY_${config}})
      set_target_properties(${module} PROPERTIES IMPORTED_LOCATION_${config} ${${module}_LIBRARY_${config}})
    endif()
  else()
     # if the module was required, error out
     if(OpenInventor_FIND_REQUIRED)
       message(FATAL_ERROR "failed to find OpenInventor componment ${module} in PATHS ${OIVHOME}/${OIVARCH_${config}}/bin/ ${OIVHOME}/${OIVARCH_${config}}/lib/")
     # if finding was not quiet, output some information
     elseif(NOT OpenInventor_FIND_QUIETLY)
       message(WARNING "failed to find OpenInventor componment ${module} in PATHS ${OIVHOME}/${OIVARCH_${config}}/bin/ ${OIVHOME}/${OIVARCH_${config}}/lib/")
     endif()
  endif()
endmacro()

##############################################################
#
# This macro helps finding a module, and output error/warning
# messages if the library is not found
#
##############################################################
macro(find_module module_base_name)
  set(module fei_${module_base_name})
  add_library(${module} SHARED IMPORTED GLOBAL)
  if(CMAKE_VERSION VERSION_GREATER 3.11)
    add_library(OIV::${module} ALIAS ${module})
  endif()

  if(CMAKE_BUILD_TYPE)
    string(TOUPPER ${CMAKE_BUILD_TYPE} config)
    find_module_config(${module} ${config})

    if(${module}_LIBRARY_${config})
      set(${module}_FOUND "True")
    endif()
  else()
    find_module_config(${module} RELEASE)
    find_module_config(${module} DEBUG)

    if(${module}_LIBRARY_RELEASE AND ${module}_LIBRARY_DEBUG)
      set(${module}_FOUND "True")
    endif()
  endif()

  # set the module to found only if either DEBUG or RELEASE components were found
  if(NOT ${module}_FOUND)
      set(${module}_FOUND "${module}_NOTFOUND")
      # if the module was required, error out
      if(OpenInventor_FIND_REQUIRED)
        message(FATAL_ERROR "failed to find OpenInventor componment ${module}; ${${module}_LIBRARY_RELEASE} OR ${${module}_LIBRARY_DEBUG}")
      # if finding was not quiet, output some information
      elseif(NOT OpenInventor_FIND_QUIETLY)
        message(WARNING "failed to find OpenInventor componment ${module}; ${${module}_LIBRARY_RELEASE} OR ${${module}_LIBRARY_DEBUG}")
      endif()
  endif()

  if(${module_base_name}_dependencies)
    foreach(dependency ${${module_base_name}_dependencies})
      set_property(TARGET ${module}
        APPEND
        PROPERTY INTERFACE_LINK_LIBRARIES "fei_${dependency}")
    endforeach()
  endif()

  # set the legacy name for compatibility with old CMake files
  set(${${module}_legacy_name}_LIBRARIES "${module}")

  # set a variable indicating if the module is enabled
  # this is transition to OpenInventorConfig.cmake and
  # is necessary until the DEV_BUILD is removed
  # once DEV_BUILD is removed this should not be necessary anymore
  string(TOUPPER ${module_base_name} base_name_upper)
  set(OIV_ENABLE_${base_name_upper} ON)
endmacro()

##############################################################
# If something fails later, this will be set to false
# by the component that was not found
##############################################################
set(OpenInventor_FOUND True)

##############################################################
#
# Find all the modules !
#
##############################################################
foreach(module ${OpenInventor_FIND_COMPONENTS})
    find_module(${module})
    # The module was not found set OpenInventor_FOUND to FALSE
    # as we don't want to exit sucessfully
    if(NOT ${module}_FOUND)
      set(OpenInventor_FOUND False)
    endif()
endforeach()

##############################################################
#
# Alias gui modules as Xt and Win are used on all platforms
#
##############################################################
if(WIN32)
  set(OpenInventorXt_LIBRARIES ${OpenInventorWin_LIBRARIES})
else()
  if(APPLE)
    set(DialogViz_LIBRARIES ${DialogVizQt_LIBRARIES})
    set(OpenInventorXt_LIBRARIES ${OpenInventorQt_LIBRARIES})
    set(OpenInventorWin_LIBRARIES ${OpenInventorQt_LIBRARIES})
  else()
    set(OpenInventorWin_LIBRARIES ${OpenInventorXt_LIBRARIES})
  endif()
endif()

set(DialogVizGui_LIBRARIES ${DialogViz_LIBRARIES})

##############################################################
#
# Set additional properties on the targets
# i.e. link interface and include dirs
#
##############################################################
set(TARGETED_QT_VERSION Qt5)
if(NOT OIV_FORCE_QT5 AND CMAKE_CXX_COMPILER_ID STREQUAL "MSVC")
  if(CMAKE_CXX_COMPILER_VERSION VERSION_GREATER 19.30)
    set(TARGETED_QT_VERSION Qt6)
  endif()
endif()
set_property(TARGET ${DialogViz_LIBRARIES}
  APPEND
  PROPERTY INTERFACE_LINK_LIBRARIES ${OpenInventorWin_LIBRARIES})
set_property(TARGET ${OpenInventorBase_LIBRARIES}
  APPEND
  PROPERTY INTERFACE_INCLUDE_DIRECTORIES ${OIV_INCLUDE_DIR})
set_property(TARGET ${OpenInventorQt_LIBRARIES}
  APPEND
  PROPERTY INTERFACE_LINK_LIBRARIES "${${TARGETED_QT_VERSION}Core_LIBRARIES};${${TARGETED_QT_VERSION}Gui_LIBRARIES};${${TARGETED_QT_VERSION}OpenGL_LIBRARIES};${${TARGETED_QT_VERSION}Widgets_LIBRARIES};${${TARGETED_QT_VERSION}X11Extras_LIBRARIES}")
if(NOT OIV_ENABLE_HEADLESS AND UNIX AND NOT APPLE)
  find_package(Motif REQUIRED)
  find_package(X11 REQUIRED)
  set_property(TARGET ${OpenInventorXt_LIBRARIES}
    APPEND
    PROPERTY INTERFACE_LINK_LIBRARIES "${MOTIF_LIBRARIES};${X11_Xt_LIB}")
endif()

if(OIV_ENABLE_HEADLESS)
  # property for Headless versions of Open Inventor
  set_target_properties(${OpenInventorBase_LIBRARIES} PROPERTIES COMPILE_DEFINITIONS OIV_HEADLESS)
endif()
##############################################################

# ensure autolinking works
# LEGACY - for compatibility reasons, ensure we build anything in the OIVHOME/examples/bin/OIVARCH folder
if (CMAKE_GENERATOR MATCHES "Visual Studio")
  link_directories(${OIVHOME}/${OIVARCH}-$(Configuration)/lib)
  set(OIV_OUTPUT_DIR ${OIVHOME}/examples/bin/${OIVARCH}-$(Configuration) CACHE PATH "Output directory path for executable files")
else()
  string(TOLOWER ${CMAKE_BUILD_TYPE} config)
  set(OIV_OUTPUT_DIR ${OIVHOME}/examples/bin/${OIVARCH}-${config} CACHE PATH "Output directory path for executable files")
endif()

# restore OIVARCH if needed
if (DEFINED OIVARCH_BACKUP)
  set(OIVARCH ${OIVARCH_BACKUP})
endif()
