#coding: utf-8

import json
import requests
import os
import re
import sys
import subprocess
import shlex
import xml.sax.saxutils
import datetime
import base64
import socket

cdash_server_api = "https://frbor-cdash01/api/v1"
branch_name = ""
standard_branches = [
    'upcoming/minor10',
    'upcoming/patch10',
    'upcoming/patch9',
    'stable/minor10',
    'stable/patch10',
    'stable/patch9',
    'release/stable/patch10',
    'release/stable/minor10',
    'release/upcoming/patch10',
    'release/upcoming/minor10'
]

try:
    if sys.platform == 'win32':
        import winreg

# otherwise use Python 2 import
except ImportError:
    if sys.platform == 'win32':
        import _winreg

#-----------------------------------------------
# Common Environement variables
#-----------------------------------------------
os.environ["OIV_FORCE_DEFAULT_ERROR_HANDLER"] = "1"
os.environ["OIV_MODULE_CHECK_STDOUT"] = "1"
os.environ["OIV_BETA_WATERMARK"] = "0"
os.environ["OIV_STACKWALKER_HTML_OUTPUT"] = "0"
os.environ["OIV_CHECK_GL_ERRORS"] = "1"
os.environ["SCALEVIZ_NO_DELETE"] = "1"
now = datetime.datetime.now()
if now.day % 2 == 1:
    os.environ["OIV_SHADER_DISK_CACHE"] = "0"

ddsimApp = "/dssim.exe" if sys.platform == 'win32' else "/dssim"
imageCompTool = (os.path.normpath(os.getenv("OIVHOME") + "/../ssim/") + ddsimApp).replace("\\", "/")
ssl_cert_path = (os.path.normpath(os.getenv("OIVHOME") + "/../cmake/frbor-cdash01.pem")).replace("\\", "/")

if os.path.exists(imageCompTool) == False:
    raise ImportError("SSIM tool is missing in the OivSuite folder")


# this makes the output unbeffered, which is necessary when there are multiple threads
# otherwise the calls in the threads are flushed after the calls in the main process.
class Unbuffered(object):
    def __init__(self, stream):
        self.stream = stream

    def write(self, data):
        self.stream.write(data)
        self.stream.flush()

    def writelines(self, datas):
        self.stream.writelines(datas)
        self.stream.flush()

    def __getattr__(self, attr):
        return getattr(self.stream, attr)


# ---------------------------------------------------------------------------------------
#                                   FUNCTIONS (Debugging)
# ---------------------------------------------------------------------------------------
def enablePageHeap(programName):
    if sys.platform == 'win32':
        keyHandle = _winreg.CreateKey(_winreg.HKEY_LOCAL_MACHINE, 'SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Image File Execution Options\\' + programName)
        _winreg.SetValueEx(keyHandle, "PageHeapFlags", 0, _winreg.REG_SZ, "0x3")
        _winreg.SetValueEx(keyHandle, "GlobalFlag", 0, _winreg.REG_SZ, "0x02000000")
        _winreg.FlushKey(keyHandle)
        _winreg.CloseKey(keyHandle)


def disablePageHeap(programName):
    if sys.platform == 'win32':
        _winreg.DeleteKey(_winreg.HKEY_LOCAL_MACHINE, 'SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Image File Execution Options\\' + programName)


# ---------------------------------------------------------------------------------------
#                                   FUNCTIONS
# ---------------------------------------------------------------------------------------

def customOpen(filename, mode):
    return open(filename, mode, encoding='latin-1')

# Run an executable and return the return code
def runProgram(fileFullPath):
    returncd = os.system(fileFullPath)
    return returncd

def file_exists(pathname):
    result = 1
    try:
        file = open(pathname, "r")
        file.close()
    except:
        result = None
    return result


def unixToDos(fileFullPath):
    if file_exists(fileFullPath):

        infile = open(fileFullPath, 'r')
        fixedFileContent = ""

        for line in infile:
            line = line.rstrip() + '\r\n'
            fixedFileContent += line
        infile.close()

        outfile = open(fileFullPath, 'w')
        outfile.write(fixedFileContent)
        outfile.close()


def escapeString(s):
    bytesData = str(s).encode().decode('latin-1')
    bytesData = xml.sax.saxutils.escape(bytesData)
    return bytesData


def dartOutput(title, msg, hasError=False):
    sys.stdout.write(f'<DartMeasurement name="{title}" type="text/string">\n')
    msg = escapeString(msg)
    if hasError:
        sys.stdout.write(f'<font color="#FF0000"> {msg} </font>')
    else:
        sys.stdout.write(msg)
    sys.stdout.write("</DartMeasurement>")

def dartOutputErrorWithUpdate(title, msg, path, content, action):
    name = "TestOutputWithUpdate"
    isCustomBranch = branch_name not in standard_branches
    msg = escapeString(msg)
    outputId = uploadOutput(path, content, action)
    sys.stdout.write(f'<DartMeasurement name="{name}" type="text/json">\n')
    json_msg = { "title" : title, "outputId" : outputId, "data" : msg, "isCustomBranch" : isCustomBranch }
    sys.stdout.write(json.dumps(json_msg)+"\n")
    sys.stdout.write("</DartMeasurement>")

def dartOutputImageAndError(filename: str, generatedId : int, referenceId : int, diffId : int, errorCode: bool, outputError: str):
  name = "TestImages"
  isCustomBranch = branch_name not in standard_branches
  sys.stdout.write(f'<DartMeasurement name="{name}" type="text/json">\n')
  json_msg = { "filename" : filename, "generatedId" : generatedId, "referenceId" : referenceId, "diffId" : diffId, "errorCode" : errorCode, "isCustomBranch" : isCustomBranch, "outputError" : outputError }
  sys.stdout.write(json.dumps(json_msg)+"\n")
  sys.stdout.write("</DartMeasurement>\n")

#show the reference image on standard branches
def dartOutputReferenceImage(filename: str, referenceId: int):
  name = "TestImagesOnlyReference"
  sys.stdout.write(f'<DartMeasurement name="{name}" type="text/json">\n')
  json_msg = { "filename" : filename, "referenceId" : referenceId }
  sys.stdout.write(json.dumps(json_msg)+"\n")
  sys.stdout.write("</DartMeasurement>\n")

def dartOutputCommitButton(referencesDir):
  if branch_name in standard_branches:
    hostname = getHostname(referencesDir)
    sys.stdout.write('<DartMeasurement name="CommitUpdate" type="text/json">\n')
    json_msg = { "testPath" : referencesDir.split(hostname)[1], "branchName" : branch_name, "hostName" : hostname }
    sys.stdout.write(json.dumps(json_msg)+"\n")
    sys.stdout.write("</DartMeasurement>\n")

def dartOutputEnv():
  sys.stdout.write('<DartMeasurement name="Environment" type="text/string">\n')
  for k, v in os.environ.items():
      sys.stdout.write(k+"="+v+"\n")
  sys.stdout.write("</DartMeasurement>\n")

def dartOutputGitSha(gitsha):
  sys.stdout.write('<DartMeasurement name="GitSha" type="text/string">\n'+gitsha+"</DartMeasurement>\n")

def dartOutputFile(title, filename):
  sys.stdout.write(f'<DartMeasurement name="{title}" type="text/json">\n')
  data=''
  inFile = customOpen(filename, "r")
  line = inFile.readline()
  while line:
    line = escapeString(line)
    data += line
    line = inFile.readline()
  inFile.close()
  json_msg = { "data" : data }
  sys.stdout.write(json.dumps(json_msg)+"\n")
  sys.stdout.write("</DartMeasurement>\n")

def dartOutputFileExpand(title, filename, keepComment=True):
  sys.stdout.write(f'<DartMeasurement name="{title}" type="text/string">\n')
  inFile = customOpen(filename, "r")
  line = inFile.readline()
  while line:
    line = escapeString(line)
    if keepComment or ( not line.startswith('#') and not line.strip() == ''):
      sys.stdout.write(line)
    line = inFile.readline()
  inFile.close()
  sys.stdout.write("</DartMeasurement>\n")


def getImageDiffMetric(comparaisonTool, imageFilenameA, imageFilenameB, imageDiffFilename):
    #Return the first value of the compare output
    cmd = comparaisonTool + ' -o "' + imageDiffFilename + '" "' + imageFilenameA + '" "' + imageFilenameB + '"'
    standard, error, retcode = getProgramOutput(cmd)
    if str.find(error, "has different size than") >= 0:
        return -2.0
    try:
        return float(standard.split("\t")[0])
    except:
        return -1.0


def getProgramOutput(cmdLine):
    """Run an executable and return the return standard output and error"""
    cmdArgs = shlex.split(cmdLine)
    try:
        p = None
        p = subprocess.Popen(cmdArgs, stderr=subprocess.PIPE, stdout=subprocess.PIPE, text=True, encoding='latin-1')
        (stdout, stderr) = p.communicate()
        retcode = p.returncode
    except Exception as ex:
        print("ERROR: Cannot execute command line " + cmdLine)
        print("Check that executable exists")
        raise ex

    return stdout, stderr, retcode


def getProgramStandardOutput(fileFullPath, outFileName, errFileName):
    """Run an executable and return the return standard output"""
    standard, error, retcode = getProgramOutput(fileFullPath)
    # write the result in a file. This file may contain some bench to analyse next.
    file = customOpen(outFileName, "w")
    file.write(standard)
    file.close()
    file = customOpen(errFileName, "w")
    file.write(error)
    file.close()
    return retcode


def getIvReport(pathToIvReport):
    """Return result of IvReport as a string
    Launch IvReport process only the first time in order to create IvReport.txt
	"""
    if not os.path.exists(pathToIvReport + "IvReport.txt"):
        subprocess.call([pathToIvReport + "IvReport", pathToIvReport + "IvReport.txt"])

    try:
        with open(pathToIvReport + "IvReport.txt") as inputFileHandle:
            return inputFileHandle.read()

    except IOError:
        return subprocess.Popen([pathToIvReport + "IvReport", '/?'], stdout=subprocess.PIPE, text=True).communicate()[0]


# ---------------------------------------------------------------------------------------
def getRefFileName ( name, referencesDir):
    if not (os.path.exists(referencesDir)):
        print("Missing ref because directory " + referencesDir + " does not exist")
        return ""

    if not os.path.isfile(os.path.join(referencesDir, name)):
        print("Missing ref for file " + name + " in directory " + referencesDir)
        return ""

    return name

def getHostname(filepath):
  hostname = socket.gethostname().lower().split(".")[0]
  #Check lower/upper hostname
  if filepath.find(hostname) == -1 and filepath.find(hostname.upper()) != -1:
    hostname = hostname.upper()
  return hostname

def returnFromCdashServer(url, values, funcName) -> int:
  try:
    r = requests.post(url = url, data = values, verify=ssl_cert_path)
    if r.status_code != 200 or not r.json()['success']:
      print("Error in " + funcName + "()\nStatus Code: " + str(r.status_code) + "\nError Message: " + r.json()['message'])
      return -1
    return int(r.json()['data'])
  except Exception as e:
    print("Error in " + funcName + ": " + str(e) + r.text)
    return -1

def uploadOutput(reference, content, action) -> int:
  hostname = getHostname(reference)
  path = reference.split(hostname)[1]
  url = cdash_server_api + '/uploadReferenceOutput.php'
  values = {'path': path, 'hostname': hostname, 'branch': branch_name, 'action': action, 'data' : content}
  return returnFromCdashServer(url, values, sys._getframe().f_code.co_name)

def uploadReferencePath(reference) -> int:
  hostname = getHostname(reference)
  path = reference.split(hostname)[1]
  url = cdash_server_api + '/uploadReferencePath.php'
  values = {'path': path, 'hostname': hostname, 'branch': branch_name}
  return returnFromCdashServer(url, values, sys._getframe().f_code.co_name)

def uploadReferenceImage(reference) -> int:
  with open(reference, "rb") as image_file:
    encoded_string = base64.b64encode(image_file.read())
    url = cdash_server_api + '/uploadReferenceImageCustomBranch.php'
    values = {'data': encoded_string}
    return returnFromCdashServer(url, values, sys._getframe().f_code.co_name)

def uploadGeneratedImage(generated, referenceID) -> int:
  with open(generated, "rb") as image_file:
    encoded_string = base64.b64encode(image_file.read())
    url = cdash_server_api + '/uploadGeneratedImage.php'
    values = {'data': encoded_string, 'id': referenceID}
    return returnFromCdashServer(url, values, sys._getframe().f_code.co_name)


def uploadDiffImage(diff, generatedID) -> int:
  with open(diff, "rb") as image_file:
    encoded_string = base64.b64encode(image_file.read())
    url = cdash_server_api + '/uploadDiffImage.php'
    values = {'data': encoded_string, 'id': generatedID}
    return returnFromCdashServer(url, values, sys._getframe().f_code.co_name)


def compareImage(name, outputDir, referencesDir, testConf=""):
    maxDiffValue = 0.001
    filename = os.path.basename(name)
    generatedImage = outputDir + "/" + filename

    if testConf != "":
        testName = str.replace(filename, testConf + "_", "")
    else:
        testName = str.replace(filename, testConf, "")

    RefFilename = getRefFileName(testName, referencesDir)

    referenceImagePath = referencesDir + "/" + RefFilename

    diffImage = outputDir + "/diff_" + filename

    hasError = False

    outputError = ""

    if file_exists(referenceImagePath):
        diffValue = getImageDiffMetric(imageCompTool, generatedImage, referenceImagePath, diffImage)
        # We don't want to try to upload the diff file if it doesn't exist!
        if (diffValue > maxDiffValue) or (diffValue < 0):
            if diffValue == -2.0:
                outputError = "Generated Image " + generatedImage + "\nGenerated image is too different. No diff output generated."
                diffImage = os.getenv("OIVHOME") + "/source/cmake/diff_not_generated.png"
            else:
                outputError = "Generated Image " + generatedImage +  "\nGenerated image is different: " + str(diffValue)
            hasError = True
    else:
        # No diff image generated, just upload the default one to avoid error on upload
        diffImage = os.getenv("OIVHOME") + "/source/cmake/diff_not_generated.png"
        outputError = "Reference image " + RefFilename + "\nReference file is missing: " + referencesDir
        hasError = True
        referenceImagePath = referencesDir + "/" + filename

    # when there is no error, do not upload the images at all
    # this saves considerable time
    # still show the reference image on standard branches though, as it is quite useful
    if hasError == False:
        if branch_name in standard_branches:
            referenceId = uploadReferencePath(referenceImagePath)
            dartOutputReferenceImage(filename, referenceId)
        return 0

    if branch_name in standard_branches:
        # the path is uploaded to the server because the branch is checkout on the CDASH server
        referenceId = uploadReferencePath(referenceImagePath)
        generatedId = uploadGeneratedImage(generatedImage, referenceId)
        diffId = uploadDiffImage(diffImage, generatedId)
        dartOutputImageAndError(filename, generatedId, referenceId, diffId, hasError, outputError)
    else:
        # the image is uploaded to the server because the branch is not checkout on the CDASH server
        if file_exists(referenceImagePath):
            referenceId = uploadReferenceImage(referenceImagePath)
        else:
            # Reference does not exist do not try to upload it or it will fail
            # Just upload the fake image so we have an error report
            referenceId = uploadReferenceImage(diffImage)
        generatedId = uploadGeneratedImage(generatedImage, referenceId)
        diffId = uploadDiffImage(diffImage, generatedId)
        dartOutputImageAndError(filename, generatedId, referenceId, diffId, hasError, outputError)

    return hasError

def generateStdDiff(referencesDir, filename, outputDir, outName, refName=""):
    if refName == "":
        refName = filename
    # remove OpenGL Warning messages from Chromium on VirtualBox
    lines = customOpen(outputDir + "/" + filename + outName, 'r').readlines()
    tmpLine = []
    for line in lines:
        if len(re.sub('OpenGL Warning:.*\n', '', line, re.S)) > 0:
            tmpLine.append(line)
    customOpen(outputDir + "/" + filename + outName, 'w').writelines(tmpLine)

    p = None
    p = subprocess.Popen(["diff", "-uaBw", referencesDir + "/" + refName, outputDir + "/" + filename + outName], stdout=subprocess.PIPE, text=True)
    return p.stdout.read()

def generateBenchReport(ref, gen, new, warnLimit, errorLimit):
    inFile1 = customOpen(gen, "r")
    inFile2 = customOpen(ref, "r")
    outFile = customOpen(new, "w")
    line1 = inFile1.readline()
    line2 = inFile2.readline()

    errorCode = 0

    doDiff = 0
    index_l1 = 0
    index_l2 = 0
    index2_l1 = -1
    index2_l2 = -1
    while line1:
        while index_l1 != -1:
            line1 = line1[index2_l1 + 1:]
            line2 = line2[index2_l2 + 1:]
            index_l1 = str.find(line1, "<")
            index2_l1 = str.find(line1, ">")
            if index_l1 != -1:
                doDiff = 1
                valstr1 = line1[index_l1 + 1:index2_l1]
                try:
                    value1 = float(valstr1)
                except:
                    outFile.write("<font color=\"#FF0000\">Bad value in output</font>")
                    value1 = 0
                index_l2 = str.find(line2, "<")
                index2_l2 = str.find(line2, ">")
                if index_l2 == -1 or index2_l2 == -1:
                    # file mismatch
                    outFile.write("<font color=\"#FF0000\">Output mismatch starting this line</font>")
                    return 3
                valstr2 = line2[index_l2 + 1:index2_l2]
                try:
                    value2 = float(valstr2)
                except:
                    outFile.write("<font color=\"#FF0000\">Bad value in output</font>")
                    value2 = 0

                if value2 != 0.0:
                    diff = int(((value1 - value2) * 100) / value2)
                else:
                    diff = value1
                if diff > 0:
                    sign = "+"
                else:
                    sign = ""
                errorColorStart = "<font color=\"#00C832\">"
                errorColorEnd = "</font>"
                if diff > warnLimit or diff < -warnLimit:
                    # warning limit is exceeded... change write color
                    errorColorStart = "<font color=\"#FF7700\">"
                if diff > errorLimit or diff < -errorLimit:
                    # error limit is exceeded... set errorcode and change write color
                    ### Disable errorCode = 1
                    errorColorStart = "<font color=\"#FF0000\">"
                index3 = str.find(line1, "<")
                if (line1[0:index_l1] != line2[0:index_l2]):
                    #file mismatch
                    outFile.write("<font color=\"#FF0000\">Output mismatch starting this line</font>")
                    return 3
                if (index3 == -1):
                    outFile.write(line1[0:index_l1] + errorColorStart + "[" + valstr1 + " | " + valstr2 + " | <b>" + sign + str(diff) + "%</b>]" + errorColorEnd + line1[index2_l1:])
                else:
                    outFile.write(line1[0:index_l1] + errorColorStart + "[" + valstr1 + " | " + valstr2 + " | <b>" + sign + str(diff) + "%</b>]" + errorColorEnd)
        if index_l1 == -1:
            outFile.write(line1)

        # go to next line
        line1 = inFile1.readline().replace('\r','\n')
        while(line1.startswith('\n')):
            line1 = inFile1.readline().replace('\r','\n')
        line2 = inFile2.readline().replace('\r','\n')
        while(line2.startswith('\n')):
            line2 = inFile2.readline().replace('\r','\n')
        index_l1 = str.find(line1, "<")
        index_l2 = str.find(line2, "<")
        index2_l1 = -1
        index2_l2 = -1

    outFile.close()
    inFile1.close()
    inFile2.close()
    if doDiff == 0:
        return 99
    return errorCode


def setCdashServer(cdash_servername):
    global cdash_server
    cdash_server = cdash_servername


def setBranchName(branchName):
    print("Set Branch to " + branchName)
    global branch_name
    branch_name = branchName

# ---------------------------------------------------------------------------------------
def compareOutput(testName,  outputDir, referencesDir, warnLimit, errorLimit, testConf=""):

    errorCode = 0

    filename = testName
    if testConf != "":
        filename = testConf + "_" + filename

    ############# stdout comparison #############
    outGenFilename = outputDir + "/" + filename + "_stdout.log"
    refName = getRefFileName(testName + "_stdout.log", referencesDir)
    if refName == "":
        outRefFilename = referencesDir + "/" + testName + "_stdout.log"
    else:
        outRefFilename = referencesDir + "/" + refName

    outGenFile = file_exists(outGenFilename)
    outRefFile = file_exists(outRefFilename)

    if (outRefFile == None and outGenFile != None):
        if (os.path.getsize(outGenFilename) != 0):
            with customOpen(outGenFilename, "r") as content_file:
                outGenContent = content_file.read().replace('\n', '')
            dartOutputErrorWithUpdate("Standard Output", "A stdout file generated, but no stdout reference exists. Please update stdout reference:\n\n" + outGenContent, outRefFilename, outGenContent, "r")
            errorCode = 1
    elif (outRefFile != None and outGenFile == None):
        dartOutputErrorWithUpdate("Standard Output", "No stdout file generated whereas a stdout reference is existing. Please check demo outputs.", outRefFilename, "", "d")
        errorCode = 1
    elif (outRefFile == None and outGenFile == None):
        dartOutput("Standard Output", "OK: no stdout file generated and no stdout reference existing.", hasError=True)
    else:
        with customOpen(outGenFilename, "r") as content_file:
            outGenContent = content_file.read()
        resultbench = generateBenchReport(outRefFilename, outGenFilename, outputDir + "/" + filename + "_benchresult.log", warnLimit, errorLimit)
        benchresultpath = outputDir + "/" + filename + "_benchresult.log"
        if file_exists(benchresultpath) and os.path.getsize(benchresultpath) > 0:
            dartOutputFile("Standard Output Bench Result\n[ gen | ref | % ]", outputDir + "/" + filename + "_benchresult.log")
        if resultbench == 3:
            dartOutput("Standard Output", "Output formatting does not match in files", hasError=True)
            resultStdout = generateStdDiff(referencesDir, filename, outputDir, "_stdout.log", refName)
            if len(resultStdout) > 0:
                dartOutputErrorWithUpdate("Standard Output Diff Result", resultStdout, outRefFilename, outGenContent, "r")
            errorCode = 1
        elif resultbench == 1:
            dartOutputErrorWithUpdate("Standard Output", "Output bench reports significant differences. Please refer to bench result section.", outRefFilename, outGenContent, "r")
            errorCode = 1
        elif resultbench == 99:
            # no bench performed
            resultStdout = generateStdDiff(referencesDir, filename, outputDir, "_stdout.log", refName)
            if len(resultStdout) > 0:
                dartOutputErrorWithUpdate("Standard Output Diff Result", resultStdout, outRefFilename, outGenContent, "r")
                errorCode = 1
        if resultbench == 0:
            dartOutput("Standard Output", "OK")

    ############# stderr comparison #############
    errGenFilename = outputDir + "/" + filename + "_stderr.log"
    refName = getRefFileName(testName + "_stderr.log", referencesDir)
    if refName == "":
        errRefFilename = referencesDir + "/" + testName + "_stderr.log"
    else:
        errRefFilename = referencesDir + "/" + refName

    errGenFile = file_exists(errGenFilename)
    errRefFile = file_exists(errRefFilename)

    if (errRefFile == None and errGenFile != None):
        if (os.path.getsize(errGenFilename) != 0):
            with customOpen(errGenFilename, "r") as content_file:
                errGenContent = content_file.read()
            dartOutputErrorWithUpdate("Standard Error", "A stderr file generated, but no stderr reference exists. Please update stderr reference:\n\n" + errGenContent, errRefFilename, errGenContent, "r")
            errorCode = 1
    elif (errRefFile != None and errGenFile == None):
        dartOutputErrorWithUpdate("Standard Error", "No stderr file generated whereas a stderr reference is existing. Please check demo outputs." + " erreurGen " + errGenFilename + " erreurRef " + errRefFilename, errRefFilename, "", "d")
        errorCode = 1
    elif (errRefFile == None and errGenFile == None):
        dartOutput("Standard Error", "OK: no file generated and no reference existing.")
    else:
        with customOpen(errGenFilename, "r") as content_file:
            errGenContent = content_file.read()
        # the 2 files are existing. compare them
        resultStderr = generateStdDiff(referencesDir, filename, outputDir, "_stderr.log", refName)
        if len(resultStderr) > 0:
            dartOutputErrorWithUpdate("Error output difference error", resultStderr, errRefFilename, errGenContent, "r")
            errorCode = 1

    ############# stack output #############
    stackFile = outputDir + "/" + filename + "_stack.log"
    if file_exists(stackFile):
        dartOutputFile("Stack trace", stackFile)

    return errorCode
