#!/usr/bin/env python
import os
import sys
import fileinput
import re
import argparse

# Add include files
includeFile = [
    "//!oiv_include <Inventor/oivShaderState.h>\n",
    "\n"
]

vertexIncludeFile = [
    "//!oiv_include <Inventor/oivShapeAttribute.h>\n"
]

# Define dictionary for shader state
shader_state = {
    # Transform
    "ftransform()": "OivModelViewProjectionMatrix() * OivVertexPosition()",
    "gl_ModelViewMatrixInverse": "OivModelViewMatrixInverse()",
    "gl_ModelViewMatrix": "OivModelViewMatrix()",
    "gl_NormalMatrix": "OivNormalMatrix()",
    "gl_ProjectionMatrix": "OivProjectionMatrix()",
    "gl_ModelViewProjectionMatrix": "OivModelViewProjectionMatrix()",
    # Material
    "gl_FrontMaterial.ambient": "OivFrontMaterialAmbient()",
    "gl_FrontMaterial.diffuse": "OivFrontMaterialDiffuse()",
    "gl_FrontMaterial.specular": "OivFrontMaterialSpecular()",
    "gl_FrontMaterial.shininess": "OivFrontMaterialShininess()",
    "gl_FrontMaterial.emission": "OivFrontMaterialEmissive()",
    "gl_BackMaterial.ambient": "OivBackMaterialAmbient()",
    "gl_BackMaterial.diffuse": "OivBackMaterialDiffuse()",
    "gl_BackMaterial.specular": "OivBackMaterialSpecular()",
    "gl_BackMaterial.shininess": "OivBackMaterialShininess()",
    "gl_BackMaterial.emission": "OivBackMaterialEmissive()",
    "gl_LightModel.ambient": "OivGlobalAmbient()",
    "gl_FrontLightModelProduct.sceneColor": "OivFrontSceneColor()",
}

# Define dictionary for shape attributes
shape_attribute = {
    "gl_Vertex": "OivVertexPosition()",
    "gl_Normal": "OivVertexNormal()",
    "gl_Color": "OivVertexColor()"
}

# Define changes in API since OIV9
api_change = {
    "OivGetShapeInstanceID": "OivInstanceID",
    "OivGetShapeInstanceTranslation": "OivInstanceTranslation",
    "OivGetShapeInstanceScale": "OivInstanceScale",
    "OivGetShapeInstanceRotation": "OivInstanceRotation",
    "OivGetShapeInstanceMatrix": "OivInstanceMatrix"
}

shape_attributes_regex_replacements = {
    # Attribute
    "gl_MultiTexCoord([0-7])": "OivVertexTextureCoordinate(NUM)",
    # Transform
    "gl_TextureMatrix\[(\w*)\]": "OivTextureMatrix(NUM)",
}

# Define regex-based dictionary for indexed patterns
# e.g. gl_LightSource is an array from 0 to 7
# In order to not enumerate all cases, a regex to find whatever indices of gl_LightSource
# is defined such as: gl_LightSource\[(\d+)\] where (\d+) represents the varying pattern.
# Note that pattern must be defined between parenthesis for reusing the found number.
api_regex_replacements = {
    # Lighting
    "gl_LightSource\[(\w*)\].position": "OivLightSourcePosition(NUM)",
#    "gl_LightSource\[(\w*)\].halfVector": "OivLightSourceHalfVector(NUM)",
    "gl_LightSource\[(\w*)\].spotDirection": "OivLightSourceSpotDirection(NUM)",
    "gl_LightSource\[(\w*)\].spotExponent": "OivLightSourceSpotExponent(NUM)",
    "gl_LightSource\[(\w*)\].spotCutoff": "OivLightSourceSpotCutoff(NUM)",
    "gl_LightSource\[(\w*)\].spotCosCutoff": "OivLightSourceSpotCosCutoff(NUM)",
    "gl_LightSource\[(\w*)\].ambient": "OivLightSourceAmbient(NUM)",
    "gl_LightSource\[(\w*)\].diffuse": "OivLightSourceDiffuse(NUM)",
    "gl_LightSource\[(\w*)\].specular": "OivLightSourceSpecular(NUM)",
    "gl_LightSource\[(\w*)\].constantAttenuation": "OivLightSourceConstantAttenuation(NUM)",
    "gl_LightSource\[(\w*)\].linearAttenuation": "OivLightSourceLinearAttenuation(NUM)",
    "gl_LightSource\[(\w*)\].quadraticAttenuation": "OivLightSourceQuadraticAttenuation(NUM)",
    # LightProduct
    "gl_FrontLightProduct\[(\w*)\].ambient": "OivFrontLightProductAmbient(NUM)",
    "gl_FrontLightProduct\[(\w*)\].diffuse": "OivFrontLightProductDiffuse(NUM)",
    "gl_FrontLightProduct\[(\w*)\].specular": "OivFrontLightProductSpecular(NUM)",
    "gl_BackLightProduct\[(\w*)\].ambient": "OivBackLightProductAmbient(NUM)",
    "gl_BackLightProduct\[(\w*)\].diffuse": "OivBackLightProductDiffuse(NUM)",
    "gl_BackLightProduct\[(\w*)\].specular": "OivBackLightProductSpecular(NUM)",
    # Clip planes
    "gl_ClipPlane\[(\w*)\]": "OivClipPlane(NUM)",
    "gl_ObjectPlaneS\[(\w*)\]": "OivObjectPlaneS(NUM)",
    "gl_ObjectPlaneT\[(\w*)\]": "OivObjectPlaneT(NUM)",
    "gl_ObjectPlaneR\[(\w*)\]": "OivObjectPlaneR(NUM)",
    "gl_ObjectPlaneQ\[(\w*)\]": "OivObjectPlaneQ(NUM)"
}

legacy_builtin_varyings = [
    "gl_FrontColor",
    "gl_BackColor",
    "gl_FrontSecondaryColor",
    "gl_BackSecondaryColor",
    "gl_TexCoord",
    "gl_FogFragCoord",
    "gl_FogCoord",
]

builtin_varyings_include = "//!oiv_include <Inventor/oivShaderVariables.h>\n"

varying_regex_replacements = {
    "gl_FrontColor\s*=\s*(?P<VALUE>.*);": "OivSetFrontColor(VALUE);",
    "gl_BackColor\s*=\s*(?P<VALUE>.*);": "OivSetBackColor(VALUE);",
    "gl_FrontSecondaryColor\s*=\s*(?P<VALUE>.*);": "OivSetFrontSecondaryColor(VALUE);",
    "gl_BackSecondaryColor\s*=\s*(?P<VALUE>.*);": "OivSetBackSecondaryColor(VALUE);",
    "gl_FogFragCoord\s*=\s*(?P<VALUE>.*);": "OivSetFogFragCoord(VALUE);",
    "gl_TexCoord\[(?P<TEX_INDEX>.*)\]\s*=\s*(?P<VALUE>.*);": "OivSetTexCoord(TEX_INDEX, VALUE);",
    "gl_FogCoord": "OivFogFragCoord()",
}

varying_array_regex_replacements = {
    "gl_out\[\s*gl_InvocationID\s*\]\.gl_FrontColor\s*=\s*(?P<VALUE>.*);": "OivSetFrontColor(VALUE);",
    "gl_out\[\s*gl_InvocationID\s*\]\.gl_BackColor\s*=\s*(?P<VALUE>.*);": "OivSetBackColor(VALUE);",
    "gl_out\[\s*gl_InvocationID\s*\]\.gl_FrontSecondaryColor\s*=\s*(?P<VALUE>.*);": "OivSetFrontSecondaryColor(VALUE);",
    "gl_out\[\s*gl_InvocationID\s*\]\.gl_BackSecondaryColor\s*=\s*(?P<VALUE>.*);": "OivSetBackSecondaryColor(VALUE);",
    "gl_out\[\s*gl_InvocationID\s*\]\.gl_FogFragCoord\s*=\s*(?P<VALUE>.*);": "OivSetFogFragCoord(VALUE);",
    "gl_out\[\s*gl_InvocationID\s*\]\.gl_TexCoord\[(?P<TEX_INDEX>.*)\]\s*=\s*(?P<VALUE>.*);": "OivSetTexCoord(TEX_INDEX, VALUE);",

    "gl_in\[(?P<IN_INDEX>.*)\]\.gl_FrontColor": "OivFrontColor(IN_INDEX)",
    "gl_in\[(?P<IN_INDEX>.*)\]\.gl_BackColor": "OivBackColor(IN_INDEX)",
    "gl_in\[(?P<IN_INDEX>.*)\]\.gl_FrontSecondaryColor": "OivFrontSecondaryColor(IN_INDEX)",
    "gl_in\[(?P<IN_INDEX>.*)\]\.gl_BackSecondaryColor": "OivSetBackSecondaryColor(IN_INDEX)",
    "gl_in\[(?P<IN_INDEX>.*)\]\.gl_FogFragCoord": "OivFogFragCoord(IN_INDEX)",
    "gl_in\[(?P<IN_INDEX>[^\]]*)\]\.gl_TexCoord\[(?P<TEX_INDEX>[^\]]*)\]": "OivTexCoord(IN_INDEX, TEX_INDEX)",
}

fragment_shader_replacements = {
    "gl_FogFragCoord": "OivFogFragCoord()",
    "gl_Color": "OivFragmentColor()",
    "gl_SecondaryColor": "OivFragmentSecondaryColor()",
    "gl_TexCoord\[(?P<TEX_INDEX>[^\]]*)\]": "OivFragmentTexCoord(TEX_INDEX)",
    "gl_FragColor\s*=\s*(?P<VALUE>.*);": "OivFragmentOutput(VALUE);",
    "gl_FragData\[(?P<BUF_INDEX>.*)\]\s*=\s*(?P<VALUE>.*);": "OivFragmentOutput(BUF_INDEX, VALUE);"
}

def replace_strings_in_line(line, replacement_map):
    ret_line = "%s" % line
    for original, replacement in replacement_map.items():
        m = re.search(original, ret_line)
        while m:
            replacement_str = "%s" % replacement
            for group_name, group_value in m.groupdict().items():
                replacement_str = re.sub(group_name, group_value, replacement_str)
            ret_line = ret_line.replace(m.group(0), replacement_str)
            m = re.search(original, ret_line)
    return ret_line


# run converter on a filename
def convert(filename):
    input_file = open(filename, 'r')
    output_file = open(filename + ".new", 'w')

    input_file.seek(0, os.SEEK_SET)

    file_has_shader_state = False
    file_has_shape_attributes = False
    file_has_builtin_varying = False
    file_has_gl_color = False
    file_is_vertex = re.match("vert", filename, re.IGNORECASE) or re.match("vtx", filename, re.IGNORECASE)
    file_is_fragment = re.match("frag", filename, re.IGNORECASE)

    # list of built in we can find only in vertex shader.
    # this will help us to decide if file is a vertex shader or not
    vertex_specific_builtin = {"gl_FrontColor", "gl_BackColor", "gl_Normal", "gl_Vertex", "gl_Position"}
    fragment_specific_builtin = {"gl_SecondaryColor", "gl_FragCoord", "gl_FrontFacing", "gl_PointCoord", "gl_FragColor", "gl_FragData"}

    alreadyIncluded = []
    # do a first pass to determine what kind of changes will be done
    # This is used to add the correct include headers, and only add them when necessary
    for line in input_file:
        for inc in includeFile + vertexIncludeFile:
            if inc in line:
                alreadyIncluded.append(inc)

        for key in shader_state.keys():
            if key in line:
                file_has_shader_state = True

        for key in api_regex_replacements.keys():
            if re.search(key, line) is not None:
                file_has_shader_state = True

        for key in shape_attribute.keys():
            # gl_Color may appears in a fragment shader, so don't use it to determine if shader has shapeAttribute
            if "gl_Color" not in key and key in line:
                file_has_shape_attributes = True

        for key in shape_attributes_regex_replacements.keys():
            if re.search(key, line) is not None:
                file_has_shape_attributes = True

        for key in legacy_builtin_varyings:
            if re.search(key, line) is not None:
                file_has_builtin_varying = True

        # See list-comprehensions in python
        # basically, this do: [ expression for item in list ]
        # this check if any of the item in vertex_specific_builtin is present in line
        # and none of the items in fragment_specific_builtin are present...
        file_is_vertex = file_is_vertex or (any([builtin in line for builtin in vertex_specific_builtin]) and not any([builtin in line for builtin in fragment_specific_builtin]))
        file_is_fragment = file_is_fragment or (any([builtin in line for builtin in fragment_specific_builtin]) and not any([builtin in line for builtin in vertex_specific_builtin]))

    input_file.seek(0, os.SEEK_SET)

    # we find a "gl_Color" but we don't know if it is used as shape attribute or not.
    # If file is a vertex shader, then it is a shape attribute
    if "gl_Color" in key and file_is_vertex:
        file_has_shape_attributes = True

    # to make sure that #version is on the first line
    first_line = input_file.readline()
    m = re.search("^\#version", first_line)
    if m:
        output_file.write(first_line)
    else:
        input_file.seek(0, os.SEEK_SET)

    # add vertex include files
    if file_has_shape_attributes:
        for inc in vertexIncludeFile:
            if inc not in alreadyIncluded:
                output_file.write(inc)

    # add common include files
    if file_has_shader_state or file_has_shape_attributes:
        for inc in includeFile:
            if inc not in alreadyIncluded:
                output_file.write(inc)

    # add builtin varying include
    if file_is_fragment or file_has_builtin_varying:
        output_file.write(builtin_varyings_include)

    # parse file and replace uses of shader state/state attributes/...
    for line in input_file:

        for key in shader_state.keys():
            line = line.replace(key, shader_state[key])

        for key in api_change.keys():
            line = line.replace(key, api_change[key])

        for key in api_regex_replacements.keys():
            m = re.search(key, line)
            while m:
                t = re.sub(r'NUM', m.group(1), api_regex_replacements[key])
                line = line.replace(m.group(0), t)
                m = re.search(key, line)

        # Shape Attributes
        if file_has_shape_attributes:
            for key in shape_attribute.keys():
                line = re.sub(r"\b"+key+r"\b", shape_attribute[key], line)

        for key in shape_attributes_regex_replacements.keys():
            m = re.search(key, line)
            while m:
                t = re.sub(r'NUM', m.group(1), shape_attributes_regex_replacements[key])
                line = line.replace(m.group(0), t)
                m = re.search(key, line)

        # Fragment Builtin Varyings
        if file_is_fragment:
            line = replace_strings_in_line(line, fragment_shader_replacements)

        # Builtin Varyings
        if file_has_builtin_varying:
            line = replace_strings_in_line(line, varying_array_regex_replacements)
            line = replace_strings_in_line(line, varying_regex_replacements)

        output_file.write(line)

    input_file.close()
    output_file.close()


def convert_file_with_backup(filename, remove_original):
    print("Convert file {}...".format(filename))
    backupname = filename + ".origin"
    convert(filename)
    if not remove_original:
        i = 0
        while os.access(backupname, os.R_OK):
            i = i + 1
            backupname = filename + "." + i + ".origin"
        os.rename(filename, backupname)
        print("Original file has been renamed to {}".format(backupname))
    else:
        os.remove(filename)
    os.rename(filename + ".new", filename)


# main func
if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Convert shaders from OIV 9 and older to the new OIV 10 format')
    parser.add_argument(
        'filename',
        help='The GLSL shader file to convert or a folder in which to convert all *.glsl files'
    )
    parser.add_argument('--remove_original', dest='remove_original', action='store_true', help="Do not keep the original file as *.glsl.original")

    args = parser.parse_args()

    if not os.path.exists(args.filename):
        print("filename " + args.filename + " does not exist !")
    else:
        print( """
**********************************************************************************************************
* IMPORTANT:                                                                                             *
*   This script only replaces legacy OpenGL GLSL uniforms and varyings by                                *
*   OpenInventor Shader API calls. Additional manual modifications might be                              *
*   required in order to make sure your shader files are correctly converted                             *
*   to be supported by OIV 10. Please read the associated ReadMe.txt file and                            *
*   also have a look at the documentation on how to write OIV 10 compliant                               *
*   shaders:                                                                                             *
*   https://developer.openinventor.com/index.php/general-documentation/how-to-write-shaders-with-oiv-10/ *
**********************************************************************************************************
""")
        if os.path.isfile(args.filename):
            convert_file_with_backup(args.filename, args.remove_original)
        elif os.path.isdir(args.filename):
            # iterate over every file in the folder recursively
            for root, subdirs, files in os.walk(args.filename):
                for f in files:
                    # if the current file ends with ".glsl", convert it
                    if f.endswith(".glsl"):
                        convert_file_with_backup(os.path.join(root, f), args.remove_original)
