// Print information about available OpenGL pixel formats
//
// Copyright (C) 1996-2004 Template Graphics Software, Inc.
//
// Notes:
// 1) Build as a console app with setting OGLINFO_CONSOLE
//
// Created: mmh
// Updated: mmh 30-May-96 - Implement OpenGL vendor queries
//          ket  2-Feb-99 - Implement stereo query
//          mmh 14-Jan-00 - Fix call to DescribePixelFormat (some
//                          impls don't allow zero or NULL args).
//                          Added screen size & color depth info.
//                          Fixed HW accel to detect ICD and MCD.
//          mmh 25-Jan-00 - Now shows if useable with window or bitmap
//          mmh 18-Jun-01 - Also list wgl extension strings
//                          Clean up printing of extension strings
//          co  17-Nov-04 - Added display for Floatint point RGBA and pbuffer pixel formats

#include <windows.h>
#include <stdio.h>
#include <Inventor/sys/SoGL.h>
#include <Inventor/devices/SoGLContext.h>

//#define OGLINFO_CONSOLE

int  getOpenGLInfo();
int  getPixFormatInfo();
int printFormat(int format, PIXELFORMATDESCRIPTOR &pfd);
int printFormatARB(int format, int* iResults);
void printMultiString(const char *string, int startCount = 0,
                      int maxWidth = 80, int indent = 4);
const char EOS = '\0';
bool isAvailableMS = FALSE;

#define ERROR_RETURN(funcname) {\
  DWORD error;                  \
  error = GetLastError();       \
  fprintf(stderr, "*** %s failed, error = %d (0x%x)\n", \
          funcname, error, error);                     \
  return 0;                     \
}

#ifndef OGLINFO_CONSOLE
#include <DialogViz/SoDialogVizAll.h>
#include <stdarg.h>
#include <Inventor/STL/string>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoTexture.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoGeometryShader.h>
#include <Inventor/elements/SoTextureUnitElement.h>
#include <Inventor/components/SoGLGraphicDevice.h> 
#include <Inventor/components/SoGLGraphicConfig.h> 
#include <Inventor/SoWinApp.h>
#include <Inventor/nodes/SoShadowGroup.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <Inventor/Win/SoWinRenderArea.h>

void    getOIVInfo(HDC hdc) ;

SoTopLevelDialog *TopLevelDialog ;
SoDialogEditText *OGLGenDialog ;
SoDialogEditText *OGLExtDialog ;
SoDialogEditText *OGLPixelFormDialog ;
SoDialogEditText *OGLIVExtDialog ;

std::string           OutputStream ;
std::string           OGLInfoText ;

/*************************************************************************/
void 
TGSprintf(const char *first, ...)
{
  va_list vaList ;

  va_start(vaList, first);
  char tmpStr[512] ;
  vsprintf(tmpStr, first, vaList) ;
  OutputStream += tmpStr ;
  va_end(vaList);
}

/*************************************************************************/
void
TGSputchar(int c)
{
  OutputStream += c ;
}

/*************************************************************************/
void
TGSputs(char *str)
{
  OutputStream += str ;
  OutputStream += '\n' ;
}

/*************************************************************************/

// Exit
class ExitButtonAuditor : public SoMenuPushButtonAuditor {
public:
  virtual void menuPushButton(SoMenuPushButton * cpt)
  { exit(0) ; }
};

/*************************************************************************/
// Save As
class SaveAsButtonAuditor : public SoMenuFileSelectionAuditor {
public:
  virtual void menuFileSelection(SoMenuFileSelection* cpt)
  {   
    SbString filename =  cpt->fileDirectory.getValue() ;
             filename += "/" ;
             filename += cpt->filename.getValue();
    
    if(!filename) return;
    FILE *fp = fopen(filename.getString(), "w") ;
    fprintf(fp,"%s", OGLInfoText.c_str()) ;
    fclose(fp) ;
 }
};


/*************************************************************************/
SoTopLevelDialog* 
initDialog()
{
  int nHeight = -MulDiv(8, GetDeviceCaps(GetDC(NULL), LOGPIXELSY), 72);
  HFONT myFont = CreateFont(nHeight,                   // nHeight
                            0,                         // nWidth
                            0,                         // nEscapement
                            0,                         // nOrientation
                            FW_NORMAL,                 // nWeight
                            FALSE,                     // bItalic
                            FALSE,                     // bUnderline
                            0,                         // cStrikeOut
                            ANSI_CHARSET,              // nCharSet
                            OUT_DEFAULT_PRECIS,        // nOutPrecision
                            CLIP_DEFAULT_PRECIS,       // nClipPrecision
                            DEFAULT_QUALITY,           // nQuality
                            DEFAULT_PITCH|FF_MODERN,   // nPitchAndFamily
                            "Courier New");

  SoDialogViz::setFont(myFont) ;
  // Save As
  SoMenuFileSelection * saveAsButton = new SoMenuFileSelection;
  saveAsButton->label    = "Save As...";
  saveAsButton->mode     = SoMenuFileSelection::SAVE ;
  saveAsButton->filter.set1Value(0, "*.*");
  saveAsButton->filter.set1Value(1, "Text file");
  saveAsButton->filename = "TGSOGLInfo.txt" ;
  saveAsButton->addAuditor(new SaveAsButtonAuditor()) ;

  // Exit
  SoMenuPushButton * exitButton = new SoMenuPushButton;
  exitButton->label = "Exit";
  exitButton->addAuditor(new ExitButtonAuditor()) ;

  // FILE Popup
  SoMenuPopup * fileMenuPopup = new SoMenuPopup;
  fileMenuPopup->label = "File";
  fileMenuPopup->addChild(saveAsButton);
  fileMenuPopup->addChild(exitButton);

  // Menu Bar
  SoMenuBar *menuBar = new SoMenuBar;
  menuBar->addChild(fileMenuPopup) ;

  // Tab Box
  OGLGenDialog = new SoDialogEditText;
  OGLGenDialog->editable = FALSE;
  OGLGenDialog->multiLine = TRUE;

  OGLExtDialog = new SoDialogEditText;
  OGLExtDialog->editable = FALSE;
  OGLExtDialog->multiLine = TRUE;

  OGLPixelFormDialog = new SoDialogEditText;
  OGLPixelFormDialog->editable = FALSE;
  OGLPixelFormDialog->multiLine = TRUE;

  OGLIVExtDialog = new SoDialogEditText;
  OGLIVExtDialog->editable = FALSE;
  OGLIVExtDialog->multiLine = TRUE;

  SoRowDialog *row1 = new SoRowDialog;
  row1->label = "General";
  row1->addChild(OGLGenDialog) ;
  
  SoRowDialog *row2 = new SoRowDialog;
  row2->label = "Extensions";
  row2->addChild(OGLExtDialog) ;

  SoRowDialog *row3 = new SoRowDialog;
  row3->label = "Pixel Formats";
  row3->addChild(OGLPixelFormDialog) ;

  SoRowDialog *row4 = new SoRowDialog;
  row4->label = "OIV Extensions";
  row4->addChild(OGLIVExtDialog) ;

  SoTabDialog *shaderTabDialog = new SoTabDialog;
  shaderTabDialog->addChild(row1);
  shaderTabDialog->addChild(row2);
  shaderTabDialog->addChild(row3);
  shaderTabDialog->addChild(row4);

  TopLevelDialog = new SoTopLevelDialog;
  TopLevelDialog->label = "OGLInfo" ;
  TopLevelDialog->height = 400 ;
  TopLevelDialog->addChild(menuBar);
  TopLevelDialog->addChild(shaderTabDialog);
  
  return TopLevelDialog;
}

#define printf  TGSprintf
#define putchar TGSputchar
#define puts    TGSputs

#endif // OGLINFO_CONSOLE

/*************************************************************************/
bool
findExtension(const char *extension, const char *extensions)
// Description :
//-------------------------------------------------------------------------
{
  if(!extensions)
    return FALSE ;

	const GLubyte *start;
	GLubyte *where, *terminator;

	where = (GLubyte *) strchr(extension, ' ');
	if (where || *extension == '\0')
		return 0;
	start = (GLubyte*)extensions;
	for (;;) {
		where = (GLubyte *) strstr((const char *) start, extension);
		if (!where)
			break;
		terminator = where + strlen(extension);
		if (where == start || *(where - 1) == ' ') {
			if (*terminator == ' ' || *terminator == '\0') {
				return true;
			}
		}
		start = terminator;
	}
	return false;
}

/*************************************************************************/

/*************************************************************************/
int
main(int argc, char** argv)
// Description :
//-------------------------------------------------------------------------
{
  int rc;
  
#ifndef OGLINFO_CONSOLE
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) 
    exit(1);
  SoDialogViz::init() ;
  SoVolumeRendering::init() ;
  TopLevelDialog = initDialog() ;
#endif

  // OpenGL vendor information
  rc = getOpenGLInfo();

#ifndef OGLINFO_CONSOLE
  TopLevelDialog->buildDialog(myWindow, TRUE);
  TopLevelDialog->show() ;
  SoXt::show(myWindow);
  SoXt::mainLoop();

  TopLevelDialog->close();
  SoXt::finish();
#endif

  return rc;
}

void
parseOpenGL11Capabilities(const char* extensions)
{
  printf("\n_____________________________________________________________\n");
  printf("Compliance OpenGL v1.1\n");
  bool op = true;
  while (1) {
    op ? printf("  Supported:\n") : printf("  Unsupported:\n");
    int i = 0;
    int j = 0;
    if (!op ^ findExtension("GL_EXT_vertex_array", extensions)) { printf("    GL_EXT_vertex_array\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_texture_object", extensions)) { printf("    GL_EXT_texture_object\n"); i++;} j++;
    printf("    => %d%% (%d/%d)\n", (int)((float)i/(float)j * 100.0), i, j);
    if (!op)
      break;
    op = !op;
  }
}

void
parseOpenGL12Capabilities(const char* extensions)
{
  printf("\n_____________________________________________________________\n");
  printf("Compliance OpenGL v1.2\n");
  bool op = true;
  while (1) {
    op ? printf("  Supported:\n") : printf("  Unsupported:\n");
    int i = 0;
    int j = 0;
    if (!op ^ findExtension("GL_EXT_bgra", extensions)) { printf("    GL_EXT_bgra\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_blend_color", extensions)) { printf("    GL_EXT_blend_color\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_blend_minmax", extensions)) { printf("    GL_EXT_blend_minmax\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_blend_subtract", extensions)) { printf("    GL_EXT_blend_substract\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_draw_range_elements", extensions)) { printf("    GL_EXT_draw_range_elements\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_packed_pixels", extensions)) { printf("    GL_EXT_packed_pixels\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_rescale_normal", extensions)) { printf("    GL_EXT_rescale_normal\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_separate_specular_color", extensions)) { printf("    GL_EXT_separate_specular_color\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_texture3D", extensions)) { printf("    GL_EXT_texture3D\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_texture_edge_clamp", extensions)) { printf("    GL_EXT_texture_edge_clamp\n"); i++;} j++;
    if (!op ^ findExtension("GL_SGIS_texture_lod", extensions)) { printf("    GL_SGIS_texture_lod\n"); i++;} j++;
    printf("    => %d%% (%d/%d)\n", (int)((float)i/(float)j * 100.0), i, j);
    if (!op)
      break;
    op = !op;
  }
}

void
parseOpenGL13Capabilities(const char* extensions)
{
  printf("\n_____________________________________________________________\n");
  printf("Compliance OpenGL v1.3\n");
  bool op = true;
  while (1) {
    op ? printf("  Supported:\n") : printf("  Unsupported:\n");
    int i = 0;
    int j = 0;
    if (!op ^ findExtension("GL_ARB_multisample", extensions)) { printf("    GL_ARB_multisample\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_multitexture", extensions)) { printf("    GL_ARB_multitexture\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_texture_border_clamp", extensions)) { printf("    GL_ARB_texture_border_clamp\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_texture_compression", extensions)) { printf("    GL_ARB_texture_compression\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_texture_cube_map", extensions)) { printf("    GL_ARB_texture_cube_map\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_texture_env_add", extensions)) { printf("    GL_ARB_texture_env_add\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_texture_env_combine", extensions)) { printf("    GL_ARB_texture_env_combine\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_texture_env_dot3", extensions)) { printf("    GL_ARB_texture_env_dot3\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_transpose_matrix", extensions)) { printf("    GL_ARB_transpose_matrix\n"); i++;} j++;
    printf("    => %d%% (%d/%d)\n", (int)((float)i/(float)j * 100.0), i, j);
    if (!op)
      break;
    op = !op;
  }
}

void
parseOpenGL14Capabilities(const char* extensions)
{
  printf("\n_____________________________________________________________\n");
  printf("Compliance OpenGL v1.4\n");
  bool op = true;
  while (1) {
    op ? printf("  Supported:\n") : printf("  Unsupported:\n");
    int i = 0;
    int j = 0;
    if (!op ^ findExtension("GL_ARB_imaging", extensions)) { printf("    GL_ARB_imaging\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_point_parameters", extensions)) { printf("    GL_ARB_point_parameters\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_texture_mirrored_repeat",extensions)) { printf("    GL_ARB_texture_mirrored_repeat\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_shadow", extensions)) { printf("    GL_ARB_shadow\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_vertex_program", extensions)) { printf("    GL_ARB_vertex_program\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_blend_func_separate", extensions)) { printf("    GL_EXT_blend_func_separate\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_fog_coord", extensions)) { printf("    GL_EXT_fog_coord\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_multi_draw_arrays", extensions)) { printf("    GL_EXT_multi_draw_arrays\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_secondary_color", extensions)) { printf("    GL_EXT_secondary_color\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_stencil_wrap", extensions)) { printf("    GL_EXT_stencil_wrap\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_texture_lod_bias", extensions)) { printf("    GL_EXT_texture_lod_bias\n"); i++;} j++;
    if (!op ^ findExtension("GL_NV_blend_square", extensions)) { printf("    GL_NV_blend_square\n"); i++;} j++;
    if (!op ^ findExtension("GL_NV_texture_env_combine4", extensions)) { printf("    GL_NV_texture_env_combine4\n"); i++;} j++;
    if (!op ^ findExtension("GL_SGIS_generate_mipmap", extensions)) { printf("    GL_SGIS_generate_mipmap\n"); i++;} j++;
    if (!op ^ findExtension("GL_WIN_swap_hint", extensions)) { printf("    GL_WIN_swap_hint\n"); i++;} j++;
    printf("    => %d%% (%d/%d)\n", (int)((float)i/(float)j * 100.0), i, j);
    if (!op)
      break;
    op = !op;
  }
}

void
parseOpenGL15Capabilities(const char* extensions)
{
  printf("\n_____________________________________________________________\n");
  printf("Compliance OpenGL v1.5\n");
  bool op = true;
  while (1) {
    op ? printf("  Supported:\n") : printf("  Unsupported:\n");
    int i = 0;
    int j = 0;
    if (!op ^ findExtension("GL_ARB_fragment_program", extensions)) { printf("    GL_ARB_fragment_program\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_occlusion_query", extensions)) { printf("    GL_ARB_occlusion_query\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_point_sprite", extensions)) { printf("    GL_ARB_point_sprite\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_vertex_buffer_object", extensions)) { printf("    GL_ARB_vertex_buffer_object\n"); i++;} j++;
    if (!op ^ findExtension("GL_EXT_shadow_funcs", extensions)) { printf("    GL_EXT_shadow_funcs\n"); i++;} j++;
    if (!op ^ findExtension("GL_NV_texture_rectangle", extensions)) { printf("    GL_NV_texture_rectangle\n"); i++;} j++;
    printf("    => %d%% (%d/%d)\n", (int)((float)i/(float)j * 100.0), i, j);
    if (!op)
      break;
    op = !op;
  }
}

void
parseOpenGL20Capabilities(const char* extensions)
{
  printf("\n_____________________________________________________________\n");
  printf("Compliance OpenGL v2.0 (OpenGL SL)\n");
  bool op = true;
  while (1) {
    op ? printf("  Supported:\n") : printf("  Unsupported:\n");
    int i = 0;
    int j = 0;
    if (!op ^ findExtension("GL_ARB_fragment_shader", extensions)) { printf("    GL_ARB_fragment_shader\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_shader_objects", extensions)) { printf("    GL_ARB_shader_objects\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_shading_language_100", extensions)) { printf("    GL_ARB_shading_language_100\n"); i++;} j++;
    if (!op ^ findExtension("GL_ARB_vertex_shader", extensions)) { printf("    GL_ARB_vertex_shader\n"); i++;} j++;
    printf("    => %d%% (%d/%d)\n", (int)((float)i/(float)j * 100.0), i, j);
    if (!op)
      break;
    op = !op;
  }
}
int 
getOpenGLInfo()
{
  // Register class for OpenGL window
  WNDCLASS wndclass;
  wndclass.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
  wndclass.lpfnWndProc = DefWindowProc;
  wndclass.cbClsExtra  = 0;
  wndclass.cbWndExtra  = 0;
  wndclass.hInstance   = GetModuleHandle(NULL);
  wndclass.hIcon       = NULL;
  wndclass.hCursor     = NULL;
  wndclass.hbrBackground = NULL;
  wndclass.lpszMenuName  = NULL;
  wndclass.lpszClassName = "OGLInfoWindowMCS";
  
  ATOM result = RegisterClass(&wndclass);
  if (result == 0) ERROR_RETURN("RegisterClass");
  
  // Create window for OpenGL queries
  // Note: WS_CLIPxxx are required for MS-OpenGL to work...
  HWND hwnd;
  hwnd = CreateWindow("OGLInfoWindowMCS",
                      "OGLInfoWindowMCS",
                      WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
                      0,                     // X position
                      0,                     // Y position
                      1,                     // Width
                      1,                     // Height
                      NULL,                  // Parent
                      NULL,                  // Menu
                      GetModuleHandle(NULL), // App instance handle
                      NULL);                // Window creation data
  if (hwnd == NULL) ERROR_RETURN("CreateWindow");
  
  // Get device context for OpenGL window
  HDC hdc;
  hdc = GetDC(hwnd);

  // Setup pixel format descriptor
  // Shouldn't matter exactly what we use here, but these are the
  // default values used by Inventor.
  PIXELFORMATDESCRIPTOR pfd;
  pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
  pfd.nVersion = 1;
  pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
  pfd.iPixelType = PFD_TYPE_RGBA;
  pfd.cColorBits = 24;
  pfd.cRedBits = 0;
  pfd.cRedShift = 0;
  pfd.cGreenBits = 0;
  pfd.cGreenShift = 0;
  pfd.cBlueBits = 0;
  pfd.cBlueShift = 0;
  pfd.cAlphaBits = 0;
  pfd.cAlphaShift = 0;
  pfd.cAccumBits = 0;
  pfd.cAccumRedBits = 0;
  pfd.cAccumGreenBits = 0;
  pfd.cAccumBlueBits = 0;
  pfd.cAccumAlphaBits = 0;
  pfd.cDepthBits = 32;
  pfd.cStencilBits = 0;
  pfd.cAuxBuffers = 0;
  pfd.iLayerType = PFD_MAIN_PLANE;
  pfd.bReserved = 0;
  pfd.dwLayerMask = 0;
  pfd.dwVisibleMask = 0;
  pfd.dwDamageMask = 0;
  
  // Get a pixel format
  int pixelformat;
  if ((pixelformat = ChoosePixelFormat(hdc, &pfd)) == 0)
    ERROR_RETURN("ChoosePixelFormat");
  
  // Set the pixel format
  if (SetPixelFormat(hdc, pixelformat, &pfd) == FALSE)
    ERROR_RETURN("SetPixelFormat");
  
  // Create an OpenGL render context
  SoRef<SoGLContext> ctx = new SoGLContext( hdc, &pfd, NULL );
  
  // Make OpenGL context current
  ctx->bind();
  
  // Inquire OpenGL vendor information
  const GLubyte *pVenInfo = glGetString(GL_VENDOR);
  const GLubyte *pRenInfo = glGetString(GL_RENDERER);
  const GLubyte *pVerInfo = glGetString(GL_VERSION);
  const GLubyte *pExtInfo = glGetString(GL_EXTENSIONS);
  printf("OpenGL Info (A Thermo Fisher Scientific Utility - %s)\n"
         "  Vendor:   %s\n"
         "  Renderer: %s\n"
         "  Version:  %s\n"
         ,__DATE__, pVenInfo, pRenInfo, pVerInfo, pExtInfo);

#ifndef OGLINFO_CONSOLE
  OGLGenDialog->editText = OutputStream.c_str() ; 
  OGLInfoText += OutputStream ;
  OutputStream.erase() ;
#else
  puts("Extensions:");  // automatically writes a newline char
#endif

  printMultiString((const char*)pExtInfo);

  // Now get wgl extension info
  //
  // Note: wglGetExtensionsStringARB is the generally supported (Nvidia,
  //       ATI, etc) extension function, so try that first.  However the
  //       3DLabs Wildcat boards do not support this extension, they
  //       support wglGetExtensionsStringEXT, which has a different
  //       function signature.                           mmh Jan-02
  const char *wglExt=NULL;
  if (wglGetExtensionsStringARB) {
    wglExt = wglGetExtensionsStringARB(wglGetCurrentDC());
#ifdef OGLINFO_CONSOLE
    puts("  WGL Extensions:");  // writes a newline char
#endif
    printMultiString(wglExt);
  }
  else {
    if (wglGetExtensionsStringEXT) {
      wglExt = wglGetExtensionsStringEXT();
#ifdef OGLINFO_CONSOLE
      puts("  WGL Extensions:");  // writes a newline char
#endif
      printMultiString(wglExt);
    }
  }
  putchar('\n');
  
  if (findExtension("WGL_ARB_pixel_format", wglExt))
  if (findExtension("GL_ARB_multisample", (const char*)pExtInfo))
    isAvailableMS = TRUE;

#ifndef OGLINFO_CONSOLE
  OGLExtDialog->editText = OutputStream.c_str() ;
  OGLInfoText += OutputStream ;
  OutputStream.erase() ;
#endif

  getPixFormatInfo();

#ifndef OGLINFO_CONSOLE
  OGLPixelFormDialog->editText = OutputStream.c_str() ; 
  putchar('\n');
  OGLInfoText += OutputStream ;
  OutputStream.erase() ;
  
  getOIVInfo(hdc) ;

  OGLIVExtDialog->editText = OutputStream.c_str() ; 
  OGLInfoText += OutputStream ;
  OutputStream.erase() ;
#endif

  // Cleanup
  ctx->unbind();
  ctx = NULL;
  ReleaseDC(hwnd, hdc);
  DestroyWindow(hwnd);

  return 0;
}

//////////////////////////////////////////////////////////////////////////////
int
getPixFormatInfo()
{
  // Create DC for query functions
  // Get some info about the current display
  int horzres = GetDeviceCaps(wglGetCurrentDC(), HORZRES);
  int vertres = GetDeviceCaps(wglGetCurrentDC(), VERTRES);
  int bitspix = GetDeviceCaps(wglGetCurrentDC(), BITSPIXEL);
  
  int numpfd;
  PIXELFORMATDESCRIPTOR pfd;
  int status;
  int* iAttributes = new int[50];
  int* iResults = new int[50];

  if (wglGetPixelFormatAttribivARB) {
    iAttributes[0] = WGL_NUMBER_PIXEL_FORMATS_ARB;
    iAttributes[1] = 0;
    status = wglGetPixelFormatAttribivARB(wglGetCurrentDC(), 1, 0, 1, iAttributes, iResults);
    numpfd = iResults[0];
  }
  else {
    // Get number of pixel formats
    // Note: Generic MS OpenGL allows passing zero and NULL to just get
    // the max, but some impls are broken.            -- mmh 14-Jan-00
    // int numpfd = DescribePixelFormat(theDC, 0, 0, NULL);
    numpfd = DescribePixelFormat(wglGetCurrentDC(), 1, sizeof(PIXELFORMATDESCRIPTOR), &pfd);
  }
  printf("%d pixel formats at %dx%dx%d:\n", numpfd, horzres, vertres, bitspix);

  // Print header
  if (isAvailableMS && wglGetPixelFormatAttribivARB) {
    printf("------------------------------------------------------------------------------------------\n");
    printf("                                        Accu     Dpth HW  Use GDI Over Ste     Draw\n");
    printf("     Type           Color Bits      ZBf Bits Stn Buf  Acc Pal OK  lay  reo     to      Smp\n");
    printf("------------------------------------------------------------------------------------------\n");
  }
  else {
    if(wglGetPixelFormatAttribivARB) {
      printf("----------------------------------------------------------------------------------------\n");
      printf("                                  Depth Accu Sten Dbl HW  Use GDI Over Ste    Draw\n");
      printf("    Type           Color Bits     Bits  Bits Bits Buf Acc Pal OK  lay  reo    to\n");
      printf("----------------------------------------------------------------------------------------\n");
    }
    else {
      printf("---------------------------------------------------------------------------------------\n");
      printf("                             Depth  Accu  Sten  Dbl  HW   Use  GDI Over Ste    Draw\n");
      printf("    Type        Color Bits   Bits   Bits  Bits  Buf  Acc  Pal  OK  lay  reo    to\n");
      printf("---------------------------------------------------------------------------------------\n");
    }
  }  
  // Loop over pixel formats
  int rc, format;
  int i=0;
  iAttributes[i++] = WGL_DRAW_TO_WINDOW_ARB;
  iAttributes[i++] = WGL_DRAW_TO_BITMAP_ARB;
  iAttributes[i++] = WGL_ACCELERATION_ARB;
  iAttributes[i++] = WGL_NEED_PALETTE_ARB;
  iAttributes[i++] = WGL_NEED_SYSTEM_PALETTE_ARB;
  iAttributes[i++] = WGL_SWAP_LAYER_BUFFERS_ARB;
  iAttributes[i++] = WGL_SWAP_METHOD_ARB;
  iAttributes[i++] = WGL_NUMBER_UNDERLAYS_ARB;
  iAttributes[i++] = WGL_TRANSPARENT_ARB;
  iAttributes[i++] = WGL_TRANSPARENT_RED_VALUE_ARB;
  iAttributes[i++] = WGL_TRANSPARENT_GREEN_VALUE_ARB;
  iAttributes[i++] = WGL_TRANSPARENT_BLUE_VALUE_ARB;
  iAttributes[i++] = WGL_TRANSPARENT_ALPHA_VALUE_ARB;
  iAttributes[i++] = WGL_TRANSPARENT_INDEX_VALUE_ARB;
  iAttributes[i++] = WGL_SHARE_DEPTH_ARB;
  iAttributes[i++] = WGL_SHARE_STENCIL_ARB;
  iAttributes[i++] = WGL_SHARE_ACCUM_ARB;
  iAttributes[i++] = WGL_SUPPORT_GDI_ARB;
  iAttributes[i++] = WGL_SUPPORT_OPENGL_ARB;
  iAttributes[i++] = WGL_DOUBLE_BUFFER_ARB;
  iAttributes[i++] = WGL_STEREO_ARB;
  iAttributes[i++] = WGL_PIXEL_TYPE_ARB;
  iAttributes[i++] = WGL_COLOR_BITS_ARB;
  iAttributes[i++] = WGL_RED_BITS_ARB;
  iAttributes[i++] = WGL_RED_SHIFT_ARB;
  iAttributes[i++] = WGL_GREEN_BITS_ARB;
  iAttributes[i++] = WGL_GREEN_SHIFT_ARB;
  iAttributes[i++] = WGL_BLUE_BITS_ARB;
  iAttributes[i++] = WGL_BLUE_SHIFT_ARB;
  iAttributes[i++] = WGL_ALPHA_BITS_ARB;
  iAttributes[i++] = WGL_ALPHA_SHIFT_ARB;
  iAttributes[i++] = WGL_ACCUM_BITS_ARB;
  iAttributes[i++] = WGL_ACCUM_RED_BITS_ARB;
  iAttributes[i++] = WGL_ACCUM_GREEN_BITS_ARB;
  iAttributes[i++] = WGL_ACCUM_BLUE_BITS_ARB;
  iAttributes[i++] = WGL_ACCUM_ALPHA_BITS_ARB;
  iAttributes[i++] = WGL_DEPTH_BITS_ARB;
  iAttributes[i++] = WGL_STENCIL_BITS_ARB;
  iAttributes[i++] = WGL_AUX_BUFFERS_ARB;
  iAttributes[i++] = WGL_DRAW_TO_PBUFFER_ARB;

  if (isAvailableMS) {
    iAttributes[i++] = WGL_SAMPLE_BUFFERS_ARB;
    iAttributes[i++] = WGL_SAMPLES_ARB;
  }

  iAttributes[i] = 0;

  for (format = 1; format <= numpfd; format++) {
    if (wglGetPixelFormatAttribivARB) {
      status = wglGetPixelFormatAttribivARB(wglGetCurrentDC(), format, 0, i, iAttributes, iResults);
      if (status) {
        if (!isAvailableMS) {
          iResults[i] = iResults[i+1] = 0;
        }

        printFormatARB(format, iResults);
      }
    }
    else {
      rc = DescribePixelFormat(wglGetCurrentDC(), format, sizeof(PIXELFORMATDESCRIPTOR), &pfd);
      if (rc == 0) ERROR_RETURN("DescribePixelFormat");
    
      // Skip formats that don't support OpenGL
      if (pfd.dwFlags & PFD_SUPPORT_OPENGL) {
        printFormat(format, pfd);
      }
    }
  }

  delete [] iAttributes;
  delete [] iResults;

  return numpfd;
}

//////////////////////////////////////////////////////////////////////////////
int 
printFormatARB(int format, int* iResults)
{
  printf("%3d:", format);
  
  int isRGBA     = (iResults[22] == WGL_TYPE_RGBA_ARB);
  int isDBuf     = (iResults[20] == GL_TRUE);
  int isStereo   = (iResults[21] == GL_TRUE);
  int isSwapLay  = (iResults[5] == GL_TRUE);
  
  // If      PFD_GENERIC_FORMAT not set  ==> ICD acceleration
  // else if PFD_GENERIC_ACCELERATED set ==> MCD acceleration
  // else not accelerated
  char isAccel = '-';
  if (iResults[2] == WGL_NO_ACCELERATION_ARB)
    isAccel = 'I';  // ICD
  else if (iResults[2] == WGL_FULL_ACCELERATION_ARB)
    isAccel = 'M';  // MCD
  
  //printf(" %s", (isRGBA ? "RGBA " : "Index"));
  if (iResults[22] == WGL_TYPE_RGBA_ARB)
    printf(" %s", "RGBA       ");
  else if (iResults[22] == WGL_TYPE_RGBA_FLOAT_ATI)
    printf(" %s", "FLOAT_RGBA ");
  else 
    printf(" %s", "Indexed    ");
  printf(" %3d (%2d/%2d/%2d/%2d)", iResults[23], iResults[24], iResults[26], iResults[28], iResults[30]);
  printf("   %2d" , iResults[37]);
  printf("  %3d" , iResults[32]);
  printf("  %2d" , iResults[38]);
  printf("    %s", isDBuf  ? "Y"  : "-");
  printf("   %c", isAccel);
  
  // Note: NEED_PALETTE means the format is RGBA but the device is indexed,
  //       so the application must create an appropriate logical palette.
  // Note: NEED_SYSTEM_PALETTE means the device must take over the entire
  //       palette (including the 20 standard Windows colors) in order to
  //       accelerate rendering.  App must call SetSystemPaletteUse.
  printf("   %s", (iResults[3] == GL_TRUE) ? "Y" : "-");
  // Remove following per MMH
  printf("   %s", (iResults[18] == GL_TRUE) ? "Y" : "-");
  
  printf("    %s", "-");
  printf("   %s", isStereo  ? "Y" : "-");

  SbString drawto;

  if (iResults[0] == GL_TRUE)
    drawto +="Win";
  if (iResults[1] == GL_TRUE) {
    if (drawto.getLength() != 0)
      drawto +="/";
    drawto +="Bmp";
  }
  if (iResults[40] == GL_TRUE) {
    if (drawto.getLength() != 0)
      drawto +="/";
    drawto +="PBuf";
  }

  while (drawto.getLength() != 8) {
    drawto +=" ";
  }

  printf("   %s", drawto.getString());

  if(isAvailableMS)
    printf("  %2d", iResults[42]);
  
  printf("\n");
  return 0;
}

//////////////////////////////////////////////////////////////////////////////
int 
printFormat(int format, PIXELFORMATDESCRIPTOR& pfd)
{
  printf("%3d:", format);
  
  int isRGBA     = (pfd.iPixelType == PFD_TYPE_RGBA);
  int isDBuf     = (pfd.dwFlags & PFD_DOUBLEBUFFER);
  int isStereo   = (pfd.dwFlags & PFD_STEREO);
  int isSwapLay  = (pfd.dwFlags & PFD_SWAP_LAYER_BUFFERS);
  
  // If      PFD_GENERIC_FORMAT not set  ==> ICD acceleration
  // else if PFD_GENERIC_ACCELERATED set ==> MCD acceleration
  // else not accelerated
  char isAccel = '-';
  if ((pfd.dwFlags & PFD_GENERIC_FORMAT) == 0)
    isAccel = 'I';  // ICD
  else if ((pfd.dwFlags & PFD_GENERIC_ACCELERATED) != 0)
    isAccel = 'M';  // MCD
  
  printf(" %s", (isRGBA ? "RGBA " : "Index"));
  printf("  %2d (%d/%d/%d/%d)",
         pfd.cColorBits, pfd.cRedBits,
         pfd.cGreenBits, pfd.cBlueBits, pfd.cAlphaBits);
  printf("  %2d" , pfd.cDepthBits);
  printf("    %3d" , pfd.cAccumBits);
  printf("   %3d" , pfd.cStencilBits);
  printf("    %s", isDBuf  ? "Y"  : "-");
  printf("    %c", isAccel);
  
  // Note: NEED_PALETTE means the format is RGBA but the device is indexed,
  //       so the application must create an appropriate logical palette.
  // Note: NEED_SYSTEM_PALETTE means the device must take over the entire
  //       palette (including the 20 standard Windows colors) in order to
  //       accelerate rendering.  App must call SetSystemPaletteUse.
  printf("    %s", (pfd.dwFlags & PFD_NEED_PALETTE) ? "Y" : "-");
  // Remove following per MMH
  printf("    %s", (pfd.dwFlags & PFD_SUPPORT_GDI) ? "Y" : "-");
  
  printf("   %s", "-");
  printf("   %s", isStereo  ? "Y" : "-");

  char *drawto = "";
  if ((pfd.dwFlags & PFD_DRAW_TO_WINDOW) != 0) {
    drawto = "win";
    if ((pfd.dwFlags & PFD_DRAW_TO_BITMAP) != 0)
      drawto = "w/b";
  }
  else {
    if ((pfd.dwFlags & PFD_DRAW_TO_BITMAP) != 0) {
      drawto = "bm";
    }
  }
  printf("  %s", drawto);
    
  printf("\n");
  return 0;
}

//////////////////////////////////////////////////////////////////////////////

void 
printMultiString(const char *string, int startCount,
                 int maxWidth, int indent)
{
  const int MAXSTRINGS = 512;
  
  if (!string || *string == EOS)
    return;
  
  // Make copy of multistring so we can insert EOS chars
  char *tempStr = _strdup(string);
  
  // Make array of pointers to be sorted
  char **strings = new char*[MAXSTRINGS];
  
  // Find strings
  int num = 0;
  char *pbeg = tempStr;
  while (1) {
    char *pend;
    while (*pbeg && *pbeg == ' ') pbeg++; // skip whitespace
    if (*pbeg == EOS) break;
    pend = pbeg + 1;
    while (*pend && *pend != ' ') pend++; // find whitespace
    
    // Handle special case where pend now points to end of string
    // (means we are at the end of the entire multistring)  --mmh
    if (*pend == EOS) {
      strings[num++] = pbeg;
      break;
    }
    else {
      *pend = EOS;
      strings[num++] = pbeg;
      pbeg = pend + 1;
    }
  }
  
  // Sort strings (very inefficiently but it won't matter)
  char *ptemp;
  int i, j;
  for (i = 0; i < (num-1); ++i) {
    for (j = i; j < num; ++j) {
      if (strcmp(strings[i],strings[j]) > 0) {
        ptemp = strings[i];
        strings[i] = strings[j];
        strings[j] = ptemp;
      }
    }
  }
  
  // Print strings
#ifdef OGLINFO_CONSOLE
  int width;
  if (startCount == 0) {
    for (int i = 0; i < (indent-1); ++i) putchar(' ');
    width = indent;
  }
  else
    width = startCount;
  
  for (i = 0; i < num; ++i) {
    int length = strlen(strings[i]);
    if ((width + length + 1) >= maxWidth) { // start new line
      putchar('\n');
      for (int i = 0; i < indent; ++i) putchar(' ');
      width = indent;
    }
    else
      putchar(' ');
    printf("%s", strings[i]); //Can't use puts, it writes a newline char!
    width += length + 1;
  }
  putchar('\n');
#else
   for (i = 0; i < num; i++)
       printf("%s\n", strings[i]) ;
#endif
  
  // Free memory allocated by us and by strdup function
  delete [] strings;
  free(tempStr);
}

/*************************************************************************/

#ifndef OGLINFO_CONSOLE
void
getOIVInfo(HDC hdc)
{
  std::string tmpStr ;

  printf("Information about support for advanced Open Inventor features on this graphics board.\n") ;

  printf( "\nTexture features:\n" );
  printf( "  3D Textures:               %s\n", 
          ((SoVolumeRendering::isSupported(SoVolumeRendering::HW_3DTEXMAP) == SoVolumeRendering::YES) ? 
          "OK" : "not available"));
  printf( "  Compression:               %s\n", 
          ((SoVolumeRendering::isSupported(SoVolumeRendering::HW_TEXCOMPRESSION) == SoVolumeRendering::YES) ? 
          "OK" : "not available") );
  printf( "  Paletted:                  %s\n", 
          ((SoVolumeRendering::isSupported(SoVolumeRendering::HW_TEXCOLORMAP) == SoVolumeRendering::YES) ? 
          "OK" : "not available") );
  int nbUnits = SoTextureUnitElement::getMaxTextureUnit() ;
  printf( "  Multitextures:             %s (%d unit(s))\n", 
          (nbUnits > 1 ? "OK" : "not available"), nbUnits );
  printf( "  Non Power of 2:            %s\n", 
    SoTexture::isSupported(SoTexture::HW_NPOT)?"OK" : "not available");
  printf( "  Float format:              %s\n", 
    SoTexture::isSupported(SoTexture::HW_FLOATFORMAT)?"OK" : "not available");
  printf( "  Depth format:              %s\n", 
    SoTexture::isSupported(SoTexture::HW_DEPTHFORMAT)?"OK" : "not available");
  printf( "  Auto mipmap generation:    %s\n", 
    SoTexture::isSupported(SoTexture::HW_AUTOMIPMAP)?"OK" : "not available");
  printf( "  Border clamp:              %s\n", 
    SoTexture::isSupported(SoTexture::HW_BORDER_CLAMP)?"OK" : "not available");
  printf( "  Edge clamp:                %s\n", 
    SoTexture::isSupported(SoTexture::HW_EDGE_CLAMP)?"OK" : "not available");
  printf( "  Mirrored repeat:           %s\n", 
    SoTexture::isSupported(SoTexture::HW_MIRRORED_REPEAT)?"OK" : "not available");
  printf( "  DDS support (S3TC):        %s\n\n", 
    SoTexture::isSupported(SoTexture::HW_COMPRESSION_S3TC)?"OK" : "not available");

  printf( "Floating point buffer:       %s\n\n", 
    SoWinRenderArea::isFloatingColorBufferSupported()?"OK" : "not available");

  printf( "Sorted Layers Blend:         %s\n\n", 
          ((SoGLRenderAction:: isSortedLayersSupported ()) ? "OK" : "not available") );


  printf( "Shadows:                     %s\n\n", 
          ((SoShadowGroup:: isSupported()) ? "OK" : "not available") );

  SoGLGraphicConfigTemplate glTemplate ;
  glTemplate.setFullSceneAntialiasing(SoGLGraphicConfigTemplate::REQUIRED) ;
  SoGLGraphicDevice graphicDevice(hdc) ;
  SoGLGraphicConfig *graphicConfig = (SoGLGraphicConfig *) graphicDevice.getBestGraphicConfig(glTemplate) ;
  int nbs=0 ;
  SbBool isFSAA = FALSE ;
  if(graphicConfig)
    isFSAA = graphicConfig->getFullSceneAntialiasing(nbs) ;


  printf( "FSAA:                        %s (%d sample(s))\n\n",  
          ((isFSAA) ? "OK" : "not available"), nbs );

  printf( "VBO:                         %s\n\n", 
    ((SoShapeHints::isVBOSupported()) ? "OK" : "not available") );

  printf( "Shader features:\n" );

#define CHECK_SHADERS(_language)\
{\
  SbBool isVtx ;\
  if((isVtx = SoVertexShader::isSupported(_language)) &&\
     SoFragmentShader::isSupported(_language))\
    tmpStr = "OK" ;\
  else if(isVtx)\
    tmpStr = "only vertex program" ;\
  else\
    tmpStr = "not available" ;\
}

  CHECK_SHADERS(SoShaderObject::GLSL_PROGRAM);
  printf( "  GLSL:                      %s\n", tmpStr.c_str()) ;

  CHECK_SHADERS(SoShaderObject::CG_PROGRAM);
  printf( "  NVIDIA Cg:                 %s\n", tmpStr.c_str()) ;

  CHECK_SHADERS(SoShaderObject::ARB_PROGRAM);
  printf( "  ARB_LANGUAGE:              %s\n\n", tmpStr.c_str()) ;

  printf( "Geometry shaders features:\n" );
#define CHECK_GEOMETRY_SHADERS(_language)\
{\
  if((SoGeometryShader::isSupported(_language)))\
    tmpStr = "OK" ;\
  else\
    tmpStr = "not available" ;\
}

  CHECK_GEOMETRY_SHADERS(SoShaderObject::GLSL_PROGRAM);
  printf( "  GLSL:                      %s\n", tmpStr.c_str()) ;

  CHECK_GEOMETRY_SHADERS(SoShaderObject::CG_PROGRAM);
  printf( "  NVIDIA Cg:                 %s\n", tmpStr.c_str()) ;

  CHECK_GEOMETRY_SHADERS(SoShaderObject::ARB_PROGRAM);
  printf( "  ARB_LANGUAGE:              %s\n\n", tmpStr.c_str()) ;
}
#endif

