/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

#include <string.h>

#ifdef _WIN32
#  include <io.h> 
#  include "getopt/getopt.h"
#else
#  include <unistd.h>
#endif

#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoInteraction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/actions/SoReorganizeAction.h>
#include <Inventor/nodes/SoSeparator.h>

#ifdef _WIN32
#  define strdup _strdup
#  define fileno _fileno
#  define isatty _isatty
#endif

/////////////////////////////////////////////////////////////////////////////
//
// This structure is used to store and pass around command-line options.
//
/////////////////////////////////////////////////////////////////////////////

struct OptionInfo{
  OptionInfo();
  
  SbBool verbose; 
  SbBool writeAscii;
  SbBool doAnyNormals;
  SbBool doMatching;
  SbBool doAnyTexCoords;
  SbBool writeStrips;
  SbBool writeVertexProperty;
  SoInput inFile;
  SoOutput outFile;
};

OptionInfo::OptionInfo()
  : verbose(FALSE),
    writeAscii(FALSE),
    doAnyNormals(TRUE),
    doMatching(TRUE),
    doAnyTexCoords(TRUE),
    writeStrips(TRUE),
    writeVertexProperty(TRUE)
{
}

/////////////////////////////////////////////////////////////////////////////
//
// Forward references.
//

static void printUsage(const char *progname);
static void parseArgs(int argc, char **argv, OptionInfo &options);

/////////////////////////////////////////////////////////////////////////////
//
// Mainline.
//

int
main(int argc, char **argv)
{
  OptionInfo options;

  SoInteraction::init();

  // Parse arguments
  parseArgs(argc, argv, options);
  SoReorganizeAction orgAct;
  orgAct.setVerbosity(options.verbose);
  orgAct.generateTriangleStrips(options.writeStrips);
  orgAct.generateNormals(options.doAnyNormals);
  orgAct.matchIndexArrays(options.doMatching);
  orgAct.generateTexCoords(options.doAnyTexCoords);
  orgAct.generateNormals(options.doAnyNormals);
  orgAct.generateVPNodes(options.writeVertexProperty);

  // Read stuff:
  orgAct.startReport("Reading file");
  SoSeparator *root = SoDB::readAll(&options.inFile);
  if (root == NULL) {
    fprintf(stderr, "\n%s: Problem reading data from file\n", argv[0]);
    printUsage(argv[0]);
  }
  orgAct.finishReport();
  root->ref();
  // We don't need to ref the root since we want the IfFixer to get
  // rid of the scene graph when it's done with it
  orgAct.apply(root);
  SoNode *resultRoot = orgAct.getSimplifiedSceneGraph();
  if (resultRoot == NULL) {
    fprintf(stderr, "%s: No shapes found in data\n", argv[0]);
    exit(-1);
  }

  resultRoot->ref();

  // Write out the results
  orgAct.startReport("Writing result");
  SoWriteAction wa(&options.outFile);
  if (! options.writeAscii)
    wa.getOutput()->setBinary(TRUE);
  wa.apply(resultRoot);
  orgAct.finishReport();

  // Clean up
  resultRoot->unref();

  SoInteraction::finish();

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//
// Prints usage message and exits.
//

static void
printUsage(const char *progname)
{
  fprintf(stderr, "Usage: %s [options] [infile] [outfile]\n", progname);
  fprintf(stderr,
        "\t-a     : Write out an ascii file.  Default is binary\n"
        "\t-d dir : Add 'dir' to the list of directories to search\n"
        "\t-f     : Produce independent faces rather than tri strips\n"
        "\t-h     : Print this message (help)\n"
        "\t-m     : Do not try to match up index arrays\n"
        "\t-n     : Do not generate any normals\n"
        "\t-p     : Do not produce SoVertexProperty nodes for properties\n"
        "\t-t     : Do not generate any texture coordinates\n"
        "\t-v     : (Verbose) Display status info during processing\n");

  exit(99);
}

/////////////////////////////////////////////////////////////////////////////
//
// Parses input options, filling in OptionInfo structure.
//

static void
parseArgs(int argc, char **argv, OptionInfo &options)
{
  SbBool uhoh = FALSE;
  int c;
    
  while ((c = getopt(argc, argv, "ad:fhmnptv")) != -1) {
    switch(c) {
    case 'a':
      options.writeAscii = TRUE;
      break;
    case 'd':
      options.inFile.addDirectoryLast(optarg);
      break;
    case 'f':
      options.writeStrips = FALSE;
      break;
    case 'n':
      options.doAnyNormals = FALSE;
      break;
    case 'm':
      options.doMatching = FALSE;
      break;
    case 'p':
      options.writeVertexProperty = FALSE;
      break;
    case 't':
      options.doAnyTexCoords = FALSE;
      break;
    case 'v':
      options.verbose = TRUE;
      break;
    case 'h': // Help
    default:
      uhoh = TRUE;
      break;
    }
  }

  // Handle optional input file name
  if (optind < argc) {
    char *inFileName = argv[optind++];
    if (! options.inFile.openFile(inFileName)) {
      fprintf(stderr, "%s: Could not open file %s", argv[0], inFileName);
      printUsage(argv[0]);
    }
    // If the filename includes a directory path, add the directory name
    // to the list of directories where to look for input files
    const char *slashPtr;
    char *searchPath = NULL;
    if ((slashPtr = strrchr(inFileName, '/')) != NULL) {
      searchPath = strdup(inFileName);
      searchPath[slashPtr - inFileName] = '\0';
      SoInput::addDirectoryFirst(searchPath);
      free(searchPath);
    }
  }
  // Trying to read stdin from a tty is a no-no
  else if (isatty(fileno(stdin)))
    uhoh = TRUE;

  // Handle optional output file name
  if (optind < argc) {
    char *outFileName = argv[optind++];
    if (! options.outFile.openFile(outFileName)) {
      fprintf(stderr, "%s: Could not open file %s", argv[0], outFileName);
      printUsage(argv[0]);
    }
  }

  // Too many arguments?
  if (optind < argc)
    uhoh = TRUE;

  if (uhoh)
    printUsage(argv[0]);
}



