/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/

#ifdef _WIN32
#  include <stdlib.h>
#  include <io.h>
#  include "getopt/getopt.h"
#else
#  include <unistd.h>
#endif

#include <Inventor/SoDB.h>
#include <Inventor/SoInteraction.h>
#include <Inventor/SoInput.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoInfo.h>

//
// Counts and returns the number of nodes under the given root node.
// This is recursive.
//

static int
countNodes(const SoNode *root)
{
    int numNodes = 1;

    if (root->isOfType(SoGroup::getClassTypeId())) {

	const SoGroup	*group = (const SoGroup *) root;
	int		i;

	for (i = 0; i < group->getNumChildren(); i++)
	    numNodes += countNodes(group->getChild(i));
    }

    return numNodes;
}

//
// Searches for all SoInfo nodes in the input graph, printing out the
// string field for any that are found.
//

static void
printInfo(SoNode *root)
{
    SoSearchAction	sa;
    const SoInfo	*info;
    int			i;

    // Set up action to search for SoInfo nodes
    sa.setType(SoInfo::getClassTypeId());
    sa.setInterest(SoSearchAction::ALL);
    sa.setSearchingAll(TRUE);

    sa.apply(root);

    for (i = 0; i < sa.getPaths().getLength(); i++) {
	info = (const SoInfo *) sa.getPaths()[i]->getTail();
	printf("%s\n", info->string.getValue().getString());
    }
}

static void
printUsage()
{
    fprintf(stderr, "Usage: ivInfo [-h] [file]\n");
    fprintf(stderr,
	    "-h : Print this message (help)\n"
	    "If no filename is given, standard input will be read\n");
}

static void
parseArgs(int argc, char **argv, SoInput &in)
{
    int err = 0;	// Flag: error in options?
    int c;
    
    while ((c = getopt(argc, argv, "h")) != -1) {
	switch(c) {
	  case 'h':	// Help
	  default:
	    err = 1;
	    break;
	}
    }
    if (optind+1 == argc) {	// One optional filename
	if (! in.openFile(argv[optind])) {
	    fprintf(stderr, "Could not open file %s\n", argv[optind]);
	    err = 1;
	}
    }
    else if (optind == argc) {	// No filename
#ifdef _WIN32
	if (_isatty(_fileno(stdin))) {
#else
	if (isatty(fileno(stdin))) {
#endif
	    fprintf(stderr, "Trying to read from standard input, "
	            "but standard input is a tty!\n");
	    err = 1;
	}
    }
    else err = 1;
    if (err) {
	printUsage();
	exit(1);
    }
}
int
main(int argc, char *argv[])
{
    SoInput	in;
    SoNode	*root;
    SbBool	gotAny = FALSE;

    SoInteraction::init();

    parseArgs(argc, argv, in);

    while (TRUE) {

	if (! SoDB::read(&in, root)) {
	    fprintf(stderr, "bad data in input\n");
	    return 1;
	}

	if (root == NULL) {
	    if (! gotAny)
		printf("no data in input\n");
	    break;
	}

	root->ref();

	if (! gotAny) {
	    printf("data format is %s\n",
		   in.isBinary() ? "binary" : "ASCII");
	    gotAny = TRUE;
	}

	else
	    printf("-----------------------------------------------------\n");

	printf("number of nodes under root is %d\n",
	       countNodes(root));

	printInfo(root);

	root->unref();
    }

    return(0);
}


