/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

//
// Code for the edges class, that supports fast querying of the faces
// around an edge.
//

#include <assert.h>
#include "Faces.h"
#include "Edges.h"

EdgeDict::EdgeDict(int initial_size)
{
    assert(initial_size > 0);

    hash_table = new Edge[initial_size];
    for (int i = 0; i < initial_size; i++)
    {
	hash_table[i].v1 = hash_table[i].v2 = (-1);
	hash_table[i].faces.truncate(0);
    }
    n_hash = initial_size;
    n_edges = 0;
}

EdgeDict::~EdgeDict()
{
    delete[] hash_table;
}

void
EdgeDict::Add(Face *f, int32_t v1, int32_t v2)
{
    if (v1 > v2) { int32_t t = v1; v1 = v2; v2 = t; }

    if (v1 == v2)
    {
	fprintf(stderr, "Warning: Degenerate edge found"
		" (v1 = v2 = %d)\n", v1);
	return;
    }
    assert(f != NULL);

    Edge *e = Enter(v1, v2);
    assert(e != NULL);

    e->faces.append(f);
}

void
EdgeDict::OtherFaces(Face *f, int32_t v1, int32_t v2, FaceList &result)
{
    assert(f != NULL);

    result.truncate(0);	// Empty list

    if (v1 == v2) return;	// Degenerate edge, no neighbors

    Edge *e = Find(v1, v2);

    for (int i = 0; i < e->faces.getLength(); i++)
    {
	if (e->faces[i] != f) result.append(e->faces[i]);
    }
}

Edge *
EdgeDict::Enter(int32_t v1, int32_t v2)
{
    assert(v1 != v2);

    int32_t t;
    if (v1 > v2) { t = v1; v1 = v2; v2 = t; }

    Edge *e = Find(v1, v2);

    if (e != NULL) return e;

    // Create a new entry.
    // First, see if we need to dynamically increase the hash table's
    // size.
    ++n_edges;
    if (n_edges*2 >= n_hash) ReHash(n_hash*2);

    //
    // Look for an empty spot in the table
    //
    int hpos = HashFunc(v1, v2);
    while (hash_table[hpos].v1 != (-1))
    {
	hpos = (hpos+1) % n_hash;
    }

    hash_table[hpos].v1 = v1;
    hash_table[hpos].v2 = v2;
 
    return hash_table+hpos;
}

//
// These fairly large prime numbers were picked randomly.
//
int
EdgeDict::HashFunc(int32_t v1, int32_t v2)
{
    int result = (v1*1400627 + v2*79823) % n_hash;
    return (result > 0 ? result : -result);
}

void
EdgeDict::ReHash(int newsize)
{
    int i;
    int oldsize = n_hash;
    Edge *old_table = hash_table;

    hash_table = new Edge[newsize];
    for (i = 0; i < newsize; i++)
    {
	hash_table[i].v1 = hash_table[i].v2 = (-1);
	hash_table[i].faces.truncate(0);
    }
    n_hash = newsize;

    for (i = 0; i < oldsize; i++)
    {
	const Edge *old = old_table+i;
	if (old->v1 != (-1))
	{
	    Edge *n = Enter(old->v1, old->v2);
	    n->faces.copy(old->faces);
	}
    }
    delete[] old_table;
}


Edge *
EdgeDict::Find(int32_t v1, int32_t v2)
{
    int32_t t;
    if (v1 > v2) { t = v1; v1 = v2; v2 = t; }

    int hpos = HashFunc(v1, v2);

    while (1)
    {
	Edge *e = hash_table+hpos;
	if (e->v1 == (-1)) return NULL;
	if (e->v1 == v1 && e->v2 == v2) return e;

	hpos = (hpos+1)%n_hash;	// Look in next bucket
    }
}


