/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

//
// Takes a scene graph and generates normals for it
//

#include <assert.h>
#include <math.h>

#include <Inventor/SoDB.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoIndexedFaceSet.h>
#include <Inventor/nodes/SoNormal.h>
#include <Inventor/nodes/SoNormalBinding.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/actions/SoSearchAction.h>

#include "FindNormals.h"
#include "Edges.h"

FindNormals::FindNormals()
{
    defaultOrientation = Face::UNKNOWN;
    creaseAngle = static_cast<float>(M_PI/6);
    verbose = FALSE;
}

FindNormals::~FindNormals()
{
}

void
FindNormals::AssumeOrientation(FaceOrientation o)
{
    defaultOrientation = o;
}    

void
FindNormals::apply(SoNode *root, int doVertexNormals)
{
    //
    // IndexedFaceSets:
    // First, search for them:
    //
    if (verbose)
	fprintf(stderr, "FindNormals: searching for IndexedFaceSets.\n");
    SoSearchAction sa;
    sa.setType(SoIndexedFaceSet::getClassTypeId());
    sa.setInterest(SoSearchAction::ALL);
    sa.apply(root);
    SoPathList &paths = sa.getPaths();
    if (verbose) fprintf(stderr, "FindNormals: searching finished.\n");
    for (int i=0; i < paths.getLength(); i++) {
	if (verbose && !(i%50))
	    fprintf(stderr, "FindNormals: faceSet %d of %d\n",
			    i+1, paths.getLength());

	SoGroup *parent = (SoGroup*)paths[i]->getNodeFromTail(1);

	// disable notification, or we'll have n^2 time spent notifying
	// the entire path list for every change.
#ifndef INVENTOR1
	SbBool pNot = parent->enableNotify(FALSE);
#else
	parent->setNotification(FALSE);
#endif

	doIndexedFaceSet(paths[i], doVertexNormals);

	// restore notification
#ifndef INVENTOR1
	parent->enableNotify(pNot);
#else
	parent->setNotification(TRUE);
#endif
    }
}

SoNode *
FindNormals::searchLastType(SoPath *p, SoType t)
{
    SoSearchAction sa;
    sa.setType(t);
    sa.setInterest(SoSearchAction::LAST);
    sa.apply(p);
    SoPath *foundPath = sa.getPath();

    if (foundPath != NULL)
	return foundPath->getTail();

    return NULL;
}

void
FindNormals::doIndexedFaceSet(SoPath *p, int doVertexNormals)
{
    // If there is already a normal node, bail

    if (searchLastType(p, SoNormal::getClassTypeId()) != NULL)
	return;

    //
    // First, find Coordinates in path:
    //
    SoCoordinate3 *coords = (SoCoordinate3 *)
	searchLastType(p, SoCoordinate3::getClassTypeId());

    SoIndexedFaceSet *ifs = (SoIndexedFaceSet *)p->getTail();

    assert(coords != NULL);

    FaceList faces(coords->point.getValues(0), ifs);

    if (defaultOrientation == Face::UNKNOWN)
    {
	faces.findOrientation();
    }
    else
    {
	for (int i = 0; i < faces.getLength(); i++)
	{
	    faces[i]->orientation = defaultOrientation;
	}
	faces.correctOrientation();
    }

    SoNormal *n = new SoNormal;
    SoNormalBinding *normb = new SoNormalBinding;

    SoNode *tn = p->getNode(p->getLength()-2);
    assert(tn->isOfType(SoGroup::getClassTypeId()));

    SoGroup *g = (SoGroup *)tn;
    
    int index = g->findChild(ifs);

    assert(index != -1);

    g->insertChild(normb, index);
    g->insertChild(n, index);

    faces.findShapeInfo();
    if (faces.isSolid()) {
	SoShapeHints *hints = new SoShapeHints;
	hints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
	hints->shapeType = SoShapeHints::SOLID;
	g->insertChild(hints, index);
    }

    if (doVertexNormals)
    {
	faces.findVertexNormals(n, ifs, creaseAngle);
	normb->value.setValue(SoNormalBinding::PER_VERTEX_INDEXED);
    }
    else
    {
	faces.findFacetNormals(n);
	normb->value.setValue(SoNormalBinding::PER_FACE);
    }
}


