/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

//
//______________________________________________________________________
//_____________  S I L I C O N   G R A P H I C S   I N C .  ____________
//
//   ivnorm
//
//      This is the mainline for the IRIS Inventor `ivnorm' program.
//   This program reads an Inventor data file, finds all of the 
//   indexed face set nodes, generates normals for each, and writes the
//   result in tact (it does not remove any properties or hierarchy)
//   to a new file.
//
//   Author(s)          : Gavin Bell
//
//   Notes:
//
//_____________  S I L I C O N   G R A P H I C S   I N C .  ____________
//______________________________________________________________________

#include <stdio.h>

#ifdef _WIN32
#  include <io.h> 
#  include "getopt/getopt.h"
#else
#  include <unistd.h>
#include <unistd.h>
#include <getopt.h>
#endif


#include <math.h>
#include <assert.h>

#include <Inventor/SoInteraction.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/helpers/SbFileHelper.h>

#include "FindNormals.h"

#ifdef _WIN32
#  define fileno _fileno
#  define isatty _isatty
#endif

Face::Orientation orientation = Face::UNKNOWN;
float 	    creaseAngle = (float)(M_PI / 6.0);		// 30 degrees
int 	    findVNorms  = 0;
int	    verbose = 0;
char	*inFileName = (char *)"stdin";
char	*outFileName = (char *)"stdout";

//-----------------------------------------------------------------------------
//
// My standard command-line parser for tools that can take two
// filenames or act as a filter.
//
//-----------------------------------------------------------------------------
static void
ParseCommandLine(int argc, char **argv, FILE **in, FILE **out)
{
  int err;	/* Flag: error in options? */
  int c;
  *in = stdin;
  *out = stdout;
  err = 0;
  
  while ((c = getopt(argc, argv, "cCva:bVh")) != -1) {
    switch(c) {
    case 'c':
      orientation = Face::CCW;
      break;
    case 'C':
      orientation = Face::CW;
      break;
    case 'v':
      findVNorms = TRUE;
      break;
    case 'a':
      findVNorms = TRUE;
      // convert from given arg (in degrees) to radians
      creaseAngle = (float)(atof(optarg) * M_PI / 180.0);
      break;
    case 'V':
      verbose = TRUE;
      break;
    case 'h':	/* Help */
    default:
      err = 1;
      break;
    }
  }
  /* Handle optional filenames */
  for (; optind < argc; optind++) {
    if (*in == stdin) {
      inFileName = argv[optind];
      *in = SbFileHelper::open(inFileName, "r");
      if (*in == NULL) {
	char buf[500];
	sprintf(buf, "%s: %s", argv[0], inFileName);
	perror(buf);
	exit(99);
      }
    }
    else if (*out == stdout) {
      outFileName = argv[optind];
      *out = fopen(outFileName, "w");
      if (*out == NULL) {
	char buf[500];
	sprintf(buf, "%s: %s", argv[0], outFileName);
	perror(buf);
	exit(99);
      }
    }
    else err = 1;	/* Too many arguments */
  }
  
  /* If stdin is a terminal, spit out usage */
  if (*in == stdin && isatty(fileno(stdin))) err = 1;
  
  if (err) {
    (void)fprintf(stderr, "Usage: %s [options]", argv[0]);
    (void)fprintf(stderr, " [infile] [outfile]\n");
    (void)fprintf(stderr, "-c        Assume counter-clockwise faces\n");
    (void)fprintf(stderr, "-C        Assume clockwise faces\n");
    (void)fprintf(stderr, "-v        Find vertex normals\n");
    (void)fprintf(stderr, "-a angle  Use angle (in degrees) as crease angle\n");
    (void)fprintf(stderr, "-V        verbose trace\n");
    (void)fprintf(stderr, "-h        This message (help)\n");
    exit(99);
  }
}


int
main(int argc, char **argv)
{
  FILE *fin;
  FILE *fout;
  
  SoInteraction::init();
  
  ParseCommandLine(argc, argv, &fin, &fout);
  
  SoNode *node = NULL;
  SbBool ok;
  
  SoInput in;
  in.setFilePointer(fin);
  
  SoOutput out;
  out.setFilePointer(fout);
  SoWriteAction wa(&out);
  
  FindNormals normalFinder;
  normalFinder.AssumeOrientation(orientation);
  normalFinder.setCreaseAngle(creaseAngle);
  normalFinder.setVerbose(verbose);
  
  for (;;) {
    if (verbose) 
      fprintf(stderr, "%s: reading graph from '%s'.\n", argv[0], inFileName);
    ok = SoDB::read(&in, node);
    if (verbose) 
      fprintf(stderr, "%s: finished reading graph.\n", argv[0]);
    if (!ok || !node)
      break;
    
    node->ref();
    // Generate normals
    normalFinder.apply(node, findVNorms);
    
    // Write out the (modified) scene graph
    if (verbose) 
      fprintf(stderr, "%s: writing graph to '%s'.\n", argv[0], outFileName);
    out.setBinary(in.isBinary());
    wa.apply(node);
    if (verbose) 
      fprintf(stderr, "%s: finished writing graph.\n", argv[0]);
  }
  
  if (!ok) {
    fprintf(stderr, "Bad data.\n");
    return 1;
  }
  
  return 0;
}

