/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Ajay Sreekanth (MMM yyyy)
**=======================================================================*/

#ifdef _WIN32
#  include <windows.h>
#  include "getopt/getopt.h"
#else
#  include <getopt.h>
#endif

#include <Inventor/sys/SoGLU.h>
#include <Inventor/sys/SoGLX.h>
#include <stdio.h>

#include <Inventor/Sb.h>
#include <Inventor/SbElapsedTime.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoInteraction.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/actions/SoActions.h>
#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/actions/SoSubAction.h>
#include <Inventor/fields/SoSFTime.h>
#include <Inventor/misc/SoChildList.h>
#include <Inventor/nodekits/SoBaseKit.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFile.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoPackedColor.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTransform.h>

// default window size
#define WINDOW_X 400        
#define WINDOW_Y 400        

// number of frame to draw before autocaching kicks in
// typically needs only 2 but we use 4 just in case
#define NUM_FRAMES_AUTO_CACHING        8

// default number of frames 
#define NUM_FRAMES 120

// the name used in the scene graph to indicate that the application
// will modify something below this separator
// the children of these nodes are touched to destroy the caches there
#define NO_CACHE_NAME "NoCache"

// the name of the transformation node added to spin the scene
#define SCENE_XFORM_NAME "SCENE_XFORM_#####"

struct Options {
  // fields initialized by the user
  SbBool showBars;
  int numFrames;
  const char *inputFileName;
  unsigned int windowX, windowY;
  // fields set based on structure of scene graph
  SbBool hasLights;
  SbBool hasTextures;
  // fields used internally
  SbBool noClear;
  SbBool noMaterials;
  SbBool noTextures;
  SbBool oneTexture;
  SbBool noXforms;
  SbBool noFill;
  SbBool noVtxXforms;
  SbBool noLights;
  SbBool freeze;
  // fields for testing
  SbBool showThreshold;
  int threshold;
};

#ifdef WIN32
#define Display UINT*
#endif

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Callback used to count triangles.
//

static void
countTriangleCB(void *userData, SoCallbackAction *,
                const SoPrimitiveVertex *,
                const SoPrimitiveVertex *,
                const SoPrimitiveVertex *)
{
  int32_t *curCount = (int32_t *)userData;

  (*curCount)++;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Callback used to count Line Segments.
//

static void
countLinesCB(void *userData, SoCallbackAction *,
             const SoPrimitiveVertex *,
             const SoPrimitiveVertex *)
{
  int32_t *curCount = (int32_t *)userData;

  (*curCount)++;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Callback used to count points.
//

static void
countPointsCB(void *userData, SoCallbackAction *,
              const SoPrimitiveVertex *)
{
  int32_t *curCount = (int32_t *)userData;

  (*curCount)++;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Callback used to count nodes.
//

#ifdef _WIN32
// I don't know why we had to change the return type for Win32... -mmh
static int
#else
static SoCallbackAction::Response
#endif
countNodesCB(void *userData, SoCallbackAction *, const SoNode *)
{
  int32_t *curCount = (int32_t *) userData;

  (*curCount)++;

  return SoCallbackAction::CONTINUE;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Callback used to count text2.
//

#ifdef _WIN32
// I don't know why we had to change the return type for Win32... -mmh
static int
#else
static SoCallbackAction::Response
#endif
countText2CB(void *userData, SoCallbackAction *, const SoNode *)
{
  int32_t *curCount = (int32_t *) userData;

  (*curCount)++;

  return SoCallbackAction::CONTINUE;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Counts triangles/lines/points in given graph using primitive generation,
//    returning total.
//

void
countPrimitives(SoNode *root, int32_t &numTris, int32_t &numLines, 
                int32_t &numPoints, int32_t &numNodes, int32_t &numText2)
{
  numTris = 0;
  numLines = 0;
  numPoints = 0;
  numNodes = 0;
  numText2 = 0;

  SoCallbackAction ca;

  ca.addPointCallback(SoShape::getClassTypeId(),
                      countPointsCB,
                      (void *) &numPoints);
  ca.addLineSegmentCallback(SoShape::getClassTypeId(),
                            countLinesCB,
                            (void *) &numLines);
  ca.addTriangleCallback(SoShape::getClassTypeId(),
                         countTriangleCB,
                         (void *) &numTris);
  ca.addPreCallback(SoNode::getClassTypeId(),
                    countNodesCB, 
                    (void *)&numNodes);
  ca.addPreCallback(SoText2::getClassTypeId(),
                    countText2CB, 
                    (void *)&numText2);
  ca.apply(root);
}

//////////////////////////////////////////////////////////////
//
// Description:
//    Parses command line arguments, setting options.
//

static SbBool
parseArgs(int argc, char *argv[], Options &options)
{
  SbBool ok = TRUE;
  int c, curArg;

  // Initialize options
  options.showBars = FALSE;
  options.numFrames = NUM_FRAMES;
  options.inputFileName = NULL;
  options.windowX = WINDOW_X;
  options.windowY = WINDOW_Y;
  options.hasLights = FALSE;
  options.hasTextures = FALSE;
  options.noClear = FALSE;
  options.noMaterials = FALSE;
  options.noTextures = FALSE;
  options.oneTexture = FALSE;
  options.noXforms = FALSE;
  options.noFill = FALSE;
  options.noVtxXforms = FALSE;
  options.freeze = FALSE;
  options.noLights = FALSE;
  options.showThreshold = FALSE;
  options.threshold = 500;

  while ((c = getopt(argc, argv, "bf:w:t:")) != -1) {
    switch (c) {
    case 'b':
      options.showBars = TRUE;
      break;
    case 'f':
      options.numFrames = atoi(optarg);
      break;
    case 'w':
      sscanf(optarg, " %d , %d", &options.windowX, &options.windowY);
      break;
    case 't':
      options.showThreshold = TRUE;
      options.threshold = atoi(optarg);
      break;
    default:
      ok = FALSE;
      break;
    }
  }

  curArg = optind;

  // Check for input filename at end
  if (curArg < argc)
    options.inputFileName = argv[curArg++];

  // Not enough or extra arguments? Error!
  if (options.inputFileName == NULL || curArg < argc)
    ok = FALSE;

  // Report options and file names
  if (ok) {
    printf("\n");
    if ( !SoPreferences::getBool("ATT_RUNNING", FALSE) )
      printf("Reading graph from %s\n", options.inputFileName);
    printf("Number of frames: %d\n", options.numFrames);
    printf("Window size: %d x %d pixels\n", 
           options.windowX, options.windowY);
    printf("\n");
  }

  return ok;
}

//////////////////////////////////////////////////////////////
//
// Description:
//    Callback used by openWindow() and by main()
//

#ifndef _WIN32
static Bool
waitForNotify(Display *, XEvent *e, char *arg)
{
  return ((e->type == MapNotify) || (e->type == UnmapNotify)) &&
    (e->xmap.window == (Window) arg);
}
#else // _WIN32
#define ERROR_RETURN(funcname) {\
  DWORD error;                  \
  error = GetLastError();       \
  fprintf(stderr, "*** %s failed, error = %d (0x%x)\n", \
          funcname, error, error);                      \
  return;                       \
}

////////////////////////////////////////////////////////////////////////
LRESULT CALLBACK WndProc(HWND hwnd, UINT message,
                         WPARAM wParam, LPARAM lParam)
{
  if (SoWin::isInventorMessage(hwnd, message, wParam, lParam))
    return TRUE;
  
  switch (message) {
  case WM_DESTROY:
    PostQuitMessage(0);
    return 0;
  }
  return DefWindowProc(hwnd, message, wParam, lParam);
}
#endif // _WIN32

//////////////////////////////////////////////////////////////
//
// Description:
//    Creates and initializes GL/X window.
//

static void
openWindow(Display *&display, Window &window, unsigned int windowX,
           unsigned int windowY, char *title)
{
#ifdef _WIN32
  // Register class for OpenGL window
  WNDCLASS wndclass;
  wndclass.style = CS_HREDRAW | CS_VREDRAW;
  wndclass.lpfnWndProc = WndProc;
  wndclass.cbClsExtra = 0;
  wndclass.cbWndExtra = 0;
  wndclass.hInstance = GetModuleHandle(NULL);
  wndclass.hIcon = LoadIcon(NULL, IDI_APPLICATION);
  wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
  wndclass.hbrBackground = NULL;
  wndclass.lpszMenuName = NULL;
  wndclass.lpszClassName = "IVPERF Class";
    
  ATOM result = RegisterClass(&wndclass);
  if (result == 0) 
    ERROR_RETURN("RegisterClass");

  // Create window for OpenGL queries
  // Note: WS_CLIPxxx are required for MS-OpenGL to work...
  HMODULE hApp = GetModuleHandle(NULL);
  HWND hwnd = CreateWindow("IVPERF Class",
                           title,
                           WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
                           10,                    // X position
                           10,                    // Y position
                           windowX,               // Width
                           windowY,               // Height
                           NULL,                  // Parent
                           NULL,                  // Menu
                           hApp,                  // App instance handle
                           NULL);                 // Window creation data
  if (hwnd == NULL) 
    ERROR_RETURN("CreateWindow");

  // Make window visible
  ShowWindow(hwnd, SW_SHOWNORMAL);

  // Get device context for OpenGL window
  HDC hdc = GetDC(hwnd);

  // Setup pixel format descriptor
  // Shouldn't matter exactly what we use here, but these are the
  // default values used by Inventor.
  PIXELFORMATDESCRIPTOR pfd;
  pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
  pfd.nVersion = 1;
  pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL; //| PFD_DOUBLEBUFFER;
  pfd.iPixelType = PFD_TYPE_RGBA;
  pfd.cColorBits = 24;
  pfd.cRedBits = 0;
  pfd.cRedShift = 0;
  pfd.cGreenBits = 0;
  pfd.cGreenShift = 0;
  pfd.cBlueBits = 0;
  pfd.cBlueShift = 0;
  pfd.cAlphaBits = 0;
  pfd.cAlphaShift = 0;
  pfd.cAccumBits = 0;
  pfd.cAccumRedBits = 0;
  pfd.cAccumGreenBits = 0;
  pfd.cAccumBlueBits = 0;
  pfd.cAccumAlphaBits = 0;
  pfd.cDepthBits = 32;
  pfd.cStencilBits = 0;
  pfd.cAuxBuffers = 0;
  pfd.iLayerType = PFD_MAIN_PLANE;
  pfd.bReserved = 0;
  pfd.dwLayerMask = 0;
  pfd.dwVisibleMask = 0;
  pfd.dwDamageMask = 0;

  // Get a pixel format
  int pixelformat;
  if ((pixelformat = ChoosePixelFormat(hdc, &pfd)) == 0)
    ERROR_RETURN("ChoosePixelFormat");

  // Set the pixel format
  if (SetPixelFormat(hdc, pixelformat, &pfd) == FALSE)
    ERROR_RETURN("SetPixelFormat");

  // Create an OpenGL render context
  HGLRC ctx = wglCreateContext(hdc);
  if (ctx == NULL) ERROR_RETURN("wglCreateContext");

  SoGLContext* context = new SoGLContext(hdc, &pfd, NULL, ctx);
  context->ref();
  context->bind();

  window = hwnd;

#else //Motif
  XVisualInfo *vi;
  Colormap cmap;
  XSetWindowAttributes swa;
  GLXContext cx;
  XEvent event;
  static int attributeList[] = {
    GLX_RGBA,
    GLX_RED_SIZE, 1,
    GLX_GREEN_SIZE, 1,
    GLX_BLUE_SIZE, 1,
    GLX_DEPTH_SIZE, 1,
    None,                // May be replaced w/GLX_DOUBLEBUFFER
    None
  };

  display = XOpenDisplay(0);
  vi = glXChooseVisual(display,
                       DefaultScreen(display),
                       attributeList);
  cx = glXCreateContext(display, vi, 0, GL_TRUE);
  cmap = XCreateColormap(display,
                         RootWindow(display, vi->screen),
                         vi->visual, AllocNone);
  swa.colormap = cmap;
  swa.border_pixel = 0;
  swa.event_mask = StructureNotifyMask;
  window = XCreateWindow(display,
                         RootWindow(display, vi->screen),
                         10, 10, windowX, windowY,
                         0, vi->depth, InputOutput, vi->visual,
                         (CWBorderPixel | CWColormap | CWEventMask), &swa);

  // Make the window appear in the lower left corner
  XSizeHints *xsh = XAllocSizeHints();
  xsh->flags = USPosition;
  XSetWMNormalHints(display, window, xsh);
  XSetStandardProperties(display, window, title, title, None, 0, 0, 0);
  XFree(xsh);

  XMapWindow(display, window);
  XIfEvent(display, &event, waitForNotify, (char *) window);

  SoGLContext* context = new SoGLContext(display, vi, window, cx);
  context->ref();
  context->bind();
#endif //_WIN32

  glEnable(GL_DEPTH_TEST);
  glClearColor(0.5, 0.5, 0.5, 1);
    
  // clear the graphics window and depth planes
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

//////////////////////////////////////////////////////////////
//
// Description:
//    Creates and returns scene graph containing given
//    scene. Adds a perspective camera, a directional
//    light, and a transform.
//    Returns NULL on error.
//

static SoSeparator *
setUpGraph(const SbViewportRegion &vpReg,
           SoInput *sceneInput,
           Options &options)
{
  // Create a root separator to hold everything. Turn
  // caching off, since the transformation will blow
  // it anyway.
  SoSeparator *root = new SoSeparator;
  root->ref();

  // Add a camera to view the scene
  SoPerspectiveCamera *camera = new SoPerspectiveCamera;
  root->addChild(camera);

  // Add a transform node to spin the scene
  SoTransform *sceneTransform = new SoTransform;
  sceneTransform->setName(SCENE_XFORM_NAME);
  root->addChild(sceneTransform);

  // Read and add input scene graph
  SoSeparator *inputRoot = SoDB::readAll(sceneInput);
  if (inputRoot == NULL) {
    fprintf(stderr, "Cannot read scene graph\n");
    root->unref();
    exit(1);
  }
  root->addChild(inputRoot);

  SoPath *path;
  SoGroup *parent, *group;
  SoSearchAction act;

  // expand out all File nodes and replace them with groups
  //    containing the children
  SoFile *fileNode;
  act.setType(SoFile::getClassTypeId());
  act.setInterest(SoSearchAction::FIRST);
  act.apply(inputRoot);
  while ((path = act.getPath()) != NULL) {
    fileNode = (SoFile *) path->getTail();
    path->pop();
    parent = (SoGroup *) path->getTail();
    group = fileNode->copyChildren();
    if (group) {
      parent->replaceChild(fileNode, group);
      // apply action again and continue
      act.apply(inputRoot);
    }
  }

  // expand out all node kits and replace them with groups
  //    containing the children
  SoBaseKit *kitNode;
  SoChildList *childList;
  act.setType(SoBaseKit::getClassTypeId());
  act.setInterest(SoSearchAction::FIRST);
  act.apply(inputRoot);
  while ((path = act.getPath()) != NULL) {
    kitNode = (SoBaseKit *) path->getTail();
    path->pop();
    parent = (SoGroup *) path->getTail();
    group = new SoGroup;
    childList = kitNode->getChildren();
    for (int i=0; i<childList->getLength(); i++) 
      group->addChild((*childList)[i]);
    parent->replaceChild(kitNode, group);
    act.apply(inputRoot);
  }

  // check to see if there are any lights
  // if no lights, add a directional light to the scene
  act.setType(SoLight::getClassTypeId());
  act.setInterest(SoSearchAction::FIRST);
  act.apply(inputRoot);
  if (act.getPath() == NULL) { // no lights
    SoDirectionalLight *light = new SoDirectionalLight;
    root->insertChild(light, 1);
  }
  else 
    options.hasLights = TRUE;

  // check to see if there are any texures in the scene
  act.setType(SoTexture2::getClassTypeId());
  act.setInterest(SoSearchAction::FIRST);
  act.apply(inputRoot);
  if (act.getPath() != NULL)
    options.hasTextures = TRUE;

  camera->viewAll(root, vpReg);

  // print out information about the scene graph

  int32_t numTris, numLines, numPoints, numNodes, numText2;
  countPrimitives(inputRoot, numTris, numLines, numPoints, numNodes, numText2);
  printf("Number of nodes in scene graph:     %d\n", numNodes);
//  printf("Number of text2D in scene graph:    %d\n", numText2);
  printf("Number of triangles in scene graph: %d\n", numTris);
  printf("Number of lines in scene graph:     %d\n", numLines);
  printf("Number of points in scene graph:    %d\n\n", numPoints);

  // Make the center of rotation the center of
  // the scene
  SoGetBoundingBoxAction bba(vpReg);
  bba.apply(root);
  sceneTransform->center = bba.getBoundingBox().getCenter();

  return root;
}


//////////////////////////////////////////////////////////////
//
// Description:
//    Replaces all group nodes in the graph with new ones
//    to reset autocaching threshold in separators.  Recursive.
//

SoNode *
replaceSeparators(SoNode *root)
{
  //
  // if it's a group, make a new group and copy its
  //  children over
  //
  if (root->isOfType(SoGroup::getClassTypeId())) {
    SoGroup *group = (SoGroup *) root;
    SoGroup *newGroup = (SoGroup *) group->getTypeId().createInstance();
    newGroup->SoNode::copyContents(group, FALSE);

    int i;
    for (i = 0; i < group->getNumChildren(); i++) {
      SoNode *child = replaceSeparators(group->getChild(i));
      newGroup->addChild(child);
    }
    return newGroup;
  }
  //
  // if not a group, return the node
  //
  else 
    return root;
}

//////////////////////////////////////////////////////////////
//
// Description:
//    Removes nodes of the given type from the given graph.
//

void
removeNodes(SoGroup *root, SoType type)
{
  SoSearchAction act;
  act.setInterest(SoSearchAction::ALL);
  act.setType(type);
  act.apply(root);
  SoPathList &paths = act.getPaths();
  for (int i = 0; i < paths.getLength(); i++) {
    SoNode *kid = paths[i]->getTail();
    paths[i]->pop();
    SoGroup *parent = (SoGroup *)paths[i]->getTail();
    parent->removeChild(kid);
  }
}

//////////////////////////////////////////////////////////////
//
// Description:
//    Prints usage message.
//

static void
printUsage(const char *progName)
{
  fprintf(stderr,
          "Usage: %s [-b] [-f N] [-w X,Y] datafile\n",
          progName);
  fprintf(stderr,
          "\t-b      display results as bar chart\n"
          "\t-f N    render N frames for each test\n"
          "\t-w X,Y  make window size X by Y pixels\n");
}

//////////////////////////////////////////////////////////////
//
// Description:
//    Returns the time taken PER TRAVERSAL to compute teh bbox 
//    graph anchored at root
static double
bboxComputing(Options &options,
              const SbViewportRegion &vpr,
              SoSeparator *&root)
{
  double timeDiff;
  SbElapsedTime startTime;
  int frameIndex;
  SoTransform *sceneTransform;
  SoGetBoundingBoxAction bba(vpr);
  SoNodeList noCacheList;
  SoSeparator *newRoot;

  //
  // reset autocaching threshold before each experiment
  //   done by replacing every separator in the scene graph
  //   with a new one
  //
  newRoot = (SoSeparator *)replaceSeparators(root);
  newRoot->ref();
  //newRoot->boundingBoxCaching = SoSeparator::OFF;

  // get a list of separators marked as being touched by the application
  newRoot->getByName(NO_CACHE_NAME, noCacheList);

  // find the transform node that spins the scene
  SoNodeList xformList;
  newRoot->getByName(SCENE_XFORM_NAME, xformList);
  sceneTransform = (SoTransform *) xformList[0];

  for (frameIndex = 0; ; frameIndex++) {

    // wait till autocaching has kicked in then start timing
    if (frameIndex == NUM_FRAMES_AUTO_CACHING)
      startTime.reset();

    // stop timing and exit loop when requisite number of
    //    frames have been drawn
    if (frameIndex == options.numFrames + NUM_FRAMES_AUTO_CACHING) {
      glFinish();
      timeDiff = startTime.getElapsed();
      break;
    }
            
    // if not frozen, update realTime and destroy labelled caches
    if (! options.freeze) { 

      // update realTime 
      SoSFTime *realTime = (SoSFTime *) SoDB::getGlobalField("realTime");
      realTime->setValue(SbTime::getTimeOfDay());

      // touch the separators marked NoCache 
      for (int i=0; i<noCacheList.getLength(); i++)
        ((SoSeparator *) noCacheList[i])->getChild(0)->touch();
    }

    // Rotate the scene
    sceneTransform->rotation.setValue(SbVec3f(1, 1, 1), 
                                      (float)(frameIndex * 2 * M_PI / options.numFrames));
    bba.apply(newRoot);
  }

  // Get rid of newRoot
  newRoot->unref();

  return (double)(timeDiff / options.numFrames);
}

//////////////////////////////////////////////////////////////
//
// Description:
//    Returns the time taken PER FRAME to render the scene 
//    graph anchored at root
//    Different types of rendering performance are measured
//    depending on options
//    

static double
timeRendering(Options &options,
              const SbViewportRegion &vpr,
              SoSeparator *&root)
{
  double timeDiff;
  SbElapsedTime startTime;
  int frameIndex;
  SoTransform *sceneTransform;
  SoGLRenderAction ra(vpr);
  SoNodeList noCacheList;
  SoSeparator *newRoot;

  // clear the window
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  //
  // reset autocaching threshold before each experiment
  //   done by replacing every separator in the scene graph
  //   with a new one
  //
  newRoot = (SoSeparator *)replaceSeparators(root);
  newRoot->ref();

  // get a list of separators marked as being touched by the application
  newRoot->getByName(NO_CACHE_NAME, noCacheList);

  // find the transform node that spins the scene
  SoNodeList xformList;
  newRoot->getByName(SCENE_XFORM_NAME, xformList);
  sceneTransform = (SoTransform *) xformList[0];

  if (options.noMaterials) {  // nuke material node
    removeNodes(newRoot, SoMaterial::getClassTypeId());
    removeNodes(newRoot, SoPackedColor::getClassTypeId());
    removeNodes(newRoot, SoBaseColor::getClassTypeId());
  }

  if (options.noXforms) {  // nuke transforms
    removeNodes(newRoot, SoTransformation::getClassTypeId());
  }

  if (options.noTextures || options.oneTexture) {  // override texture node

    removeNodes(newRoot, SoTexture2::getClassTypeId());

    if (options.oneTexture) {
      // texture node with simple texture
      static unsigned char img[] = {
        255, 255, 0, 0,
        255, 255, 0, 0,
        0, 0, 255, 255,
        0, 0, 255, 255
      };
      SoTexture2 *overrideTex = new SoTexture2;        
      overrideTex->image.setValue(SbVec2s(4, 4), 1, img);
      newRoot->insertChild(overrideTex, 1);
    }
  }

  if (options.noFill) {  // draw as points
    SoDrawStyle *overrideFill = new SoDrawStyle;
    overrideFill->style.setValue(SoDrawStyle::POINTS);
    overrideFill->lineWidth.setIgnored(TRUE);
    overrideFill->linePattern.setIgnored(TRUE);
    overrideFill->setOverride(TRUE);
    newRoot->insertChild(overrideFill, 0);

    // cull backfaces so that extra points don't get drawn
    SoShapeHints *cullBackfaces = new SoShapeHints;
    cullBackfaces->shapeType = SoShapeHints::SOLID;
    cullBackfaces->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
    cullBackfaces->setOverride(TRUE);
    newRoot->insertChild(cullBackfaces, 0);
  }

  if (options.noVtxXforms) {  // draw invisible
    SoDrawStyle *overrideVtxXforms = new SoDrawStyle;
    overrideVtxXforms->style.setValue(SoDrawStyle::INVISIBLE);
    overrideVtxXforms->setOverride(TRUE);
    newRoot->insertChild(overrideVtxXforms, 0);
  }

  if (options.noLights) {  // set lighting model to base color
    SoLightModel *baseColor = new SoLightModel;
    baseColor->model = SoLightModel::BASE_COLOR;
    newRoot->insertChild(baseColor, 0);
  }
 
  for (frameIndex = 0; ; frameIndex++) {

    // wait till autocaching has kicked in then start timing
    if (frameIndex == NUM_FRAMES_AUTO_CACHING)
    {
      startTime.reset();
    }

    // stop timing and exit loop when requisite number of
    //    frames have been drawn
    if (frameIndex == options.numFrames + NUM_FRAMES_AUTO_CACHING) {
      glFinish();
      timeDiff = startTime.getElapsed();
      break;
    }
            
    // if not frozen, update realTime and destroy labelled caches
    if (! options.freeze) { 

      // update realTime 
      SoSFTime *realTime = (SoSFTime *) SoDB::getGlobalField("realTime");
      realTime->setValue(SbTime::getTimeOfDay());

      // touch the separators marked NoCache 
      for (int i=0; i<noCacheList.getLength(); i++)
        ((SoSeparator *) noCacheList[i])->getChild(0)->touch();
    }

    // Rotate the scene
    sceneTransform->rotation.setValue(SbVec3f(1, 1, 1), 
                                      (float)(frameIndex * 2 * M_PI / options.numFrames));

    if (! options.noClear)
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    ra.apply(newRoot);
  }

  // Get rid of newRoot
  newRoot->unref();

  return (double)(timeDiff / options.numFrames);
}

void
displayResult(const char *label, double s, Options &options)
{
  double fps = 1.0/s;
  if (strlen(label) > 20)
  {
    if (options.showThreshold)
      printf("%34s:\t  %s %4d\n", label,
             (fps > (double)options.threshold ? ">" : "<"), options.threshold);
    else
      printf("%34s:\t%10.0f\n", label, fps);
    
  }
  else
  {
    if (options.showThreshold)
      printf("%20s:\t   %s %4d fps\n", label,
             (fps > (double)options.threshold ? ">" : "<"), options.threshold);
    else
      printf("%20s:\t%10.2f f/s\n", label, fps);
  }
}

//////////////////////////////////////////////////////////////
//
// Description:
//    Mainline
//
int
main(int argc, char **argv)
{
  Options options;
  SoSeparator *root;
  Display *display;
  Window window;
  double asisTime, ptsTime, invisTime, freezeTime,
    noMatTime, noTexTime, noLitTime, noXformTime,
    noClearTime, oneTexTime, bboxTime;

  // Init Inventor
  SoInteraction::init();

  // Parse arguments
  if (! parseArgs(argc, argv, options)) {
    printUsage(argv[0]);
    return 1;
  }
  SoFieldContainer::initCopyDict();

  // Open scene graphs
  SoInput sceneInput;
  if (!sceneInput.openFile(options.inputFileName)) {
    fprintf(stderr, "Cannot open %s\n", options.inputFileName);
    return 1;
  }

  SbViewportRegion vpr((int)options.windowX, (int)options.windowY);
  root = setUpGraph(vpr, &sceneInput, options);

  // Create and initialize window
  openWindow(display, window, options.windowX, options.windowY, argv[0]);

  // Timing tests
  printf("Timing tests:\n");

  // Bounding box computation with bbox cache
  bboxTime = bboxComputing(options, vpr, root);
  displayResult("BBox computing", bboxTime, options);

  // as is rendering
  asisTime = timeRendering(options, vpr, root);
  displayResult("As-Is rendering", asisTime, options);

  // time for rendering without clear
  options.noClear = TRUE;
  noClearTime = timeRendering(options, vpr, root);
  options.noClear = FALSE;
  displayResult("No Clear", noClearTime, options);
    
  // time for rendering without materials
  options.noMaterials = TRUE;
  noMatTime = timeRendering(options, vpr, root);
  options.noMaterials = FALSE;
  displayResult("No Materials", noMatTime, options);
    
  // time for rendering without xforms
  options.noXforms = TRUE;
  noXformTime = timeRendering(options, vpr, root);
  options.noXforms = FALSE;
  displayResult("No Transforms", noXformTime, options);

  if (options.hasTextures) { // do tests only if scene has textures

    // time for rendering without any textures
    options.noTextures = TRUE;
    noTexTime = timeRendering(options, vpr, root);
    options.noTextures = FALSE;
    displayResult("No Textures", noTexTime, options);

    // time for rendering without only one texture
    options.oneTexture = TRUE;
    oneTexTime = timeRendering(options, vpr, root);
    options.oneTexture = FALSE;
    displayResult("One Texture", oneTexTime, options);
  }
  else 
    noTexTime = oneTexTime = asisTime;


  // time for rendering without fill
  options.noFill = TRUE;
  ptsTime = timeRendering(options, vpr, root);
  options.noFill = FALSE;    
  displayResult("No Fills", ptsTime, options);

  if (options.hasLights) { // do test only if scene has lights
    // time for rendering with lights turned off
    options.noLights = TRUE;
    noLitTime = timeRendering(options, vpr, root);
    options.noLights = FALSE;
    displayResult("No Lights", noLitTime, options);
  }
  else {
    noLitTime = asisTime;
  }
  printf("\n");

  // time for rendering without vertex xforms
  options.noVtxXforms = TRUE;
  invisTime = timeRendering(options, vpr, root);
  options.noVtxXforms = FALSE;
  displayResult("Vertex transformations per second", ptsTime-invisTime, options);
    
  // time for rendering with scene graph frozen
  options.freeze = TRUE;
  freezeTime = timeRendering(options, vpr, root);
  options.freeze = FALSE;
  displayResult("Scene graph traversal per second", asisTime-freezeTime, options);

  // kill window
#ifdef _WIN32
  ShowWindow(window, SW_HIDE);
#else
  XEvent event;
  XUnmapWindow(display, window);
  XIfEvent(display, &event, waitForNotify, (char *) window);
#endif

  return 0;
}


