/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <Inventor/SbPList.h>
#include <Inventor/Qt/SoQt.h>

#include <Inventor/actions/SoWriteAction.h>

#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoIndexedFaceSet.h>
#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/nodes/SoIndexedPointSet.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoTriangleStripSet.h>
#include <Inventor/nodes/SoIndexedTriangleStripSet.h>
#include <Inventor/nodes/SoTriangleSet.h>
#include <Inventor/nodes/SoIndexedTriangleSet.h>
#include <Inventor/nodes/SoQuadMesh.h>
#include <Inventor/nodes/SoIndexedQuadMesh.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoTexture3.h>
#include <Inventor/nodes/SoTextureUnit.h>
#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/nodes/SoShaderParameter.h>
#include <Inventor/STL/sstream>

#include <ScaleViz/SoScaleViz.h>

#include <Inventor/devices/SoGpuBufferObject.h>
#include <Inventor/devices/SoCpuDevice.h>

#include <bencher.h>
#include <viewer.h>
#include <geometry.h>

#ifdef WIN32
# include <windows.h>
# include <psapi.h>
#else
# include <stdexcept>
# include <cmath>
# include <sys/time.h>
# include <sys/resource.h>
#endif

#if !defined(OUTPUT_DIR)
#  define OUTPUT_DIR ""
#endif


//------------------------------------------------------------------------------
unsigned long long
MemoryTool::current()
{
  SoCpuDevice* cpu = SoCpuDevice::getDevice();
  if ( cpu )
    return cpu->getProcessMemory( );
  return -1;
}


//------------------------------------------------------------------------------
unsigned long long
MemoryTool::peak()
{
  SoCpuDevice* cpu = SoCpuDevice::getDevice();
  if ( cpu )
    return cpu->getPeakProcessMemory( );
  return -1;
}

//------------------------------------------------------------------------------
BenchTool::BenchTool()
{
}


//------------------------------------------------------------------------------
BenchTool::~BenchTool()
{
}


//------------------------------------------------------------------------------
std::vector< BenchTool::BenchResult >
BenchTool::bench(std::string outputDir, Viewer* viewer, std::string benchString)
{
  SoDB::init();

  // 1) We decode the bench strings
  m_benchData.clear();
  m_benchResults.clear();

  decodeBenchString( benchString );

  for (size_t i = 0; i < m_benchData.size(); i++)
  {
    // Format the dump filename
    char filename[128];

    std::string texFilename;
    std::string ivFilename;

    std::string texBaseFilename;
    std::string ivBaseFilename;

    sprintf( filename, "output_%d", m_benchData[i].id );

    texBaseFilename = std::string( filename ) + std::string( ".png" );
    texFilename = outputDir + std::string( "//" ) + texBaseFilename;

    ivBaseFilename = std::string( filename ) + std::string( ".iv" );
    ivFilename = outputDir + std::string( "//" ) + ivBaseFilename;

    /*SbString file = outputDisplayFile;
    if (file == "")
    {
    int npos = benchString.find_first_of(" ");
    std::string shapename = benchString.substr(0, npos);
    std::string sub = benchString.substr(npos + 1);
    npos = sub.find_first_of(" ");
    std::string shapeconfig1 = sub.substr(0, npos);
    sub = sub.substr(npos + 1);
    npos = sub.find_first_of(" ");
    std::string shapeconfig2 = sub.substr(0, npos);
    file = shapename + "_"
    + shapeconfig1 + "_"
    + shapeconfig2 + "_"
    + m_benchData[i].shapeInfo.width + " "
    + m_benchData[i].shapeInfo.height + " "
    + m_benchData[i].shapesCount;
    }
    file.replace(" ", "_");
    file.replace(":", "_");
    file.replace("-", "_");
    file += ".png";*/

    performBench( ivFilename, true, texFilename, viewer, m_benchData[i] );
  }

  SoDB::finish();

  return m_benchResults;  
}

std::vector< BenchTool::BenchResult >
BenchTool::bench(std::string outputIvFile, bool outputDisplay,
                 std::string outputDisplayFile, Viewer* viewer,
                 BenchData benchData, std::string geometryType)
{
  SoDB::init();

  // 1) We decode the bench strings
  m_benchData.clear();
  m_benchResults.clear();

  buildGeometry( benchData, geometryType.c_str() );

  for (size_t i = 0; i < m_benchData.size(); i++)
    performBench( outputIvFile, outputDisplay, outputDisplayFile, viewer, m_benchData[i] );

  SoDB::finish();

  return m_benchResults;
}


//------------------------------------------------------------------------------
std::vector< BenchTool::BenchResult >
BenchTool::bench(std::string outputIvFile, bool outputDisplay,
                 std::string outputDisplayFile, Viewer* viewer,
                 std::string benchString)
{
  SoDB::init();

  // 1) We decode the bench strings
  m_benchData.clear();
  m_benchResults.clear();

  decodeBenchString( benchString );

  for (size_t i = 0; i < m_benchData.size(); i++)
  {
    SbString file = outputDisplayFile;
    if (file == "")
    {
      size_t npos = benchString.find_first_of(" ");
      std::string shapename = benchString.substr(0, npos);
      std::string sub = benchString.substr(npos + 1);
      npos = sub.find_first_of(" ");
      std::string shapeconfig1 = sub.substr(0, npos);
      sub = sub.substr(npos + 1);
      npos = sub.find_first_of(" ");
      std::string shapeconfig2 = sub.substr(0, npos);
      file = shapename + "_";

      if ( m_benchData[i].shapeInfo.neighborTolerance != .0f )
      {
        // Search for tolerance string parameter
        // Don't use the one in shapeInfo as it was computed by toFloat()
        for (int i = 0; i < 10; i++)
        {
          sub = sub.substr(npos + 1);
          npos = sub.find_first_of(" ");
        }
        file += "tol=";
        file += sub.substr(0, npos);
        file += " ";
      }
      file += shapeconfig1 + "_" + shapeconfig2 + "_";
      file += m_benchData[i].drawStyle == SoDrawStyle::FILLED ? "F" : m_benchData[i].drawStyle == SoDrawStyle::LINES ? "L" : "P";
      file += " ";
      file += m_benchData[i].shapeInfo.width;
      file += " ";
      file += m_benchData[i].shapeInfo.height;
      file += " ";
      file += m_benchData[i].shapesCount;
      file.replace(" ", "_");
      file.replace(":", "_");
    }

    SbString filePath = SbFileHelper::toUnixPath( SbFileHelper::getDirName( DEMO_ROOT_DIR ) );
    filePath.replace( ROOT_DIR, "" );
    SbString outputPath = SbFileHelper::toUnixPath(SbFileHelper::expandString(SbString(OUTPUT_DIR)) + filePath);

    file += ".png";	
    file = outputPath + "ivShapeBenchmark/benches/ivShapeBenchmark-" + file;

    performBench( outputIvFile, outputDisplay, file.getString(), viewer, m_benchData[i] );
  }
  SoDB::finish();

  return m_benchResults;
}

std::string
BenchTool::buildTestBunch(std::string flags, std::string bindingFlags,
                          std::string indexedBindingFlags,
                          int geomIndex,
                          int geomWidth, int geomHeight, int geomCount,
                          int frameCount, int frameStart, int frameEnd, int frameStepping,
                          std::string config)
{
  char parameters[4096];
  char str[ 4096 ];

  std::string result;

  sprintf( parameters, "%s %s %d %d %d %d %d %d %d %d %s\n", flags.c_str(), bindingFlags.c_str(),
    geomIndex, geomWidth, geomHeight, geomCount, frameCount, frameStart, 
    frameEnd, frameStepping, config.c_str() );

  sprintf( str, "FaceSet:3 %s", parameters );  result += str;
  sprintf( str, "FaceSet:4 %s", parameters );  result += str;
  sprintf( str, "FaceSet:5 %s", parameters );  result += str;
  sprintf( str, "PointSet %s", parameters );  result += str;
  sprintf( str, "LineSet %s", parameters );  result += str;
  sprintf( str, "QuadMesh %s", parameters );  result += str;
  sprintf( str, "TriangleSet %s", parameters );  result += str;
  sprintf( str, "TriangleStripSet %s", parameters );  result += str;

  sprintf( parameters, "%s %s %d %d %d %d %d %d %d %d %s\n", flags.c_str(), indexedBindingFlags.c_str(),
    geomIndex, geomWidth, geomHeight, geomCount, frameCount, frameStart, 
    frameEnd, frameStepping, config.c_str() );

  sprintf( str, "IndexedFaceSet:3 %s", parameters );  result += str;
  sprintf( str, "IndexedFaceSet:4 %s", parameters );  result += str;
  sprintf( str, "IndexedFaceSet:5 %s", parameters );  result += str;
  sprintf( str, "IndexedPointSet %s", parameters );  result += str;
  sprintf( str, "IndexedLineSet %s", parameters );  result += str;
  sprintf( str, "IndexedQuadMesh %s", parameters );  result += str;
  sprintf( str, "IndexedTriangleSet %s", parameters );  result += str;
  sprintf( str, "IndexedTriangleStripSet %s", parameters );  result += str;

  return result;
}


//------------------------------------------------------------------------------
void
BenchTool::decodeBenchString(std::string str)
{
  SbString sbStr( str );
  sbStr.replace( "\r", "" ); // remove carriage return chars

  // First we create the list of bench strings. 
  SbStringList list = SbStringList::split( sbStr, "\n" );

  // Then for each bench string we create a BenchData object
  size_t count = (size_t)list.getLength();

  static int n = 1;

  // Format is:
  // "OIVShapeName FLAGS BINDING_FLAGS geomIndex geomWidth geomHeight geomCount frameNum benchFrameStart benchFrameEnd benchFrameStepping"
  //
  // FLAGS is a combination of VNC where V stands for vertices, N is normal, C is color,
  // T for textures, 3 for 3D textures.
  //      V is mandatory.
  // OVERALL_FLAGS: xx where:
  //  - each x is O for overall, V per vertex, F per face, G per face indexed, W per vertex indexed.
  //  - The first x is for the normals
  //  - The second x is for the colors
  //
  // frameNum is the number of frame to render; -1 for infinite; -2 to display just display viewer
  // benchFrameStart is the number of the first benched frame.
  // benchFrameEnd is the number of the last benched frame; -1 for all the frames
  // benchFrameStepping is the number of frames to use for a single bench result.

  for (unsigned long long i = 0; i < count; i++)
  {
    if (list[i]->isEmpty() || (list[i]->contains("#")))
      continue;

    BenchData benchData;

    benchData.id = n++;
    benchData.benchString = list[i]->toStdString();

    // We split the parameters
    SbStringList parameters = SbStringList::split( *list[i], " " );

    // We check that we have the required parameters count
    if (parameters.getLength() <11 )
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid parameters count for the bench: ")
      + list[i]->toStdString() );

    int param = 0;

    SbString nodeName = *parameters[ param++ ];
    SbString flags = *parameters[ param++ ];
    SbString overallFlags = *parameters[ param++ ];

    if (!flags.contains( "V" ))
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: V is mandatory in the flags: ")
      + list[i]->toStdString() );

    benchData.shapeInfo.hasColors = flags.contains( "C" )? true : false;
    benchData.shapeInfo.hasNormals = flags.contains( "N" )? true : false;
    benchData.shapeInfo.hasTexCoords = flags.contains( "T" )? true : false;
    benchData.shapeInfo.hasTexCoords3 = flags.contains( "3" )? true : false;
    benchData.shapeInfo.hasVertexAttribs = flags.contains( "A" )? true : false;

    if (overallFlags.getLength() != 2)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: you must have 2 overall flags (normals, colors): ")
      + list[i]->toStdString() );

    benchData.shapeInfo.normalsBinding = getBinding( overallFlags.toLatin1()[0] );
    benchData.shapeInfo.colorsBinding = getBinding( overallFlags.toLatin1()[1] );

    SbString drawStyleFlag = *parameters[ param++ ];
    benchData.drawStyle = drawStyleFlag.contains( "F" ) ? SoDrawStyle::FILLED :
                          drawStyleFlag.contains( "L" ) ? SoDrawStyle::LINES :
                                                          SoDrawStyle::POINTS;

    SbBool ok;
    benchData.shapeInfo.geomIndex = parameters[ param++ ]->toInt( &ok );
    if (!ok)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid geomIndex parameter in bench: ")
      + list[i]->toStdString() );


    benchData.shapeInfo.width = parameters[ param++ ]->toInt( &ok );
    if (!ok)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid geomWidth parameter in bench: ")
      + list[i]->toStdString() );

    benchData.shapeInfo.height = parameters[ param++ ]->toInt( &ok );
    if (!ok)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid geomHeight parameter in bench: ")
      + list[i]->toStdString() );

    benchData.shapesCount = parameters[ param++ ]->toInt( &ok );
    if (!ok)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid geomCount parameter in bench: ")
      + list[i]->toStdString() );

    benchData.frameNum = parameters[ param++ ]->toInt( &ok );
    if (!ok)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid numFrame parameter in bench: ")
      + list[i]->toStdString() );

    benchData.frameStart = parameters[ param++ ]->toInt( &ok );
    if (!ok)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid benchFrameStart parameter in bench: ")
      + list[i]->toStdString() );

    benchData.frameEnd = parameters[ param++ ]->toInt( &ok );
    if (!ok)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid benchFrameEnd parameter in bench: ")
      + list[i]->toStdString() );

    benchData.frameStepping = parameters[ param++ ]->toInt( &ok );
    if (!ok)
      throw std::runtime_error(
      std::string ("BenchTool::decodeBenchString: Invalid frameStepping parameter in bench: ")
      + list[i]->toStdString() );

    if ( ( parameters.getLength() - param ) >= 2 )
    {
      benchData.shapeInfo.neighborTolerance = parameters[ param++ ]->toFloat( &ok );
      if (!ok)
        throw std::runtime_error(
        std::string ("BenchTool::decodeBenchString: Invalid neighborTolerance parameter in bench: ")
        + list[i]->toStdString() );
    } else
      benchData.shapeInfo.neighborTolerance = 0.f;

    SbString configName("standalone");
    if ( param < parameters.getLength() )
      configName = *parameters[ param++ ];
    benchData.configName = configName;

    buildGeometry( benchData, nodeName );

    // We free the memory
    for (int j = 0; j < parameters.getLength(); j++)
      delete parameters[j];
  }

  // We must free the memory
  for (unsigned long long i = 0; i < count; i++)
    delete list[i];
}

void
BenchTool::buildGeometry(BenchData& benchData, SbString nodeName)
{
  nodeToGeometryInfo( nodeName, benchData );

  bool isAUTORUN = SoPreferences::getBool("SCALEVIZ_AUTORUN",FALSE);
  if (!isAUTORUN && SoScaleViz::setSelectedConfiguration( benchData.configName ))
    SoScaleViz::connect();

  // Ok we store this bench
  m_benchData.push_back( benchData );
}

//------------------------------------------------------------------------------
void
BenchTool::performBench(std::string outputIvFile, bool outputDisplay,
                        std::string outputDisplayFile, Viewer* viewer,
                        BenchData benchData)
{
  bool isTempViewer = false;
  BenchResult results;

  // No viewer so we must create one

  if (!viewer)
  {
    QWidget* mainWindow = SoQt::init( "ivShapeBenchmark" );
    {
      viewer = new Viewer( mainWindow );
      isTempViewer = true; 
    }
  }

  // We must generate some geometry
  // - First we set some settings

  // 1) Geometry settings
#define SO_DEFAULT_MESH_SIZE 500

  benchData.shapeInfo.amplitude = SoPreferences::getInt( "BENCHER_GEOMETRY_AMPLITUDE", benchData.shapeInfo.width / 10 );
  benchData.shapeInfo.width = SoPreferences::getInt( "BENCHER_GEOMETRY_WIDTH", benchData.shapeInfo.width );
  benchData.shapeInfo.height = SoPreferences::getInt( "BENCHER_GEOMETRY_HEIGHT", benchData.shapeInfo.height );

  // 2) Viewer settings

  SbColor backroungColor = SoPreferences::getColor( "BENCHER_VIEWER_BACKGROUND_COLOR", SbColor( 0.0, 0.0, 1.0 ) );
  viewer->setBackgroundColor( backroungColor );

  int vWidth = SoPreferences::getInt( "BENCHER_VIEWER_WIDTH", 640 );
  int vHeight = SoPreferences::getInt( "BENCHER_VIEWER_HEIGHT", 480 );
  viewer->setSize( SbVec2s( vWidth, vHeight) );

  viewer->show();
  viewer->bindNormalContext();
  viewer->setDrawStyle( benchData.drawStyle );
  viewer->removeAllNodes();

  // - Results must be formated at some point
  results.nodeName = benchData.shapeInstance[0]->getTypeId().getName().getString();
  results.beforeMemoryUsage = MemoryTool::current();

  printf( "[D] Benching %s\n", results.nodeName.c_str() );

  viewer->setTitle( QString( "ivShapeBenchmark: " ) + QString( results.nodeName.c_str() ) );

  // 3) We add the geometry and we start the rendering
  SbBool useTranslation = SoPreferences::getBool( "BENCHER_STACKED_SHAPES", FALSE );

  // - First we generate the geometry
  for (int i = 0; i < benchData.shapesCount; i++)
  {
    int xOffset = 0;
    int yOffset = 0;
    int zOffset = 0;
    if (!useTranslation && benchData.shapesCount > 1)
    {
      int perRow = int( std::sqrt( float(benchData.shapesCount) ) );
      yOffset = i / perRow;
      xOffset = i - perRow * yOffset;

      xOffset *= benchData.shapeInfo.width;
      yOffset *= benchData.shapeInfo.height;
    }
    else
    {
      zOffset = i * (benchData.shapeInfo.width + benchData.shapeInfo.height) / 10.0;
    }

    benchData.shapeInfo.xOffset = xOffset;
    benchData.shapeInfo.yOffset = yOffset;
    benchData.shapeInfo.zOffset = zOffset;

    GeometryTool::generateGeometry( benchData.shapeInstance[i], benchData.shapeInfo );
  }

  results.beforeRenderingMemoryUsage = MemoryTool::current();

  if (benchData.shapeInfo.hasTexCoords)
  {
    SoTextureUnit* texUnit = new SoTextureUnit;
    texUnit->unit = 0;
///////////////////////////TODO
    SoTexture2* texture2 = new SoTexture2;
    texture2->filename = "$OIVHOME/tools/source/ivShapeBenchmark/texture2d.png";
    texture2->model = SoTexture2::ADD;

    viewer->addNode( texUnit );
    viewer->addNode( texture2 );
  }

  if ( benchData.shapeInfo.hasTexCoords3 )
  {
    SoTextureUnit* texUnit2 = new SoTextureUnit;
    texUnit2->unit = 1;
    SoTexture3* texture3 = new SoTexture3;
    texture3->filenames.setNum( 2 );
    texture3->filenames.set1Value( 0, "$OIVHOME/tools/source/ivShapeBenchmark/layer1.png" );
    texture3->filenames.set1Value( 1, "$OIVHOME/tools/source/ivShapeBenchmark/layer2.png" );
    texture3->model = SoTexture3::ADD;
    texture3->minFilter = SoTexture3::NEAREST;
    texture3->magFilter = SoTexture3::NEAREST;

    /*
    if (!benchData.shapeInstance[0]->isOfType( SoLineSet::getClassTypeId() ) && 
    !benchData.shapeInstance[0]->isOfType( SoIndexedLineSet::getClassTypeId() ) &&
    !benchData.shapeInstance[0]->isOfType( SoIndexedTriangleStripSet::getClassTypeId() ) )
    {*/
    viewer->addNode( texUnit2 );
    viewer->addNode( texture3 );
    //}
  }

  if (benchData.shapeInfo.hasVertexAttribs)
  {
    SoVertexShader* vertexShader = new SoVertexShader;
    vertexShader->sourceType = SoVertexShader::FILENAME;
    vertexShader->sourceProgram.setValue( "$OIVHOME/tools/source/ivShapeBenchmark/vertexShader.glsl" );

    SoShaderProgram *shaderProgram = new SoShaderProgram;
    {
      shaderProgram->shaderObject.set1Value( 0, vertexShader );
    }

    viewer->addNode( shaderProgram );

    SoVertexShaderParameter3f* vertexParameters = new SoVertexShaderParameter3f;
    {
      size_t num = 0;
      SoVertexProperty* vp = NULL;

      SoBufferedShape* bufferedShape = 
        dynamic_cast< SoBufferedShape* >( benchData.shapeInstance[0] );

      if (dynamic_cast< SoVertexShape* >( benchData.shapeInstance[0] ))
      {
        vp = (SoVertexProperty*)(((SoVertexShape*)benchData.shapeInstance[0])->
          vertexProperty.getValue());
        num = vp->vertex.getNum();
      }
      else
        num = bufferedShape->vertexBuffer.getValue()->getSize() / (3*sizeof( float ));

      vertexParameters->name = "vertexAttribs";
      vertexParameters->identifier = 0;

      vertexParameters->value.setNum( int(num) );
      SbVec3f* params = vertexParameters->value.startEditing();

      if (vp)
      {
        SbVec3f* vertices = vp->vertex.startEditing();
        {
          for (size_t i = 0; i < num; i++)
            params[i].setValue( 0.f, 0.f, float(sin( vertices[i][0] * benchData.shapeInfo.width * 0.5) ) );
          /*
          // What is the best, keep the cache fetching or recompute the sin for each value ?
          for (int j = 0; j< benchData.shapeInfo.height; j++)
          for (int i = 0; i< benchData.shapeInfo.width; i++)
          params[ j * benchData.shapeInfo.width + i ].setValue( 0.f, 0.f, float(sin( double(i) )) );
          */
        }
        vertexParameters->value.finishEditing();
        vp->vertex.finishEditing();
      }
      else
      {
        SoGpuBufferObject* vertexBuffer = 
          (SoGpuBufferObject*)bufferedShape->vertexBuffer.getValue();

        float* vertices = (float *)vertexBuffer->map( SoBufferObject::READ_ONLY );

        if (vertices)
          for (size_t i = 0; i < num; i++)
            params[i].setValue( 0.f, 0.f, float(sin( vertices[3*i] * benchData.shapeInfo.width * 0.5) ) );

        vertexBuffer->unmap();
      }
    }
    viewer->addNode( vertexParameters );
  }

  // - We set our brand new shape...
  int idCount = 1;
  for (int i = 0; i < benchData.shapesCount; i++)
  {
    SoSeparator* newSep = new SoSeparator();
    newSep->renderUnitId = idCount++;
    viewer->addNode( newSep );
    if ( SoPreferences::getBool("BENCHER_TRANSP_SHAPES", false)
      && ((i % 3) == 1 || benchData.shapesCount == 1)) 
    {
      idCount--;
      newSep->renderUnitId = 0;

      SoMaterial* mat = new SoMaterial();
      mat->transparency = 0.5;
      mat->override = true;
      newSep->addChild( mat );
    }

    newSep->addChild( benchData.shapeInstance[i] );
  }

  if (benchData.shapeInfo.hasHoles)
    viewer->setWindindType( SoShapeHints::ODD_TYPE );
  else
    viewer->setWindindType( SoShapeHints::NO_WINDING_TYPE );

  // set tolerance
  viewer->setNeighborTolerance( benchData.shapeInfo.neighborTolerance );

  // Dump the iv file first - so if it crashes we'll have the IV file!
  results.ivFilename = outputIvFile;

  if (outputIvFile.length())
  {
    SoWriteAction wa;
    SoNode* sg = viewer->getSceneGraph();

    if (! wa.getOutput()->openFile( outputIvFile.c_str() ))
    {
      printf( "Cannot open %s in order to write the scene graph\n", outputIvFile.c_str() );
    }
    else
    {
      wa.getOutput()->setBinary( FALSE );
      wa.apply( sg );
      wa.getOutput()->closeFile();
    }
  }

  viewer->viewAll();

#if defined(_DEBUG)
  viewer->setTitle( "ivShapeBenchmark(D): " + benchData.benchString );
#else
  viewer->setTitle( "ivShapeBenchmark(R): " + benchData.benchString );
#endif

  // Case one: we want a viewer (aka the numFrames parameter is equal to -2)
  if (benchData.frameNum == -2 )
  {
    //viewer->setAutoRedraw( TRUE );

    results.globalFps = -1;
  }
  // case two: we want a viewer but mainloop is managed by someone else (gui mode)
  else if ( benchData.frameNum == -3 )
  {
    results.globalFps = -1;
  }
  else
  {
    viewer->setAutoRedraw( FALSE );

    SbElapsedTime elapsedTime;
    double firstFrameTime = 0.0;
    double secondFrameTime = 0.0;
    double thirdFrameTime = 0.0;

    double firstFrameMem = 0;
    double secondFrameMem = 0;
    double thirdFrameMem = 0;


#if 1
    elapsedTime.reset();

    for (int i = 0; i < benchData.frameNum; i++)
    {
      viewer->getNormalWidget()->repaint();
      QCoreApplication::processEvents();

      if ( i == 0 )
      {
        firstFrameTime = elapsedTime.getElapsed();
        firstFrameMem = MemoryTool::current();
      }
      else if ( i == 1 )
      {
        secondFrameTime = elapsedTime.getElapsed();
        secondFrameMem = MemoryTool::current();
      }
      else if ( i == 2 )
      {
        thirdFrameTime = elapsedTime.getElapsed();
        thirdFrameMem = MemoryTool::current();
      }
    }
#else
    viewer->setAutoRedraw( TRUE );
    m_framesCount = 0;

    viewer->setPostRenderCallback( staticRenderCallback, this );

    elapsedTime.reset();
    SoQt::mainLoop();
#endif
    double elapsed = elapsedTime.getElapsed();
    results.globalFps = float( double( benchData.frameNum ) / elapsed );

    results.firstFrameTime = float( firstFrameTime );
    results.secondFrameTime = float( secondFrameTime - firstFrameTime );
    results.thirdFrameTime = float( thirdFrameTime - secondFrameTime );
    if ( benchData.frameNum > 3 )
      results.remainingFps = float( double( benchData.frameNum - 3 ) / (elapsed - thirdFrameTime) );
    else
      results.remainingFps = -1.;

    results.afterFirstFrameMemoryUsage = firstFrameMem;
    results.afterSecondFrameMemoryUsage = secondFrameMem;
    results.afterThirdFrameMemoryUsage = thirdFrameMem;
    results.peakMemoryUsage = MemoryTool::peak();

    results.afterRenderingMemoryUsage = MemoryTool::current();
  }

  if ( benchData.frameNum != -3 )
  {
    // Dump
    if (outputDisplay)
    {
      // ensure the viewer's contents are well drawn before capturing the snapshot
      viewer->redraw();

      viewer->saveSnapshot( outputDisplayFile );
    }

    viewer->removeAllNodes();

    results.afterCleanUpMemoryUsage = MemoryTool::current();


    // Store some info in the result structure
    results.id = benchData.id;
    results.sshotFilename = outputDisplayFile;
    results.benchString = benchData.benchString;
    results.shapeInfo = benchData.shapeInfo;
    results.configName = benchData.configName.getString();

    m_benchResults.push_back( results );

    if (isTempViewer)
    {
      delete viewer;
      SoQt::finish();
    }
  }
  else {
    viewer->unbindNormalContext();
  }
}

//------------------------------------------------------------------------------
void 
BenchTool::getGeometryFromNode(const SbString &nodeName, GeometryTool::GeometryInfo &outGeomInfo, SoNode* &pNodeOut )
{
  SoBase *pBase = NULL;
  int num = -1;
  int countPos = nodeName.find( ":" );
  SbString shapeName = nodeName;

  if (countPos == nodeName.getLength()-2)
  {
    SbBool ok;
    num = nodeName.getSubString( nodeName.getLength() - 1, nodeName.getLength() - 1 ).toInt( &ok );
    shapeName = nodeName.getSubString( 0, nodeName.getLength() - 3 );

    if (!ok)
      throw std::runtime_error(
      std::string("BencherTool::nodeToGeometryInfo: Invalid node name (extra digit is not valid): " ) + 
      nodeName.toStdString() );
  }

  outGeomInfo.isIndexed = nodeName.contains("Indexed");
  outGeomInfo.isStrip = nodeName.upper().contains("STRIP");
  outGeomInfo.isPrimitiveRestart = nodeName.contains("IndexedPR");
  outGeomInfo.isBufferedShape = false;

  if (nodeName.contains( "Buffered" ) == TRUE)
  {
    shapeName = "BufferedShape";
    outGeomInfo.isBufferedShape = true;
  }

  SoType nodeType = SoType::fromName( shapeName );

  if (nodeType.isBad())
    throw std::runtime_error(
    std::string("BencherTool::nodeToGeometryInfo: Invalid node name: " ) + 
    shapeName.toStdString() );

  pBase = (SoBase *)nodeType.createInstance();

  if (dynamic_cast< SoShape * >( pBase ) == NULL)
    throw std::runtime_error(
    std::string("BencherTool::nodeToGeometryInfo: Specified node is not a shape: " ) + 
    shapeName.toStdString() );

  if (outGeomInfo.isBufferedShape == false)
  {
    // SoFaceSet
    if (dynamic_cast< SoFaceSet * >( pBase ) || dynamic_cast< SoIndexedFaceSet * >( pBase ))
    {
      switch( num )
      {
      case 3: outGeomInfo.type = GeometryTool::TRIANGLES; break;
      case 4: outGeomInfo.type = GeometryTool::QUADS; break;
      default: outGeomInfo.type = GeometryTool::POLYGON; break;
      }
    }

    if (dynamic_cast< SoLineSet * >( pBase ) || dynamic_cast< SoIndexedLineSet * >( pBase ))
      outGeomInfo.type = GeometryTool::LINES;
    else if (dynamic_cast< SoPointSet * >( pBase ) || dynamic_cast< SoIndexedPointSet * >( pBase ))
      outGeomInfo.type = GeometryTool::POINTS;
    else if (dynamic_cast< SoTriangleSet * >( pBase ) || dynamic_cast< SoIndexedTriangleSet * >( pBase ))
      outGeomInfo.type = GeometryTool::TRIANGLES;
    else if (dynamic_cast< SoTriangleStripSet * >( pBase ) || dynamic_cast< SoIndexedTriangleStripSet * >( pBase ))
      outGeomInfo.type = GeometryTool::TRIANGLE_STRIP;
    else if (dynamic_cast< SoQuadMesh * >( pBase ) || dynamic_cast< SoIndexedQuadMesh * >( pBase ))
      outGeomInfo.type = GeometryTool::QUADS;
  }
  else
  {

    ((SoBufferedShape*)pBase)->primitiveRestartEnabled.setValue( outGeomInfo.isPrimitiveRestart );
    ((SoBufferedShape*)pBase)->shapeUsage.setValue(SoBufferedShape::STATIC);
    if (nodeName.contains("POINTS"))
    {
      outGeomInfo.type = GeometryTool::POINTS;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::POINTS;
    }
    else if (nodeName.contains("LINES"))
    {

      outGeomInfo.type = GeometryTool::LINES;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::LINES;
    }
    else if (nodeName.contains("LINE_STRIP"))
    {
      outGeomInfo.type = GeometryTool::LINE_STRIP;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::LINE_STRIP;
    }
    else if (nodeName.contains("LINE_LOOP"))
    {
      outGeomInfo.type = GeometryTool::LINE_LOOP;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::LINE_LOOP;
    }
    else if (nodeName.contains("TRIANGLES"))
    {
      outGeomInfo.type = GeometryTool::TRIANGLES;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::TRIANGLES;
    }
    else if (nodeName.contains("TRIANGLE_STRIP"))
    {        
      outGeomInfo.type = GeometryTool::TRIANGLE_STRIP;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::TRIANGLE_STRIP;
    }
    else if (nodeName.contains("TRIANGLE_FAN"))
    {
      outGeomInfo.type = GeometryTool::TRIANGLE_FAN;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::TRIANGLE_FAN;
    }
    else if (nodeName.contains("QUADS"))
    {
      outGeomInfo.type = GeometryTool::QUADS;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::QUADS;
    }
    else if (nodeName.contains("QUAD_STRIP"))
    {
      outGeomInfo.type = GeometryTool::QUAD_STRIP;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::QUAD_STRIP;
    }
    else if (nodeName.contains("POLYGON"))
    {
      outGeomInfo.type = GeometryTool::POLYGON;
      ((SoBufferedShape*)pBase)->shapeType = SoBufferedShape::POLYGON;
    }
  }

  pNodeOut = (SoNode*)pBase;

}

//------------------------------------------------------------------------------
void
BenchTool::nodeToGeometryInfo(SbString nodeName, BenchData& benchData)
{
  BenchTool::getGeometryFromNode( nodeName, benchData.shapeInfo, benchData.shapeInstance[0] );

  benchData.shapeInstance[0]->ref();

  for (int i = 1; i < benchData.shapesCount; i++)
  {
    benchData.shapeInstance[i] = benchData.shapeInstance[0]->copy();
  }
}

//------------------------------------------------------------------------------
GeometryTool::BindingType
BenchTool::getBinding(char flag)
{
  switch( flag )
  {
    case 'O': return GeometryTool::OVERALL;
    case 'F': return GeometryTool::PER_FACE;
    case 'V': return GeometryTool::PER_VERTEX;
    case 'G': return GeometryTool::PER_FACE_INDEXED;
    case 'W': return GeometryTool::PER_VERTEX_INDEXED;
    default:
    {
      char str[2];
      str[0] = flag; str[1] = '\0';

      throw std::runtime_error( std::string("BenchTool::getBinding: Invalid binding ") + 
        std::string( str ) );
    }
  }

  return GeometryTool::OVERALL;
}


//------------------------------------------------------------------------------
SbBool
BenchTool::staticRenderCallback(void* userData, SoQtRenderArea* rendArea)
{
  BenchTool* benchTool = (BenchTool *)userData;

  rendArea->swapNormalBuffers();

  SbBool breakRendering = benchTool->renderCallback( userData, rendArea );

  if (breakRendering)
  {
    rendArea->hide();
    printf("Kill\n");
  }

  return TRUE;
}


//------------------------------------------------------------------------------
SbBool
BenchTool::renderCallback(void* /*userData*/, SoQtRenderArea* /*rendArea*/)
{
  //rendArea->getWidget()->update();

  m_framesCount++;

  return (m_framesCount>100)?TRUE:FALSE;
}

//------------------------------------------------------------------------------


