/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoVertexShape.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoIndexedFaceSet.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoQuadMesh.h>
#include <Inventor/nodes/SoIndexedQuadMesh.h>
#include <Inventor/nodes/SoTriangleSet.h>
#include <Inventor/nodes/SoIndexedTriangleSet.h>
#include <Inventor/nodes/SoTriangleStripSet.h>
#include <Inventor/nodes/SoIndexedTriangleStripSet.h>
#include <Inventor/devices/SoGpuBufferObject.h>

#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/STL/stdexcept>
#include <geometry.h>

//------------------------------------------------------------------------------
GeometryTool::BindingType
GeometryTool::getBindingType(const char& value)
{
  switch( value )
  {
    case 'O': return OVERALL;
    case 'F': return PER_FACE;
    case 'V': return PER_VERTEX;
    case 'G': return PER_FACE_INDEXED;
    case 'W': return PER_VERTEX_INDEXED;
    default: return INVALID;
  }
}

#define GEN_SIMPLE( x ) \
  if (vShape){ \
  generateSimplePrimitive( vShape, info, (x) ); \
  setupSimplePrimitiveIndices( vShape, info, (x) ); \
  } \
  else \
  if (bufferedShape){ \
  generateSimplePrimitive( bufferedShape, info, (x) ); \
  setupSimplePrimitiveIndices( bufferedShape, info, (x) ); \
  }

#define GEN_HOLES( x ) \
  if (vShape){ \
  generateHole( vShape, info, (x) ); \
  setupHoleIndices( vShape, info, (x) ); \
  } \
  else \
  if (bufferedShape){ \
  generateHole( bufferedShape, info, (x) ); \
  setupHoleIndices( bufferedShape, info, (x) ); \
  }

#define GEN_GEOMETRY( x ) \
  if ( info.geomIndex == 1) {\
  GEN_HOLES( (x) ) \
  } \
   else {\
   GEN_SIMPLE( (x) ) \
}

//------------------------------------------------------------------------------
void
GeometryTool::generateGeometry(SoNode* shapeNode, GeometryInfo& info)
{
  if (!shapeNode)
    return;

  info.hasHoles = false;
  if (info.geomIndex == 1)
    info.hasHoles = true;

  // If we have a vertex based shape
  SoVertexShape* vShape = dynamic_cast< SoVertexShape *>( shapeNode );
  SoBufferedShape* bufferedShape = dynamic_cast< SoBufferedShape *>( shapeNode );

  if (vShape || bufferedShape)
  {
    switch( info.type )
    {
      case GeometryTool::TRIANGLES:
      case GeometryTool::TRIANGLE_STRIP:
        GEN_GEOMETRY( 3 )
          break;

      case GeometryTool::LINES:
      case GeometryTool::QUADS:
      case GeometryTool::QUAD_STRIP:
        GEN_GEOMETRY( 4 )
          break;

      case GeometryTool::POLYGON:
      case GeometryTool::TRIANGLE_FAN:
      case GeometryTool::LINE_STRIP:
      case GeometryTool::LINE_LOOP:
        GEN_GEOMETRY( 5 )
          break;

      case GeometryTool::POINTS:
        GEN_GEOMETRY( 1 )
          break;
    }
  }
}

//------------------------------------------------------------------------------
int
GeometryTool::getOivBinding(BindingType binding)
{
  switch( binding )
  {
    case OVERALL: return SoVertexProperty::OVERALL;
    case PER_VERTEX: return SoVertexProperty::PER_VERTEX;
    case PER_FACE: return SoVertexProperty::PER_FACE;
    case PER_VERTEX_INDEXED: return SoVertexProperty::PER_VERTEX_INDEXED;
    case PER_FACE_INDEXED: return SoVertexProperty::PER_FACE_INDEXED;
    case INVALID: return 0;
    default:
      return 0;
  }
  return 0;
}

//------------------------------------------------------------------------------
float*
GeometryTool::generateVertices(GeometryInfo& info)
{
  // We create a buffer of vertices, we'll use them later.
  float* vertices = new float[ info.width * info.height * 3 ];

  double a = 16.0 / double( info.width-1 );
  double a2 = 16.0 / double( info.height-1 );
  double b = -8.;

  for(int y=0; y<info.height; y++)
  {
    for(int x=0; x < info.width; x++) 
    {
      double ix = a * double( x ) + b;
      double iy = a2 * double( y ) + b;

      double iz = 3.14159 * sqrt( ix*ix + iy*iy );
      double z = (iz != 0.) ? sin( iz ) / iz : 1.0;

      vertices[ 3 * (y * info.width + x) + 0 ] = float( x + info.xOffset );
      vertices[ 3 * (y * info.width + x) + 1 ] = float( y + info.yOffset );
      vertices[ 3 * (y * info.width + x) + 2 ] = float(z * info.amplitude + info.zOffset);
    }
  }

  return vertices;
}

//------------------------------------------------------------------------------
float texFactor(float m, float h1, float h2)
{
  float percent;
  percent = (m - fabs(h1 - h2)) / m;

  if(percent < 0.0f) percent = 0.0f;
  else if(percent > 1.0f) percent = 1.0f;

  return percent;
}

//------------------------------------------------------------------------------
float*
GeometryTool::generateColors(GeometryInfo& info, float* vertices)
{
  float* colors = new float[ info.width * info.height * 3 ];
  float texFact[4];

  float inColors[12] = { 1.f, 0.f, 0.f, 
    0.f, 1.f, 0.f,
    0.f, 0.f, 1.f,
    1.f, 0.f, 1.f };

  float zMin = info.amplitude;
  float zMax = -info.amplitude;

  for (int j = 0; j < info.height; j++)
    for (int i = 0; i < info.width; i++)
    {
      if (vertices[ 3 * (j*info.width + i) + 2 ]<zMin)
        zMin = vertices[ 3 * (j*info.width + i) + 2 ];

      if (vertices[ 3 * (j*info.width + i) + 2 ]>zMax)
        zMax = vertices[ 3 * (j*info.width + i) + 2 ];
    }

    float zAmplitude = zMax - zMin;

    for (int j = 0; j < info.height; j++)
      for (int i = 0; i < info.width; i++)
      {
        float z = vertices[ 3 * (j*info.width + i) + 2 ];

        texFact[0] = texFactor( zAmplitude / 4.f, zAmplitude, z - zMin );
        texFact[1] = texFactor( zAmplitude / 4.f, 3.f * zAmplitude / 4.f, z - zMin );
        texFact[2] = texFactor( zAmplitude / 4.f, zAmplitude / 2.f, z - zMin );
        texFact[3] = texFactor( zAmplitude / 4.f, zAmplitude / 4.f, z - zMin );

        colors[ 3 * (j*info.width + i) + 0 ] = texFact[0] * inColors[0] + 
          texFact[1] * inColors[3] + texFact[2] * inColors[6] + texFact[3] * inColors[9];
        colors[ 3 * (j*info.width + i) + 1 ] = texFact[0] * inColors[1] + 
          texFact[1] * inColors[4] + texFact[2] * inColors[7] + texFact[3] * inColors[10];
        colors[ 3 * (j*info.width + i) + 2 ] = texFact[0] * inColors[2] + 
          texFact[1] * inColors[5] + texFact[2] * inColors[8] + texFact[3] * inColors[11];
      }

      return colors;
}

//------------------------------------------------------------------------------
float*
GeometryTool::generateTexCoords(GeometryInfo& info, float* /*vertices*/)
{
  float* coords = new float[ info.width * info.height * 2 ];

  for (int j = 0; j < info.height; j++)
    for (int i = 0; i < info.width; i++)
    {  
      coords[ (j * info.width + i) * 2 ] = float( i ) / info.width;
      coords[ (j * info.width + i) * 2 + 1 ] = float( j ) / info.height;
    }

    return coords;
}

//------------------------------------------------------------------------------
float*
GeometryTool::generateTexCoords3(GeometryInfo& info, float* /*vertices*/)
{
  float* coords = new float[ info.width * info.height * 3 ];

  for (int j = 0; j < info.height; j++)
    for (int i = 0; i < info.width; i++)
    {  
      coords[ (j * info.width + i) * 3 ] = float( i ) / info.width;
      coords[ (j * info.width + i) * 3 + 1 ] = float( j ) / info.height;
      coords[ (j * info.width + i) * 3 + 2 ] = float( i ) / info.width;
    }

    return coords;
}

//------------------------------------------------------------------------------
float*
GeometryTool::generateNormals(GeometryInfo& info, float* vertices)
{
  int width = info.width;
  int height = info.height;

  float* normals = new float[ width * height * 3 ];

  for (int y = 0; y < height-1; y++)
    for (int x = 0; x < width-1; x++)
    {
      SbVec3f v1( vertices[3*(y*width+x)], vertices[3*(y*width+x) + 1], vertices[3*(y*width+x) + 2] );
      SbVec3f v2( vertices[3*(y*width+x+1)], vertices[3*(y*width+x+1) + 1], vertices[3*(y*width+x+1) + 2] );
      SbVec3f v3( vertices[3*((y+1)*width+x)], vertices[3*((y+1)*width+x) + 1], vertices[3*((y+1)*width+x) + 2] );

      SbVec3f va = v1 - v2;
      SbVec3f vb = v1 - v3;
      SbVec3f norm = va.cross(vb);
      norm.normalize();

      normals[3*(y*width+x)] = norm[0];
      normals[3*(y*width+x) + 1] = norm[1];
      normals[3*(y*width+x) + 2] = norm[2];
    }

    return normals;
}

#define SET_VEC( index, input, output ) \
  (output)[(index)++].setValue( (input)[0], (input)[1], (input)[2] )

#define SET_TEXCOORD( index, input, output ) \
  (output)[(index)++].setValue( (input)[0], (input)[1] );

#define SET_TEXCOORD3( index, input, output ) \
  (output)[(index)++].setValue( (input)[0], (input)[1], (input)[2] )

#define SET_COL( index, input, output ) \
{ \
  uint32_t r = int(((input)[0] * 256.f)) & 255; \
  uint32_t g = int(((input)[1] * 256.f)) & 255; \
  uint32_t b = int(((input)[2] * 256.f)) & 255; \
  uint32_t a = 255; \
  (output)[(index)++]= (r << 24) | (g << 16) | (b<<8) | (a); \
}

#define SET_NORM( index, input, output ) \
  (output)[(index)++].setValue( (input)[0], (input)[1], (input)[2] )

#define INDEX( n, i, j ) ((n) * ((j)*info.width + (i)))
#define V_INDEX( v, i, j ) &((v)[ INDEX( 3 , (i), (j)) ])
#define C_INDEX( c, i, j ) &((c)[ INDEX( 3 , (i), (j)) ])
#define N_INDEX( n, i, j ) &((n)[ INDEX( 3 , (i), (j)) ])
#define T_INDEX( t, i, j ) &((t)[ INDEX( 2 , (i), (j)) ])
#define T3_INDEX( t, i, j ) &((t)[ INDEX( 3 , (i), (j)) ])

//------------------------------------------------------------------------------
void
GeometryTool::generateSimplePrimitive(SoVertexShape* vertexShape, GeometryInfo& info, int count)
{
  bool isIndexed = info.isIndexed;

  bool isQuadMesh = (  (dynamic_cast< SoQuadMesh* >( vertexShape) != NULL) ||
    (dynamic_cast< SoIndexedQuadMesh* >( vertexShape) != NULL)    ) ? true : false;

  bool isStrip = info.isStrip;

  SoVertexProperty* vp = new SoVertexProperty;
  {
    float* vertices = generateVertices( info );
    float* colors = NULL;
    float* normals = NULL;
    float* texCoords = NULL;
    float* texCoords3 = NULL;
    {
      int numPrimitives =0 ;

      if (count == 3)
        numPrimitives = 2 * (info.width - 1) * (info.height-1);
      if (count == 4 || count == 5)
        numPrimitives = (info.width - 1) * (info.height-1);
      if (count == 1)
        numPrimitives = info.width * info.height;

      if (isStrip)
        numPrimitives = info.height-1;

      int verticesCount = 0;

      if (!isStrip)
      {
        if (!isIndexed)
          verticesCount = numPrimitives * count;
        else
          verticesCount = info.width * info.height;
      }
      else
        verticesCount = info.width * numPrimitives * 2;

      info.numVertices = verticesCount;
      info.numPrimitives = numPrimitives;

      vp->vertex.setNum( verticesCount );
      SbVec3f* vecs = vp->vertex.startEditing();
      uint32_t* cols = NULL;
      SbVec3f* norms = NULL;
      SbVec2f* tcoords = NULL;
      SbVec3f* tcoords3 = NULL;

      if (info.hasColors)
      {
        if (info.colorsBinding == GeometryTool::OVERALL)
        {
          vp->orderedRGBA.setNum(1);
          vp->orderedRGBA.set1Value(0, 0x00FF00FF);
        }
        else
        {
          colors = generateColors( info, vertices );
          vp->orderedRGBA.setNum( verticesCount );
          cols = vp->orderedRGBA.startEditing();
        }
      }

      if (info.hasNormals)
      {
        if (info.normalsBinding == GeometryTool::OVERALL)
        {
          vp->normal.setNum(1);
          vp->normal.set1Value(0, SbVec3f(0.f, 0.f, 1.f));
        }
        else
        {
          normals = generateNormals( info, vertices );
          vp->normal.setNum( verticesCount );
          norms = vp->normal.startEditing();
        }
      }

      if (info.hasTexCoords)
      {
        texCoords = generateTexCoords( info, vertices );
        vp->texCoord.setNum( verticesCount );
        tcoords = vp->texCoord.startEditing();
      }

      if (info.hasTexCoords3)
      {
        texCoords3 = generateTexCoords3( info, vertices );
        vp->texCoord3.setNum( verticesCount );
        tcoords3 = vp->texCoord3.startEditing();
      }

      vp->materialBinding = getOivBinding( info.colorsBinding );      
      vp->normalBinding = getOivBinding( info.normalsBinding );

      int nextVertex = 0;
      int nextColor = 0;
      int nextNormal = 0;
      int nextTexCoord = 0;
      int nextTexCoord3 = 0;

      for (int j = 0; j < info.height; j++)
        for (int i = 0; i < info.width; i++)
        {
          if (!isIndexed && !isQuadMesh)
          {
            if (isStrip)
            {
              if (j == info.height-1)
                continue;

              SET_VEC( nextVertex, V_INDEX( vertices, i, j ), vecs );
              SET_VEC( nextVertex, V_INDEX( vertices, i, j + 1 ), vecs );
            }
            else
            {
              // We don't want the last vertex in non indexed mode
              if ( (count != 1) && ((j == info.height-1) || (i == info.width-1)))
                continue;

              if (count == 3)
              {
                SET_VEC( nextVertex, V_INDEX( vertices, i, j ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i+1, j+1 ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i+1, j ), vecs );

                SET_VEC( nextVertex, V_INDEX( vertices, i, j ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i, j+1 ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i+1, j+1 ), vecs );
              }
              else if (count == 4)
              {
                SET_VEC( nextVertex, V_INDEX( vertices, i, j ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i, j+1 ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i+1, j+1 ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i+1, j ), vecs );
              }
              else if (count == 5)
              {
                SET_VEC( nextVertex, V_INDEX( vertices, i, j ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i, j+1 ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i+1, j+1 ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i+1, j ), vecs );
                SET_VEC( nextVertex, V_INDEX( vertices, i, j ), vecs );
              }
              else if (count == 1)
                SET_VEC( nextVertex, V_INDEX( vertices, i, j ), vecs );
            }

            if (cols)
            {
              if (isStrip)
              {
                SET_COL( nextColor, C_INDEX( colors, i, j ), cols );
                SET_COL( nextColor, C_INDEX( colors, i, j + 1 ), cols );
              }
              else if (count == 3)
              {
                SET_COL( nextColor, C_INDEX( colors, i, j ), cols );
                SET_COL( nextColor, C_INDEX( colors, i+1, j+1 ), cols );
                SET_COL( nextColor, C_INDEX( colors, i+1, j ), cols );

                SET_COL( nextColor, C_INDEX( colors, i, j ), cols );
                SET_COL( nextColor, C_INDEX( colors, i, j+1 ), cols );
                SET_COL( nextColor, C_INDEX( colors, i+1, j+1 ), cols );
              }
              else if (count == 4)
              {
                SET_COL( nextColor, C_INDEX( colors, i, j ), cols );
                SET_COL( nextColor, C_INDEX( colors, i, j+1 ), cols );
                SET_COL( nextColor, C_INDEX( colors, i+1, j+1 ), cols );
                SET_COL( nextColor, C_INDEX( colors, i+1, j ), cols );
              }
              else if (count == 5)
              {
                SET_COL( nextColor, C_INDEX( colors, i, j ), cols );
                SET_COL( nextColor, C_INDEX( colors, i, j+1 ), cols );
                SET_COL( nextColor, C_INDEX( colors, i+1, j+1 ), cols );
                SET_COL( nextColor, C_INDEX( colors, i+1, j ), cols );
                SET_COL( nextColor, C_INDEX( colors, i, j ), cols );
              }
              else if (count == 1)
                SET_COL( nextColor, C_INDEX( colors, i, j ), cols );
            }

            if (norms)
            {
              if (isStrip)
              {
                SET_NORM( nextNormal, N_INDEX( normals, i, j ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i, j+1 ), norms );
              }
              else if (count == 3)
              {
                SET_NORM( nextNormal, N_INDEX( normals, i, j ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i+1, j+1 ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i+1, j ), norms );

                SET_NORM( nextNormal, N_INDEX( normals, i, j ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i, j+1 ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i+1, j+1 ), norms );
              }
              else if (count == 4)
              {
                SET_NORM( nextNormal, N_INDEX( normals, i, j ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i, j+1 ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i+1, j+1 ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i+1, j ), norms );
              }
              else if (count == 5)
              {
                SET_NORM( nextNormal, N_INDEX( normals, i, j ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i, j+1 ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i+1, j+1 ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i+1, j ), norms );
                SET_NORM( nextNormal, N_INDEX( normals, i, j ), norms );
              }
              else if (count == 1)
                SET_NORM( nextNormal, N_INDEX( normals, i, j ), norms );
            }

            if (tcoords)
            {
              if (isStrip)
              {
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j + 1 ), tcoords );
              }
              else if (count == 3)
              {
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i+1, j+1 ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i+1, j ), tcoords );

                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j+1 ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i+1, j+1 ), tcoords );
              }
              else if (count == 4)
              {
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j+1 ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i+1, j+1 ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i+1, j ), tcoords );
              }
              else if (count == 5)
              {
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j+1 ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i+1, j+1 ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i+1, j ), tcoords );
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j ), tcoords );
              }
              else if (count == 1)
                SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j ), tcoords );
            }

            if (tcoords3)
            {
              if (isStrip)
              {
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j + 1 ), tcoords3 );
              }
              else if (count == 3)
              {
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i+1, j+1 ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i+1, j ), tcoords3 );

                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j+1 ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i+1, j+1 ), tcoords3 );
              }
              else if (count == 4)
              {
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j+1 ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i+1, j+1 ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i+1, j ), tcoords3 );
              }
              else if (count == 5)
              {
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j+1 ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i+1, j+1 ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i+1, j ), tcoords3 );
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j ), tcoords3 );
              }
              else if (count == 1)
                SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j ), tcoords3 );
            }
          }
          else
          {
            SET_VEC( nextVertex, V_INDEX( vertices, i, j ), vecs );
            if (cols)
              SET_COL( nextColor, C_INDEX( colors, i, j ), cols );
            if (norms)
              SET_NORM( nextNormal, N_INDEX( normals, i, j ), norms );
            if (tcoords)
              SET_TEXCOORD( nextTexCoord, T_INDEX( texCoords, i, j ), tcoords );
            if (tcoords3)
              SET_TEXCOORD3( nextTexCoord3, T3_INDEX( texCoords3, i, j ), tcoords3 );
          }
        }

        vp->vertex.finishEditing();

        if (cols)
          vp->orderedRGBA.finishEditing();

        if (norms)
          vp->normal.finishEditing();

        if (tcoords)
          vp->texCoord.finishEditing();

        if (tcoords3)
          vp->texCoord3.finishEditing();


        // For the faceSet/lineSet we must indicate that we have triangles/lines
        SoFaceSet* faceSet = dynamic_cast< SoFaceSet* >( vertexShape );
        SoLineSet* lineSet = dynamic_cast< SoLineSet* >( vertexShape );

        if (faceSet || lineSet)
        {
          int32_t* pNumVertices = NULL;
          if (faceSet)
          {
            faceSet->numVertices.setNum( numPrimitives );
            pNumVertices = faceSet->numVertices.startEditing();
          }
          if (lineSet)
          {
            lineSet->numVertices.setNum( numPrimitives );
            pNumVertices = lineSet->numVertices.startEditing();
          }

          for (int i = 0; i < numPrimitives; i++ )
          {
            if (faceSet)
              pNumVertices[ i ] = count;
            if (lineSet)
              pNumVertices[ i ] = 4;
          }
          if (faceSet)
            faceSet->numVertices.finishEditing();
          if (lineSet)
            lineSet->numVertices.finishEditing();
        }

        SoTriangleStripSet* triangleStrip = dynamic_cast< SoTriangleStripSet* >( vertexShape );
        if (triangleStrip)
        {
          int32_t* pNumVertices = NULL;

          triangleStrip->numVertices.setNum( numPrimitives );
          pNumVertices = triangleStrip->numVertices.startEditing();

          for (int i = 0; i < numPrimitives; i++)
            pNumVertices[i] = info.width * 2;

          triangleStrip->numVertices.finishEditing();
        }

        SoQuadMesh* quadMesh = dynamic_cast< SoQuadMesh* >( vertexShape );
        SoIndexedQuadMesh* indexedQuadMesh = dynamic_cast< SoIndexedQuadMesh* >( vertexShape );

        if (quadMesh || indexedQuadMesh)
        {
          if (quadMesh)
          {
            quadMesh->verticesPerRow = info.width;
            quadMesh->verticesPerColumn = info.height;
          }
          if (indexedQuadMesh)
          {
            indexedQuadMesh->verticesPerRow = info.width;
            indexedQuadMesh->verticesPerColumn = info.height;
          }
        }
    }
    delete[] vertices;
    delete[] colors;
    delete[] normals;
    delete[] texCoords;
    delete[] texCoords3;
  }
  vertexShape->vertexProperty = vp;
}

//------------------------------------------------------------------------------
bool copyToBuffer(SoGpuBufferObject* buffer, float* input, int componentsCount, 
                  int perPrimitiveVerticesCount, const GeometryTool::GeometryInfo& info)
{
  if (!input)
    return false;

  int size = info.width * info.height;

  float* ptr = (float *)buffer->map( SoBufferObject::SET );

  if (!ptr)
    return false;

  if (info.isIndexed || (info.type == GeometryTool::POINTS) )
  {
    memcpy( ptr, input, size * componentsCount * sizeof(float) );
  }
  else
  {
    if (!info.isStrip)
    {
      switch( perPrimitiveVerticesCount )
      {
        case 3:
        {
            // x---x          x
            // |  /          /|
            // | /     +    / |
            // |/          /  |
            // x          x---x
          for (int j = 0; j < info.height-1; j++)
            for (int i = 0; i < info.width-1; i++)
            {
              memcpy( ptr, &input[ ((j+1) * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              memcpy( ptr, &input[ ((j+1) * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              memcpy( ptr, &input[ (j * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;

              memcpy( ptr, &input[ ((j+1) * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              memcpy( ptr, &input[ (j * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              memcpy( ptr, &input[ (j * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;       
            }
            break;
        }
        case 4:
        {
          if (info.type == GeometryTool::LINES)
          {
            // x-------x
            // |       
            // |       
            // |       
            // x

            for (int j = 0; j < info.height-1; j++)
              for (int i = 0; i < info.width-1; i++)
              {
                memcpy( ptr, &input[ ((j+1) * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j+1) * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j+1) * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ (j * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              }
          }
          else
          {
            // x-------x
            // |       |
            // |       |
            // |       |
            // x-------x

            for (int j = 0; j < info.height-1; j++)
              for (int i = 0; i < info.width-1; i++)
              {
                memcpy( ptr, &input[ ((j+1) * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j+1) * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ (j * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ (j * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              }
          }
          break;
        }
        case 5:
          {
            for (int j = 0; j < info.height-1; j++)
              for (int i = 0; i < info.width-1; i++)
              {
                float* p0 = &input[ ((j+1) * info.width + i)*componentsCount ];
                float* p3 = &input[ (j * info.width + i)*componentsCount ];

                float pp0[4] = { (p0[0] + p3[0]) / 2.0f, (p0[1] + p3[1]) / 2.0f, (p0[2] + p3[2]) / 2.0f, (p0[3] + p3[3]) / 2.0f };

                //   (1)
                //    x-------x (2)
                //    |       |
                // (0)x       |
                //    |       |
                //    x-------x (3)
                //   (4)

                memcpy( ptr, &pp0, componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j+1) * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j+1) * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ (j * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ (j * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              }
          }
          break;
      }
    }
    else
    {
      switch( perPrimitiveVerticesCount )
      {
        case 3:
        case 4:
        {
          if (info.type == GeometryTool::LINES)
          {
            throw std::runtime_error( "LineStrips not supported!" );
          }
          else
          {
            for (int j = 0; j < info.height-1; j++)
              for (int i = 0; i < info.width; i++)
              {
                memcpy( ptr, &input[ (j * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j+1) * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              }
          }
          break;
        }
        case 5:
        {
          if (info.type == GeometryTool::LINE_STRIP)
          {
            //  x---x           x---x
            //  |  /            |  /
            //  | /     then    | /
            //  |/              |/
            //  x---x           x---x
            for (int j = 0; j < info.height - 1; j++)
              for (int i = 0; i < info.width - 1; i++)
              {
                memcpy( ptr, &input[ (j * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j+1) * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j+1) * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ (j * info.width + i)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
                memcpy( ptr, &input[ ((j) * info.width + i+1)*componentsCount ], componentsCount * sizeof( float ) ); ptr += componentsCount;
              }
              break;
          }
        }
        default:
          throw std::runtime_error( "Strips not supported for this type of shape!" );
      }
    }
  }

  buffer->unmap();

  return true;
}

//------------------------------------------------------------------------------
int computeVerticesCount(const GeometryTool::GeometryInfo& info, int perPrimitiveVerticesCount)
{
  if (perPrimitiveVerticesCount == 1)
    return info.width * info.height;

  if (info.isIndexed)
    return info.width * info.height;
  else
  {
    if (info.isStrip)
    {
      switch( perPrimitiveVerticesCount )
      {
        case 3:
        case 4:
          return info.width * 2;
        case 5:
          return (info.width - 1) * 5;
      }
    }
    else
    {
      switch( perPrimitiveVerticesCount )
      {
          // 2 lines per vertices * 2 vertices per line...
        case 2:
          return (info.width-1) * (info.height-1) * 2 * 2;
        case 3:
          return (info.width-1) * (info.height-1) * 2 * 3;
        case 4:
          return (info.width-1) * (info.height-1) * 4;
        case 5:
          return (info.width-1) * (info.height-1) * 5;
        default:
          return 0;
      }
    }
  }

  return 0;
}

//------------------------------------------------------------------------------
void
GeometryTool::generateSimplePrimitive(SoBufferedShape* bufferedShape, GeometryInfo& info, int count)
{
  int verticesCountPerStrip = computeVerticesCount( info, count );

  // If it's not a strip we have one huge *strip*...
  int verticesCount = verticesCountPerStrip;

  // If we are supposed to render strips we have height-1 strips.
  if (info.isStrip && !info.isIndexed)
    verticesCount *= (info.height-1);

  float* vertices = generateVertices( info );
  float* colors = generateColors( info, vertices );
  float* normals = generateNormals( info, vertices );
  float* texCoords = generateTexCoords( info, vertices );
  float* texCoords3 = generateTexCoords3( info, vertices );

  SoGpuBufferObject* verticesBufferObject = new SoGpuBufferObject();
  verticesBufferObject->setSize( verticesCount * 3 * sizeof( float ) );

  bufferedShape->vertexBuffer = verticesBufferObject;

  copyToBuffer( verticesBufferObject, vertices, 3, count, info );

  if (info.hasColors)
  {
    SoGpuBufferObject* colorsBufferObject = new SoGpuBufferObject( );
    colorsBufferObject->setSize( verticesCount * 3 * sizeof( float ) );

    bufferedShape->colorBuffer = colorsBufferObject;
    copyToBuffer( colorsBufferObject, colors, 3, count, info );
  }

  if (info.hasNormals)
  {
    SoGpuBufferObject* normalsBufferObject = new SoGpuBufferObject();
    normalsBufferObject->setSize( verticesCount * 3 * sizeof( float ) );

    bufferedShape->normalBuffer = normalsBufferObject;
    copyToBuffer( normalsBufferObject, normals, 3, count, info );
  }

  if (info.hasTexCoords)
  {
    SoGpuBufferObject* texCoordsBufferObject = new SoGpuBufferObject();
    texCoordsBufferObject->setSize( verticesCount * 2 * sizeof( float ) );

    bufferedShape->texCoordsBuffer = texCoordsBufferObject;
    bufferedShape->texCoordsComponentsCount = 2;
    copyToBuffer( texCoordsBufferObject, texCoords, 2, count, info );
  }
  else
    if (info.hasTexCoords3)
    {
      SoGpuBufferObject* texCoords3BufferObject = new SoGpuBufferObject();
      texCoords3BufferObject->setSize( verticesCount * 3 * sizeof( float ) );

      bufferedShape->texCoordsBuffer = texCoords3BufferObject;
      bufferedShape->texCoordsComponentsCount = 3;
      copyToBuffer( texCoords3BufferObject, texCoords3, 3, count, info );
    }

    delete[] vertices;
    delete[] colors;
    delete[] normals;
    delete[] texCoords;
    delete[] texCoords3;

    if (!info.isIndexed)
    {
      if (info.isStrip)
      {
        for (int i = 0; i < info.height-1; i++)
          bufferedShape->numVertices.set1Value( i, verticesCountPerStrip );
      }
      else
      {
        // Use either one polygon per numVertices entry.
        // TODO use primitive restart.
        if (info.type == POLYGON || info.type == TRIANGLE_FAN || info.type == LINE_LOOP)
        {
          bufferedShape->numVertices.setNum((info.height-1)*(info.width-1));
          int32_t* ptr = bufferedShape->numVertices.startEditing();
          std::fill( ptr, ptr + (info.height-1)*(info.width-1), 5 );
          bufferedShape->numVertices.finishEditing();
          bufferedShape->numVertices.touch();
        }
        else
          bufferedShape->numVertices.set1Value( 0, verticesCount );
      }
    }
}

//------------------------------------------------------------------------------
void
GeometryTool::setupSimplePrimitiveIndices(SoVertexShape* vertexShape, GeometryInfo& info, int count)
{
  int numPrimitives = info.numPrimitives;
  int nextIndex = 0;

  bool isStrip = info.isStrip;

  bool isQuadMesh = (  (dynamic_cast< SoQuadMesh* >( vertexShape) != NULL) ||
    (dynamic_cast< SoIndexedQuadMesh* >( vertexShape) != NULL)    ) ? true : false;

  bool isTriangleSet = (dynamic_cast< SoIndexedTriangleSet* >( vertexShape));

  SoIndexedShape* indexedShape = dynamic_cast< SoIndexedShape* >( vertexShape );
  if (indexedShape)
  {
    if (isStrip)
    {
      indexedShape->coordIndex.setNum( numPrimitives * info.width * 2 + numPrimitives );
    }
    else
      if ((count == 1))
        indexedShape->coordIndex.setNum( numPrimitives );
      else if (isQuadMesh)
        indexedShape->coordIndex.setNum( info.width * info.height );
      else if (isTriangleSet)
        indexedShape->coordIndex.setNum( numPrimitives * (count) );
      else
        indexedShape->coordIndex.setNum( numPrimitives * (count +1) );

      SoIndexedTriangleStripSet* indexedTriangleStrip = dynamic_cast< SoIndexedTriangleStripSet* >( vertexShape );

      int32_t* indices = indexedShape->coordIndex.startEditing();

      for (int j = 0; j < info.height; j++)
        for (int i = 0; i < info.width; i++)
        {
          if (count == 1)
            indices[ nextIndex++ ] = j * info.width + i;

          if (isQuadMesh)
          {
            indices[ nextIndex++ ] = j * info.width + i;
            continue;
          }

          if (isStrip)
          {
            if (j != info.height-1)
            {
              indices[ nextIndex++ ] = j * info.width + i;
              indices[ nextIndex++ ] = (j+1) * info.width + i;

              if (indexedTriangleStrip)
                if (i == info.width-1)
                  indices[ nextIndex++ ] = -1;
            }

            continue;
          }

          if ((j == info.height-1) || (i == info.width-1))
            continue;

          if (count == 3)
          {
            indices[ nextIndex++ ] = j * info.width + i;
            indices[ nextIndex++ ] = (j+1) * info.width + i + 1;
            indices[ nextIndex++ ] = j * info.width + i + 1;
            if ( !isTriangleSet )
              indices[ nextIndex++ ] = -1;

            indices[ nextIndex++ ] = j * info.width + i;
            indices[ nextIndex++ ] = (j+1) * info.width + i;
            indices[ nextIndex++ ] = (j+1) * info.width + i + 1;
            if ( !isTriangleSet )
              indices[ nextIndex++ ] = -1;
          }
          else if (count == 4)
          {
            indices[ nextIndex++ ] = j * info.width + i;
            indices[ nextIndex++ ] = (j+1) * info.width + i;
            indices[ nextIndex++ ] = (j+1) * info.width + i + 1;
            indices[ nextIndex++ ] = j * info.width + i + 1;
            indices[ nextIndex++ ] = -1;
          }
          else if (count == 5)
          {
            indices[ nextIndex++ ] = j * info.width + i;
            indices[ nextIndex++ ] = (j+1) * info.width + i;
            indices[ nextIndex++ ] = (j+1) * info.width + i + 1;
            indices[ nextIndex++ ] = j * info.width + i + 1;
            indices[ nextIndex++ ] = j * info.width + i;
            indices[ nextIndex++ ] = -1;
          }
        }

        indexedShape->coordIndex.finishEditing();
  }
}

//------------------------------------------------------------------------------
int computeIndicesCount(const GeometryTool::GeometryInfo& info, int perPrimitiveVerticesCount)
{
  if (perPrimitiveVerticesCount == 1)
    return info.width * info.height;

  if (info.isStrip)
  {
    switch( perPrimitiveVerticesCount )
    {
      case 3: 
      case 4:
        return info.width * 2;
      case 5:
        return (info.width - 1) * 5;
    }
  }
  else
  {
    switch( perPrimitiveVerticesCount )
    {
        // 2 lines per vertices * 2 vertices per line...
      case 2: return (info.width-1) * (info.height-1) * 2 * 2; break;
      case 3: return (info.width-1) * (info.height-1) * 2 * 3;
      case 4: return (info.width-1) * (info.height-1) * 4;
      case 5: return (info.width-1) * (info.height-1) * 5;
    }
  } 

  return 0;
}

//------------------------------------------------------------------------------
void
GeometryTool::setupSimplePrimitiveIndices(SoBufferedShape* vertexShape, GeometryInfo& info, int perPrimitiveVerticesCount)
{
  if (!info.isIndexed)
    return;

  int indicesCount = computeIndicesCount( info, perPrimitiveVerticesCount );
  int count = indicesCount;

  if (info.isStrip)
    count *= info.height;
  if (info.isPrimitiveRestart)
    count += info.height;

  SoGpuBufferObject* indicesBufferObject = new SoGpuBufferObject();
  indicesBufferObject->setSize( count * sizeof( unsigned int ) );

  unsigned int* ptr = (unsigned int*)indicesBufferObject->map( SoBufferObject::SET );

  if (!ptr)
    return;

  int idx = 0;
  if (info.isStrip)
  {
    switch (perPrimitiveVerticesCount)
    {
      case 3:
      case 4:
      {
        for (int j = 0; j < info.height-1; j++)
        {
          for (int i = 0; i < info.width; i++)
          {
            ptr[idx++] = j * info.width + i;
            ptr[idx++] = (j+1) * info.width + i;
          }
          if (!info.isPrimitiveRestart)
            vertexShape->numVertices.set1Value( j, indicesCount );
          else
            ptr[idx++] = vertexShape->primitiveRestartValue.getValue();
        }
        break;
      }
      case 5:
      {
        if ( info.type == GeometryTool::LINE_STRIP)
        {
          for (int j = 0; j < info.height - 1; j++)
          {
            for (int i = 0; i < info.width - 1; i++)
            {
              ptr[idx++] = j * info.width + i;
              ptr[idx++] = (j+1) * info.width + i;
              ptr[idx++] = (j+1) * info.width + (i+1);
              ptr[idx++] = j * info.width + i;
              ptr[idx++] = j * info.width + (i+1);
            }
            if (!info.isPrimitiveRestart)
              vertexShape->numVertices.set1Value( j, indicesCount );
            else
              ptr[idx++] = vertexShape->primitiveRestartValue.getValue();
          }
        }
        break;
      }
      default:
        break;
    }
  }
  else
  {
    if (perPrimitiveVerticesCount == 5)
    {
      vertexShape->numVertices.setNum( (info.height - 1) * (info.width - 1) );
      int32_t* ptr = vertexShape->numVertices.startEditing();
      std::fill( ptr, ptr + (info.height-1)*(info.width-1), 5);
      vertexShape->numVertices.finishEditing();
      vertexShape->numVertices.touch();
    }
    else
      vertexShape->numVertices.set1Value( 0, count );

    switch (perPrimitiveVerticesCount)
    {
      case 1:
      {
        for (int i = 0; i < count; i++)
          ptr[ i ] = i;
        break;
      }
      case 3:
      {
        for (int j = 0; j < info.height-1; j++)
          for (int i = 0; i < info.width-1; i++)
          {
            ptr[ idx++ ] = j * info.width + i;
            ptr[ idx++ ] = (j+1) * info.width + i;
            ptr[ idx++ ] = (j+1) * info.width + i+1;

            ptr[ idx++ ] = (j+1) * info.width + i+1;
            ptr[ idx++ ] = j * info.width + i + 1;
            ptr[ idx++ ] = j * info.width + i;
          }
        break;
      }

      case 4:
      {
        if (info.type == GeometryTool::LINES)
        {
          for (int j = 0; j < info.height-1; j++)
            for (int i = 0; i < info.width-1; i++)
            {
              ptr[ idx++ ] = (j+1) * info.width + i;
              ptr[ idx++ ] = j * info.width + i;
              ptr[ idx++ ] = j * info.width + i + 1;
              ptr[ idx++ ] = j * info.width + i;
            }
        }
        else
        {
          for (int j = 0; j < info.height-1; j++)
            for (int i = 0; i < info.width-1; i++)
            {
              ptr[ idx++ ] = (j+1) * info.width + i;
              ptr[ idx++ ] = (j+1) * info.width + i+1;
              ptr[ idx++ ] = j * info.width + i + 1;
              ptr[ idx++ ] = j * info.width + i;
            }
        }
        break;
      }

      case 5:
      {
        for (int j = 0; j < info.height-1; j++)
          for (int i = 0; i < info.width-1; i++)
          {
            ptr[ idx++ ] = (j+1) * info.width + i;
            ptr[ idx++ ] = (j+1) * info.width + i+1;
            ptr[ idx++ ] = j * info.width + i + 1;
            ptr[ idx++ ] = j * info.width + i;
            ptr[ idx++ ] = (j+1) * info.width + i;
          }

        break;
      }
    }
  }

  indicesBufferObject->unmap();
  vertexShape->indexBuffer = indicesBufferObject;
  vertexShape->indexType = SbDataType::UNSIGNED_INT32;

  if (info.isPrimitiveRestart)
    vertexShape->numVertices.set1Value( 0, idx);
}

//------------------------------------------------------------------------------
void
GeometryTool::generateHole(SoVertexShape* vertexShape, GeometryInfo& info, int count)
{
  bool isStrip = (  (dynamic_cast< SoTriangleStripSet* >( vertexShape) != NULL) ||
    (dynamic_cast< SoIndexedTriangleStripSet* >( vertexShape) != NULL)    ) ? true : false;

  SoFaceSet* faceSet = dynamic_cast< SoFaceSet* >( vertexShape );
  SoQuadMesh* quadMesh = dynamic_cast< SoQuadMesh* >( vertexShape );
  SoTriangleStripSet* triangleStripSet = dynamic_cast< SoTriangleStripSet* >( vertexShape );

  SoLineSet* lineSet = dynamic_cast< SoLineSet* >( vertexShape );

  int isIndexed = info.isIndexed;

  info.numVertices = 8;
  info.numPrimitives = 2;

  SoVertexProperty* vp = new SoVertexProperty;
  vertexShape->vertexProperty = vp;

  if (quadMesh)
  {
    quadMesh->verticesPerColumn = 2;
    quadMesh->verticesPerRow = 4;
  }

  if (isIndexed || (count == 1))
  {
    vp->vertex.setNum( 8 );
    SbVec3f* vertices = vp->vertex.startEditing();

    vertices[0].setValue( 0.0, 0.0,  0.0 );
    vertices[1].setValue( 1.0, 0.0,  0.0 );
    vertices[2].setValue( 1.0, 1.0,  0.0 );
    vertices[3].setValue( 0.0, 1.0,  0.0 );

    vertices[4].setValue( 0.25, 0.25,  0.0 );
    vertices[5].setValue( 0.75, 0.25,  0.0 );
    vertices[6].setValue( 0.75, 0.75,  0.0 );
    vertices[7].setValue( 0.25, 0.75,  0.0 );

    vp->vertex.finishEditing();
    return;
  }


  if (count == 3)
  {
    if (!isStrip)
    {
      vp->vertex.setNum( 12 );
      SbVec3f* vertices = vp->vertex.startEditing();

      vertices[0].setValue( 0.0, 0.0,  0.0 );
      vertices[1].setValue( 1.0, 1.0,  0.0 );
      vertices[2].setValue( 0.0, 1.0,  0.0 );

      vertices[3].setValue( 0.0, 0.0,  0.0 );
      vertices[4].setValue( 1.0, 0.0,  0.0 );
      vertices[5].setValue( 1.0, 1.0,  0.0 );

      // The hole
      vertices[6].setValue( 0.25, 0.25,  0.0 );
      vertices[7].setValue( 0.75, 0.75,  0.0 );
      vertices[8].setValue( 0.25, 0.75,  0.0 );

      vertices[9].setValue( 0.25, 0.25,  0.0 );
      vertices[10].setValue( 0.75, 0.25,  0.0 );
      vertices[11].setValue( 0.75, 0.75,  0.0 ); 

      vp->vertex.finishEditing();

      if (faceSet)
      {
        faceSet->numVertices.set1Value( 0, 6 );
        faceSet->numVertices.set1Value( 1, 6 );
      }
    }
    else
    {
      vp->vertex.setNum( 8 );
      SbVec3f* vertices = vp->vertex.startEditing();

      vertices[0].setValue( 1.0, 0.0,  0.0 );
      vertices[1].setValue( 1.0, 1.0,  0.0 );
      vertices[2].setValue( 0.0, 0.0,  0.0 );
      vertices[3].setValue( 0.0, 1.0,  0.0 );

      vertices[4].setValue( 0.75, 0.25,  0.0 );
      vertices[5].setValue( 0.75, 0.75,  0.0 );
      vertices[6].setValue( 0.25, 0.25,  0.0 );
      vertices[7].setValue( 0.25, 0.75,  0.0 );

      vp->vertex.finishEditing();

      if (triangleStripSet)
      {
        triangleStripSet->numVertices.set1Value( 0, 4 );
        triangleStripSet->numVertices.set1Value( 1, 4 );
      } 
    }
  }
  else if (count == 4)
  {
    vp->vertex.setNum( 2*count + (lineSet?2:0) );
    SbVec3f* vertices = vp->vertex.startEditing();
    int idx = 0;

    vertices[idx++].setValue( 0.0, 0.0,  0.0 );
    vertices[idx++].setValue( 1.0, 0.0,  0.0 );
    vertices[idx++].setValue( 1.0, 1.0,  0.0 );
    vertices[idx++].setValue( 0.0, 1.0,  0.0 );
    if (lineSet)
      vertices[idx++].setValue( 0.0, 0.0,  0.0 );


    vertices[idx++].setValue( 0.25, 0.25,  0.0 );
    vertices[idx++].setValue( 0.75, 0.25,  0.0 );
    vertices[idx++].setValue( 0.75, 0.75,  0.0 );
    vertices[idx++].setValue( 0.25, 0.75,  0.0 );
    if (lineSet)
      vertices[idx++].setValue( 0.25, 0.25,  0.0 );

    vp->vertex.finishEditing();

    if (faceSet)
    {
      faceSet->numVertices.set1Value( 0, count );
      faceSet->numVertices.set1Value( 1, count );
    }
    if (lineSet)
    {
      lineSet->numVertices.set1Value( 0, count+1 );
      lineSet->numVertices.set1Value( 1, count+1 );
    }
  }
  else if (count == 5)
  {
    vp->vertex.setNum( 2*count + (lineSet?2:0) );
    SbVec3f* vertices = vp->vertex.startEditing();
    int idx = 0;

    vertices[idx++].setValue( 0.0, 0.0,  0.0 );
    vertices[idx++].setValue( 1.0, 0.0,  0.0 );
    vertices[idx++].setValue( 1.0, 1.0,  0.0 );
    vertices[idx++].setValue( 0.0, 1.0,  0.0 );
    vertices[idx++].setValue( 0.0, 0.0,  0.0 );

    if (lineSet)
      vertices[idx++].setValue( 0.0, 0.0,  0.0 );


    vertices[idx++].setValue( 0.25, 0.25,  0.0 );
    vertices[idx++].setValue( 0.75, 0.25,  0.0 );
    vertices[idx++].setValue( 0.75, 0.75,  0.0 );
    vertices[idx++].setValue( 0.25, 0.75,  0.0 );
    vertices[idx++].setValue( 0.25, 0.25,  0.0 );
    if (lineSet)
      vertices[idx++].setValue( 0.25, 0.25,  0.0 );

    vp->vertex.finishEditing();

    if (faceSet)
    {
      faceSet->numVertices.set1Value( 0, count );
      faceSet->numVertices.set1Value( 1, count );
    }
    if (lineSet)
    {
      lineSet->numVertices.set1Value( 0, count+1 );
      lineSet->numVertices.set1Value( 1, count+1 );
    }
  }
}

//------------------------------------------------------------------------------
void
GeometryTool::generateHole(SoBufferedShape* /*vertexShape*/, GeometryInfo& /*info*/, int /*count*/)
{
  throw std::runtime_error( "No hole support for buffered shapes!\n");
}

//------------------------------------------------------------------------------
void
GeometryTool::setupHoleIndices(SoVertexShape* vertexShape, GeometryInfo& /*info*/, int count)
{
  bool isStrip = (  (dynamic_cast< SoTriangleStripSet* >( vertexShape) != NULL) ||
    (dynamic_cast< SoIndexedTriangleStripSet* >( vertexShape) != NULL)    ) ? true : false;

  SoIndexedShape* indexedShape = dynamic_cast< SoIndexedShape* >( vertexShape );

  if (!indexedShape)
    return;

  if (isStrip)
  {
    indexedShape->coordIndex.setNum( 10 );
    int32_t* indices = indexedShape->coordIndex.startEditing();   
    indices[0] = 1; indices[1] = 2; indices[2] = 0; indices[3] = 3; indices[4] = -1;
    indices[5] = 5; indices[6] = 6; indices[7] = 4; indices[8] = 7; indices[9] = -1;
    indexedShape->coordIndex.finishEditing();
  }
  else if (count == 1)
  {
    indexedShape->coordIndex.setNum( 10 );
    int32_t* indices = indexedShape->coordIndex.startEditing();
    indices[0] = 0; indices[1] = 1; indices[2] = 2; indices[3] = 3; indices[4] = -1;
    indices[5] = 4; indices[6] = 5; indices[7] = 6; indices[8] = 7; indices[9] = -1;
    indexedShape->coordIndex.finishEditing();
  }
  else if (count == 3)
  {
    indexedShape->coordIndex.setNum( 16 );
    int32_t* indices = indexedShape->coordIndex.startEditing();
    indices[0] = 0; indices[1] = 1; indices[2] = 2; indices[3] = -1;
    indices[4] = 0; indices[5] = 2; indices[6] = 3; indices[7] = -1;

    indices[8] = 4; indices[9] = 5; indices[10] = 6; indices[11] = -1;
    indices[12] = 4; indices[13] = 6; indices[14] = 7; indices[15] = -1;

    indexedShape->coordIndex.finishEditing();
  }
  else if (count == 4 && dynamic_cast< SoIndexedFaceSet* >( vertexShape ))
  {
    indexedShape->coordIndex.setNum( 10 );
    int32_t* indices = indexedShape->coordIndex.startEditing();
    indices[0] = 0; indices[1] = 1; indices[2] = 2; indices[3] = 3; indices[4] = -1;
    indices[5] = 4; indices[6] = 5; indices[7] = 6; indices[8] = 7; indices[9] = -1;
    indexedShape->coordIndex.finishEditing();
  }
  else if (count == 5 && dynamic_cast< SoIndexedFaceSet* >( vertexShape ))
  {
    // TODO..TODOCHECK
    SoDebugError::post(OIV_FUNCTION,"Polygonal indexed shape with with Holes should be checked in benchTool");
    indexedShape->coordIndex.setNum( 10 );
    int32_t* indices = indexedShape->coordIndex.startEditing();
    indices[0] = 0; indices[1] = 1; indices[2] = 2; indices[3] = 3; indices[4] = -1;
    indices[5] = 4; indices[6] = 5; indices[7] = 6; indices[8] = 7; indices[9] = -1;
    indexedShape->coordIndex.finishEditing();
  }
  else if (count == 4 && dynamic_cast< SoIndexedQuadMesh* >( vertexShape ))
  {
    indexedShape->coordIndex.setNum( 8 );
    int32_t* indices = indexedShape->coordIndex.startEditing();
    indices[0] = 0; indices[1] = 1; indices[2] = 2; indices[3] = 3;
    indices[4] = 4; indices[5] = 5; indices[6] = 6; indices[7] = 7;
    indexedShape->coordIndex.finishEditing();

    SoIndexedQuadMesh* indexedQuadMesh = (SoIndexedQuadMesh*)vertexShape;
    indexedQuadMesh->verticesPerColumn = 2;
    indexedQuadMesh->verticesPerRow = 4;
  }
  else if (count == 4 && dynamic_cast< SoIndexedLineSet* >( vertexShape ))
  {
    indexedShape->coordIndex.setNum( 12 );
    int32_t* indices = indexedShape->coordIndex.startEditing();
    indices[0] = 0; indices[1] = 1; indices[2] = 2; indices[3] = 3; indices[4] = 0; indices[5] = -1;
    indices[6] = 4; indices[7] = 5; indices[8] = 6; indices[9] = 7; indices[10] = 4; indices[11] = -1;
    indexedShape->coordIndex.finishEditing();
  }
  else if (count == 5 && dynamic_cast< SoIndexedLineSet* >( vertexShape ))
  {
    // TODO..TODOCHECK
    SoDebugError::post(OIV_FUNCTION,"Polygonal indexed shape with with Holes should be checked in benchTool");

    indexedShape->coordIndex.setNum( 12 );
    int32_t* indices = indexedShape->coordIndex.startEditing();
    indices[0] = 0; indices[1] = 1; indices[2] = 2; indices[3] = 3; indices[4] = 0; indices[5] = -1;
    indices[6] = 4; indices[7] = 5; indices[8] = 6; indices[9] = 7; indices[10] = 4; indices[11] = -1;
    indexedShape->coordIndex.finishEditing();
  }
}

//------------------------------------------------------------------------------
void
GeometryTool::setupHoleIndices(SoBufferedShape* /*vertexShape*/, GeometryInfo& /*info*/, int /*count*/)
{
}


