/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <Inventor/SbColor.h>
#include <Inventor/SoPreferences.h>

#include <Inventor/Qt/SoQt.h>
#include <Inventor/devices/SoCpuDevice.h>
#include <Inventor/devices/SoGLDevice.h>

#include <Inventor/sensors/SoAlarmSensor.h>

#include <csignal>

#include <bencher.h>
#include <viewer.h>
#include <geometry.h>
#include <reporting.h>
#include <ivShapeBenchmarkGui_Design.h>

#ifdef _WIN32
  #include <windows.h>
#endif

// Functions

void performFullTest();
void performSingleTest();

void usage();

void invalidOptionError( SbString parameter, SbString value );
void invalidOptionValueError( SbString parameter, SbString value );

void printEnvironmentVariables();

// Data
int g_windowWidth = 640;
int g_windowHeight = 480;

SbBool g_fullBench = FALSE;

SbBool g_csvOutput = FALSE;

SbString g_htmlOutputPath = "";
SbBool g_dumpDisplay = FALSE;
SbString g_dumpDisplayFilename = "";
SbString g_dumpSGFilename = "";

SbString g_benchString;

// Geometry data if bench string is not specified
bool g_viewingMode = true;
bool g_useGui = false;
SbString g_renderingMode = "STANDALONE";
bool g_useOivNormals = false;
float g_neighborTolerance = 0.f;
bool g_useOivColors = false;
bool g_useOivTexCoords = true;
bool g_useOivTexCoords3 = true;
GeometryTool::BindingType g_normalsBinding = GeometryTool::PER_VERTEX_INDEXED;
GeometryTool::BindingType g_colorsBinding = GeometryTool::PER_VERTEX_INDEXED;
int g_geometryIndex = 0;
int g_geometryWidth = 100;
int g_geometryHeight = 100;
int g_geometryCount = 1;
int g_benchFrameCount = -1;
int g_benchFrameStart = 0;
int g_benchFrameEnd = -1;
int g_benchFrameSteps = 1;
SbString g_geometryType = "FaceSet:3";
SbString g_csvSepChar = ";";

QWidget* g_parent = NULL;

void parseArguments( int count, char** arguments )
{
  // Step 1: do we want some help ?
  for (int i = 1; i < count; i++)
    if (!strcmp( arguments[i], "--help" ) || !strcmp( arguments[i], "-h" ))
      usage();

  int index = 1;
  SbBool okFlag;

  if ( count == index )
    g_useGui = 1;

  while (index < count)
  {
    // - We format the parameters
    SbString option = arguments[index];
    SbString value;

    int prevIndex = index;

    if (index < count - 1)
      value = arguments[index+1];

    if (option.find( "-" ) != 0)
      invalidOptionError( option, value );

    // Base options to define the configuration

    if ((option == "--displayWidth") || (option == "--display-width"))
    {
      g_windowWidth = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if ((option == "--displayHeight") || (option == "--display-height"))
    {
      g_windowHeight = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if ((option == "--fullBench") || (option == "--bench-all"))
    {
      g_fullBench = TRUE;
      index ++;
    }

    if ((option == "--htmlOutput") || (option == "--output-html"))
    {
      index += 2;
      g_htmlOutputPath = value;
    }

    if ((option == "--csvOutput") || (option == "--output-csv"))
    {
      index ++;
      g_csvOutput = TRUE; 
    }

    if ( option == "--csvSepChar" )
    {
      index += 2;
      g_csvSepChar = value.getSubString(0, 1);
    }

    if ((option == "--dumpDisplay") || (option == "--dump-display"))
    {
      g_dumpDisplay = TRUE; 
      index ++;
    }

    if ((option == "--dumpDisplayFile") || (option == "--dump-display-file"))
    {
      index += 2;
      g_dumpDisplayFilename = value; 
    }

    if ((option == "--dumpSG") || (option == "--dump-scenegraph"))
    {
      index += 2;
      g_dumpSGFilename = value; 
    }

    // Geometry options (unused if bench-string is defined.
    if (option == "--viewing-mode")
    {
      g_viewingMode = true;
      index ++;
    }

    if (option == "--rendering-mode")
    {
      index += 2;
      g_renderingMode = value;
    }

    if (option == "--generate-normals")
    {
      g_useOivNormals = true;
      index ++;
    }

    if (option == "--neighbor-tolerance")
    {
      // g_useOivNormals = true; // this is forced
      g_neighborTolerance = value.toFloat( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--generate-colors")
    {
      g_useOivColors = true;
      index ++;
    }

    if (option == "--generate-texcoords")
    {
      g_useOivTexCoords = true;
      index ++;
    }

    if (option == "--generate-texcoords3")
    {
      g_useOivTexCoords3 = true;
      index ++;
    }

    if (option == "--normals-binding")
    {
      index += 2;

      g_normalsBinding = GeometryTool::INVALID;
      if (value.getLength() > 0)
        g_normalsBinding = GeometryTool::getBindingType( value[0] );

      if (g_normalsBinding == GeometryTool::INVALID)
        invalidOptionError( option, value );
    }

    if (option == "--normals-binding")
    {
      index += 2;

      g_normalsBinding = GeometryTool::INVALID;
      if (value.getLength() > 0)
        g_normalsBinding = GeometryTool::getBindingType( value[0] );

      if (g_normalsBinding == GeometryTool::INVALID)
        invalidOptionError( option, value );
    }

    if (option == "--colors-binding")
    {
      index += 2;

      g_colorsBinding = GeometryTool::INVALID;
      if (value.getLength() > 0)
        g_colorsBinding = GeometryTool::getBindingType( value[0] );

      if (g_colorsBinding == GeometryTool::INVALID)
        invalidOptionError( option, value );
    }

    if (option == "--geometry")
    {
      g_geometryIndex = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--geometry-type")
    {
      index += 2;
      g_geometryType = value; 
    }

    if (option == "--geometry-width")
    {
      g_geometryWidth = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--geometry-height")
    {
      g_geometryHeight = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--geometry-count")
    {
      g_geometryCount = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--bench-frame-count")
    {
      g_benchFrameCount = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--bench-frame-start")
    {
      g_benchFrameStart = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--bench-frame-end")
    {
      g_benchFrameEnd = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--bench-frame-steps")
    {
      g_benchFrameSteps = value.toInt( &okFlag );
      if (!okFlag) invalidOptionValueError( option, value );
      index += 2;
    }

    if (option == "--gui")
    {
      g_useGui = true;
      index ++;
    }

    // Bench string option

    if ((option == "--benchString") || (option == "--bench-string"))
    {
      index += 2;
      g_benchString = value; 
    }

    if (index == prevIndex)
    {
      invalidOptionError( option, value );
    }
  }

  if (g_fullBench && g_benchString.getLength())
  {
    printf( "Error/ The options --bench-full and --bench-string are incompatible\n");
    exit( 1 );
  }

  // Ok we store and use the custom parameters
  SoPreferences::setInt( "BENCHER_VIEWER_WIDTH", g_windowWidth );
  SoPreferences::setInt( "BENCHER_VIEWER_HEIGHT", g_windowHeight );
}
void
startTests( void *data, SoSensor *sensor )
{
  if ( g_fullBench )
    performFullTest();
  else
    performSingleTest();
  exit( 0 );
}


// Code
int main(int argc, char** argv)
{

  #ifdef _WIN32
    SetErrorMode(SEM_FAILCRITICALERRORS | SEM_NOGPFAULTERRORBOX);
  #endif

  // Step 1: We set a default color for the background.
  SoPreferences::setColor( "BENCHER_VIEWER_BACKGROUND_COLOR", SbColor( 0.0, 0.0, 1.0 ) );

  // Step 2: We parse the arguments.
  parseArguments( argc, argv );

  // Step 3: Let's go for the bench.
  g_parent = SoQt::init( "ivShapeBenchmark-SingleViewer" );

  if ( !g_useGui )
  {
    SoAlarmSensor* alarm = new SoAlarmSensor( startTests, NULL );
    alarm->setTimeFromNow( SbTime( 0.01f ) );
    alarm->schedule();

    SoQt::mainLoop();
  }
  else
  {
    // If we use gui, the main loop is started after displaying widget
    performSingleTest();
  }

  return 0;
}

void usage()
{
  printf("Usage:\n"
    " --display-width     value The width of the rendering display (Default: 640)\n"
    " --display-height    value The height of the rendering display (Default: 480)\n"
    "\n"
    " --bench-all              Perform a full bench of all the shapes (Default: Not active)\n"
    "\n"
    " --dump-display       allows to dump the display to a PNG file (Default: not active)\n"
    " --dump-display-file  value Dump the framebuffer to a PNG file, value is the filename (Default: the corresponding benchString)\n"
    " --dump-scenegraph   value Dump the SceneGraph to an IV file, value is the filename (Default: "")\n"
    " --output-html       value Set the path for the HTML output, value is the output path (Default: ./output)\n"
    " --output-csv        value Enable the dump, to the console, of the bench information in CSV format.(Default: Not active)\n"
    " --csvSepChar        value To set the csv file separator, when output-csv is on.(Default: ;)\n"
    "\n"
    "Optional (not used if the option --benchString is used)\n\n"
    " --viewing-mode           Enter in interactive viewing mode, no benchmarking is performed.\n"
    " --rendering-mode    value The ScaleViz/Multipipe rendering mode (ex: 2_Pipes_(Depth)), (Default STANDALONE).\n"
    " --generate-normals       Use OIV normals generator (Default: not active)\n"
    " --neighbor-tolerance value Tolerance to be used when generate-normalsis active (Default: 0.f)\n"
    " --generate-colors        Use OIV colors generator (Default: not active)\n"
    " --generate-texcoords     Use OIV textures coordinates generator (Default: active)\n"
    " --generate-texcoords3    Use OIV textures coordinates generator for 3D textures (Default: active)\n"
    " --normals-binding   value Normals binding (default V): O for overall, V per vertex, F per face, G per face indexed, W per vertex indexed.\n"
    " --colors-binding    value Colors binding (default V): O for overall, V per vertex, F per face, G per face indexed, W per vertex indexed.\n"
    " --geometry          value Use the embedded geometry (0-3) (Default 0).\n"
    " --geometry-type     value The shape type used for the geometry (Defaults: FACESET:3).\n"
    " --geometry-width    value The width of the geometry in vertices (Default 100).\n"
    " --geometry-height   value The height of the geometry in vertices (Default 100).\n"
    " --geometry-count    value The number of instances of the displayed geometry (Default 1).\n"
    " --bench-frame-count value The number of frame to bench, -1 for infinite (Default -1) else must not be lower than 11\n"
    " --bench-frame-start value The index of the first benched frame (Default 0).\n"
    " --bench-frame-end   value The index of the last benched frame, -1 for all (Default -1).\n"
    " --bench-frame-steps value The number of frames rendered between each benched frame (Default 0).\n"
    "\n"
    "\n"
    " --bench-string      value  Performs a single bench define by the specified bench string []\n"
    "\t\"OIVShapeName FLAGS OVERALL_FLAGS geomIndex geomWidth geomHeight geomCount frameNum benchFrameStart benchFrameEnd benchFrameStepping renderMode neighborTolerance\"\n"
    "\n"
    "\t\t FLAGS is a combination of VNC where:\n"
    "\t\t - V stands for vertices (mandatory),\n"
    "\t\t - N is normal,\n"
    "\t\t - C is color,\n"
    "\t\t - T for textures, 3 for 3D textures\n\n"
    "\t\t OVERALL_FLAGS: xx where:\n"
    "\t\t - each x is O for overall, V per vertex, F per face, G per face indexed, W per vertex indexed.\n"
    "\t\t - The first x is for the normals\n"
    "\t\t - The second x is for the colors\n"
    "\n"
    "\t\t frameNum is the number of frame to render; -1 for infinite; -2 to display just display viewer\n\n"
    "\t\t benchFrameStart is the number of the first benched frame.\n\n"
    "\t\t benchFrameEnd is the number of the last benched frame; -1 for all the frames.\n\n"
    "\t\t benchFrameStepping is the number of frames to use for a single bench result.\n\n"
    "\t\t renderMode is the configuration to load at application startup (ex: 2_Pipes_(Depth)). default is STANDALONE\n\n\n"
    "\t\t neighborTolerance is the tolerance to be used if normals are to be autogenerated. default is 0.f\n\n\n"
    );
  printf("Copyright FEI S.A.S.\n");

  exit( 1 );
}

void invalidOptionError( SbString parameter, SbString value )
{
  printf( "Invalid option %s with value %s\n", parameter.toLatin1(), value.toLatin1() );
  exit( 1 );
}

void invalidOptionValueError( SbString parameter, SbString value )
{
  printf( "Invalid value %s for the option %s\n", value.toLatin1(), parameter.toLatin1() );
  exit( 1 );
}

void performFullTest()
{
  SbBool singleViewer = SoPreferences::getBool( "BENCHER_SINGLE_VIEWER", TRUE );

  try
  {
    REPORTING_INIT();

    printf(" ----- Beginning benching -----\n");

    Viewer* viewer = NULL;

    if (singleViewer)
    {
      viewer = new Viewer( g_parent );
      g_parent->show();
    }

#define BENCH_COUNT 4
#define REPORTS_COUNT 4

#define BENCH_GUI

    std::string reportsDesc[ REPORTS_COUNT ] = { "Reference", "Holes", "10 shapes + vertex attrib", "1000 shapes" };

    std::string flags[ REPORTS_COUNT ] = {"VN", "VC3", "VCA3", "VC3"};
    std::string bindingFlags[ REPORTS_COUNT ] = {"VV", "VV", "VV", "VV"};
    std::string bindingIndexedFlags[ REPORTS_COUNT ] = {"WW", "WW", "WW", "WW"};

    int geomIndex[ REPORTS_COUNT ] = { 0, 1, 0, 0 };
    int geomWidth[ REPORTS_COUNT ] = { 200, 200, 150, 100 };
    int geomHeight[ REPORTS_COUNT ] = { 160, 160, 120, 80 };
    int geomCount[ REPORTS_COUNT ] = { 1, 1, 9, 100 };
#ifdef BENCH_GUI
    int frameCount[ REPORTS_COUNT ] = { -2, -2, -2, -2 };
#else
    int frameCount[ REPORTS_COUNT ] = { 200, 200, 200, 200 };
#endif
    int frameStart[ REPORTS_COUNT ] = { 0, 0, 0, 0 };
    int frameEnd[ REPORTS_COUNT ] = { -1, -1, -1, -1 };
    int frameStepping[ REPORTS_COUNT ] = { 1, 1, 1, 1 };
    std::string configName[ REPORTS_COUNT ] = { "STANDALONE", "STANDALONE", "STANDALONE", "STANDALONE" };

    for (int reportIdx = 0; reportIdx < REPORTS_COUNT; reportIdx++)
    {
      REPORTING_NEW_REPORT();

      REPORTING_REPORT_SET_DESCRIPTION( reportsDesc[ reportIdx ] );

      std::string tests[BENCH_COUNT];

      tests[0] = BenchTool::buildTestBunch( flags[reportIdx], bindingFlags[reportIdx], bindingIndexedFlags[reportIdx], geomIndex[reportIdx], geomWidth[reportIdx], geomHeight[reportIdx], geomCount[reportIdx], frameCount[reportIdx], frameStart[reportIdx], frameEnd[reportIdx], frameStepping[reportIdx], configName[reportIdx] );
      tests[1] = BenchTool::buildTestBunch( flags[reportIdx], bindingFlags[reportIdx], bindingIndexedFlags[reportIdx], geomIndex[reportIdx], geomWidth[reportIdx], geomHeight[reportIdx], geomCount[reportIdx], frameCount[reportIdx], frameStart[reportIdx], frameEnd[reportIdx], frameStepping[reportIdx], configName[reportIdx] );
      tests[2] = BenchTool::buildTestBunch( flags[reportIdx], bindingFlags[reportIdx], bindingIndexedFlags[reportIdx], geomIndex[reportIdx], geomWidth[reportIdx], geomHeight[reportIdx], geomCount[reportIdx], frameCount[reportIdx], frameStart[reportIdx], frameEnd[reportIdx], frameStepping[reportIdx], configName[reportIdx] );
      tests[3] = BenchTool::buildTestBunch( flags[reportIdx], bindingFlags[reportIdx], bindingIndexedFlags[reportIdx], geomIndex[reportIdx], geomWidth[reportIdx], geomHeight[reportIdx], geomCount[reportIdx], frameCount[reportIdx], frameStart[reportIdx], frameEnd[reportIdx], frameStepping[reportIdx], configName[reportIdx] );

      for (int testIdx = 0; testIdx < BENCH_COUNT; testIdx++)
      {
        switch( testIdx )
        {
        case 0:
        case 2:
          REPORTING_TEST_SET_DESCRIPTION( "Reference with default OpenInventor settings" );
          viewer->disbaleVBO( false );
          break;
        case 1:
        case 3:
          REPORTING_TEST_SET_DESCRIPTION( "No VBO" );
          viewer->disbaleVBO( true );
          break;
        }


        REPORTING_NEW_TEST_BUNCH();
        {

          REPORTING_BENCH( "output", viewer, tests[ REPORTING_TEST_GET_INDEX() ] );
          REPORTING_FLUSH_RESULTS();

        }
        REPORTING_END_TEST_BUNCH();
      }

      REPORTING_END_REPORT()
    }

    if (viewer)
    {
      delete viewer;
      SoQt::finish();
    }

    printf(" ----- Bench done -----\n");
    REPORTING_FINISH();
  }
  catch( const std::exception& e )
  {
    printf( "Exception: %s\n", e.what() );
    exit( 1 );
  }
}

void signalHandler(int signo)
{
  if (g_csvOutput)
  {
    printf( "0;%s\n", g_benchString.toLatin1());
  }

  printf("[E] Signal %d, bench string: %s\n", signo, g_benchString.toLatin1() );

  printf("[D] ----- Bench done (with crash)-----\n");

  exit( 1 );
}

Viewer* g_viewer = NULL;
ivShapeBenchmarkGui* g_guiDialog = NULL;

bool performSingleTestFromString(const QString& benchString)
{
  bool ret = true;
  printf( "[D] String: %s\n", benchString.toLatin1().data() );
  BenchTool bencher;
  std::vector<BenchTool::BenchResult> results;

  results = bencher.bench( g_dumpSGFilename.toStdString(), 
    g_dumpDisplay,
    g_dumpDisplayFilename.toStdString(),
    g_viewer, 
    benchString.toLatin1().data() );

  return ret;
}

void changeAnimationRate(float newRate)
{
  g_viewer->setAnimationRate(newRate);
  g_viewer->scheduleRedraw();
}

void changeDrawStyle(const QString& value)
{
  g_viewer->setDrawStyle(value);
  g_viewer->scheduleRedraw();
}

void changeTransparencyType( const QString& value )
{
  g_viewer->setTransparencyType( value );
  g_viewer->scheduleRedraw();
}

void pickCallback(const Viewer::PickInformation& information)
{
  if (g_guiDialog)
    g_guiDialog->setPickingInformation(information.screenPosition, information.isShapePicked, information.worldCoords, information.normal, information.texCoord);
}

void performSingleTest()
{
  /*
  #if defined( WIN32 ) || defined( WIN64 )
  SetErrorMode(SEM_NOGPFAULTERRORBOX | SEM_FAILCRITICALERRORS); 
  #endif

  std::signal(SIGINT, signalHandler); 
  std::signal(SIGILL, signalHandler);
  std::signal(SIGFPE, signalHandler);
  std::signal(SIGSEGV, signalHandler);
  std::signal(SIGTERM, signalHandler);
  std::signal(SIGABRT, signalHandler);
  */

  try
  {

    SoDB::init(); // this is needed by following MemoryTool => SoCpuDevice calls

    unsigned long long appStartUsedMem = MemoryTool::current();
    unsigned long long appStartPeakMem = MemoryTool::peak();

    Viewer* viewer = new Viewer( g_parent );
    viewer->show();
    QApplication::processEvents();

    BenchTool bencher;
    std::vector<BenchTool::BenchResult> results;

    printf( "[D] ----- Beginning benching -----\n" );

    printEnvironmentVariables();

    if (g_benchString.getLength())
    {
      printf( "[D] String: %s\n", g_benchString.toLatin1() );

      results = bencher.bench( g_dumpSGFilename.toStdString(), 
        g_dumpDisplay,
        g_dumpDisplayFilename.toStdString(),
        viewer, 
        g_benchString.toStdString() );
    }
    else if (g_useGui)
    {
      g_guiDialog = new ivShapeBenchmarkGui(viewer->getWidget());

      g_viewer = viewer;
      g_viewer->setPickCallback(pickCallback);
      g_guiDialog->setLaunchCallBack(performSingleTestFromString);
      g_guiDialog->setAnimationChangeCallBack(changeAnimationRate);
      g_guiDialog->setDrawStyleChangeCallBack(changeDrawStyle);
      g_guiDialog->setTransparencyTypeChangeCallBack(changeTransparencyType);
      g_guiDialog->show();

      SoQt::mainLoop();
    }
    else
    {
      BenchTool::BenchData benchData;

      benchData.benchString = "";
      benchData.id = 0;

      benchData.configName = g_renderingMode;
      benchData.frameEnd = g_benchFrameEnd;

      benchData.frameNum = g_benchFrameCount;
      if (g_viewingMode)
        benchData.frameNum = -2;
      benchData.frameStart = g_benchFrameStart;
      benchData.frameStepping = g_benchFrameSteps;
      benchData.shapesCount = g_geometryCount;

      benchData.shapeInfo.colorsBinding = g_colorsBinding;
      benchData.shapeInfo.normalsBinding = g_normalsBinding;
      benchData.shapeInfo.geomIndex = g_geometryIndex;
      benchData.shapeInfo.width = g_geometryWidth;
      benchData.shapeInfo.height = g_geometryHeight;
      benchData.shapeInfo.hasColors = !g_useOivColors;
      benchData.shapeInfo.hasNormals = !g_useOivNormals;
      benchData.shapeInfo.neighborTolerance = g_neighborTolerance;
      benchData.shapeInfo.hasTexCoords = !g_useOivTexCoords;
      benchData.shapeInfo.hasTexCoords3 = !g_useOivTexCoords3;
      benchData.shapeInfo.hasVertexAttribs = false;

      results = 
        bencher.bench( g_dumpSGFilename.toStdString(), 
        g_dumpDisplay,
        g_dumpDisplayFilename.toStdString(),
        viewer, 
        benchData, g_geometryType.toStdString() );     
    }

    viewer->unbindNormalContext();
    delete viewer;
    SoQt::finish();

    unsigned long long appEndPeakMem = MemoryTool::peak();

    if (g_csvOutput)
    {
      if (results.size() > 0)
      {
        SoCpuDevice::initClass();
        SoCpuDevice* cpu_device = SoCpuDevice::findFirstAvailableDevice();
        printf( "[I] %d - %s - %llu MB\n", cpu_device->getLogicalUnits(), cpu_device->getDeviceName().getString(), cpu_device->getTotalMemory() / (1024*1024));
        SoGLDevice::initClass();
        SoGLDevice* device = SoGLDevice::getDevice( 0 );
        printf( "[I] %d - %s - %llu MB\n", device->getDevicesCount(), device->getDeviceName().getString(), device->getTotalMemory() / (1024*1024));
        BenchTool::BenchResult mainResult = results[0];
        // column 1 is Success status
        printf( "1" );
        // column 2 is Benchstring
        printf( "%s%s", g_csvSepChar.getString(), mainResult.benchString.c_str());
        // column 3 is used memory @ application startup
        printf( "%s%.2f", g_csvSepChar.getString(), appStartUsedMem /(1024.*1024.)); 
        // column 4 is peak memory @ application startup
        printf( "%s%.2f", g_csvSepChar.getString(), appStartPeakMem /(1024.*1024.));
        // column 5 is used memory after first three frames
        printf( "%s%.2f", g_csvSepChar.getString(), mainResult.afterThirdFrameMemoryUsage  /(1024.*1024.));
        // column 6 is peak memory @ application end
        printf( "%s%.2f", g_csvSepChar.getString(), appEndPeakMem  /(1024.*1024.));
        // column 7 is frame time of the first three frames
        printf( "%s%.3f", g_csvSepChar.getString(), mainResult.thirdFrameTime + mainResult.secondFrameTime + mainResult.firstFrameTime );
        // column 8 is frame rate of the first 10 frames after the first three
        printf( "%s%.2f", g_csvSepChar.getString(), mainResult.remainingFps );
        // column 9 is global frame rate
        printf( "%s%.2f", g_csvSepChar.getString(), mainResult.globalFps );
        printf("\n");
      }
    }

    SoDB::finish();
    printf( "[D] ----- Bench done -----\n" );
  }
  catch( const std::exception& e )
  {
    if (g_csvOutput)
    {
      printf( "0;%s\n", g_benchString.toLatin1());
    }
    printf( "Exception: %s\n", e.what() );
    printf( "[D] ----- Bench done (with exception)-----\n" );
    exit( 1 );
  }
}

void printEnvironmentVariables()
{
  const SbString forcedRenderingMode = SoPreferences::getString( "OIV_FORCED_RENDERING_MODE", SbString( "NOT_FORCED" ) );
  const SbBool stackedShapes = SoPreferences::getBool( "BENCHER_STACKED_SHAPES", FALSE );
  const SbBool transparentShapes = SoPreferences::getBool( "BENCHER_TRANSP_SHAPES", FALSE );
  const SbBool bencherSingleViewer = SoPreferences::getBool( "BENCHER_SINGLE_VIEWER", FALSE );
  const int viewerHeight = SoPreferences::getInt( "BENCHER_VIEWER_HEIGHT", 0 );
  const int viewerWidth = SoPreferences::getInt( "BENCHER_VIEWER_WIDTH", 0 );
  const int geometryAmplitude = SoPreferences::getInt( "BENCHER_GEOMETRY_AMPLITUDE", 0 );
  const SbColor backgroundColor = SoPreferences::getColor( "BENCHER_VIEWER_BACKGROUND_COLOR", SbColor( 0.0f, 0.0f, 1.0f ) );

  std::cout << "OIV_FORCED_RENDERING_MODE = " << forcedRenderingMode << " ";
  std::cout << "BENCHER_STACKED_SHAPES = " << stackedShapes << " ";
  std::cout << "BENCHER_TRANSP_SHAPES = " << transparentShapes << " ";
  std::cout << "BENCHER_SINGLE_VIEWER = " << bencherSingleViewer << " ";
  std::cout << "BENCHER_VIEWER_HEIGHT = " << viewerHeight << " ";
  std::cout << "BENCHER_VIEWER_WIDTH = " << viewerWidth << " ";
  std::cout << "BENCHER_GEOMETRY_AMPLITUDE = " << geometryAmplitude << " ";
  std::cout << "BENCHER_VIEWER_BACKGROUND_COLOR = " << backgroundColor << std::endl;
}

