/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <Inventor/SbRotation.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/actions/SoRayPickAction.h>
#include <Inventor/devices/SoCpuBufferObject.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/nodes/SoVertexShape.h>
#include <Inventor/image/SbRasterImage.h>
#include <Inventor/image/SoPNGImageRW.h>
#include <Inventor/image/SoRasterImageFile.h>
#include <Inventor/sys/SoGL.h>

#include <QMouseEvent>
#include <viewer.h>

//------------------------------------------------------------------------------
Viewer::Viewer(QWidget* parent, const char* name, SbBool buildInsideParent)
: SoQtExaminerViewer(parent, name, buildInsideParent)
{
  m_root = new SoSeparator;
  SoPerspectiveCamera* perspectiveCamera;
  {
    perspectiveCamera = new SoPerspectiveCamera;
    perspectiveCamera->orientation = SbRotation( SbVec3f( 1.0, 0.0, 0.0 ), 45.0 );
    m_root->addChild( perspectiveCamera );
  
    m_drawStyle = new SoDrawStyle;
    m_root->addChild( m_drawStyle );

    m_shapeHints = new SoShapeHints;
    {
      m_shapeHints->vertexOrdering = SoShapeHints::CLOCKWISE;
      m_shapeHints->useVBO = TRUE;
      m_shapeHints->creaseAngle = 0.2f;
      m_shapeHints->windingType = SoShapeHints::NO_WINDING_TYPE;
      m_shapeHints->neighborTolerance = 0.0f; // 10e-6
    }
    m_root->addChild( m_shapeHints );

    m_shapesGroup = new SoSeparator;
    ((SoSeparator*)m_shapesGroup)->renderUnitId = -3;
    m_root->addChild( m_shapesGroup );
  }

  setBorder( FALSE );
  setDecoration( FALSE );
  setSceneGraph( m_root );
  setCamera( perspectiveCamera);
  setEventCallback(viewerEventCB, this);
  setTransparencyType( "OPAQUE_FIRST" );
  m_rebuildShapesList = true;
  m_animationRate = 0.f;
  m_framesCount = 0;

  m_pickCallback = NULL;
}


//------------------------------------------------------------------------------
Viewer::~Viewer()
{
}


//------------------------------------------------------------------------------
void 
Viewer::setAnimationRate(float rate)
{
  m_animationRate = rate;
}

//------------------------------------------------------------------------------
void 
Viewer::setDrawStyle(QString value)
{
  if (value == "LINES")
    m_drawStyle->style.setValue( SoDrawStyle::LINES);
  else if (value == "POINTS")
    m_drawStyle->style.setValue( SoDrawStyle::POINTS);
  else
    m_drawStyle->style.setValue( SoDrawStyle::FILLED);
}

//------------------------------------------------------------------------------
void 
Viewer::setTransparencyType(QString value)
{
  if (value == "NO_SORT")
    SoQtExaminerViewer::setTransparencyType( SoGLRenderAction::NO_SORT );
  else if (value == "OPAQUE_FIRST")
    SoQtExaminerViewer::setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  else if (value == "SORTED_OBJECT")
    SoQtExaminerViewer::setTransparencyType(SoGLRenderAction::SORTED_OBJECT);
  else if (value == "SORTED_PIXEL")
    SoQtExaminerViewer::setTransparencyType(SoGLRenderAction::SORTED_PIXEL);
}


//------------------------------------------------------------------------------
void 
Viewer::setDrawStyle( SoDrawStyle::Style style )
{
  m_drawStyle->style.setValue(style);
}

//------------------------------------------------------------------------------
void 
Viewer::setPickCallback(PickCallback cb)
{
  m_pickCallback = cb;
}

//------------------------------------------------------------------------------
SbBool
Viewer::viewerEventCB(void* userData, QEvent* eventData)
{
  Viewer* viewer = (Viewer*)userData;

  if (!viewer->isViewing() && eventData->type() == QMouseEvent::MouseButtonRelease)
  {
    QMouseEvent* mouseEvent = (QMouseEvent*)eventData;
    SbVec2s screenCoords = SbVec2s(mouseEvent->pos().x(), viewer->getSize()[1] - mouseEvent->pos().y());
    viewer->processPick(screenCoords);
    
    return TRUE;
  }

  return FALSE;
}

//------------------------------------------------------------------------------
void 
Viewer::processPick(const SbVec2s& screenMouseCoord)
{
  SoRayPickAction rayPickAction(getViewportRegion());

  rayPickAction.setPoint(screenMouseCoord);
  rayPickAction.setPickAll(FALSE);
  rayPickAction.setPickingMode(SoRayPickAction::POINT_PICKING);

  rayPickAction.apply(getSceneGraph());

  PickInformation info;
  info.screenPosition = screenMouseCoord;
  info.isShapePicked = false;

  SoPickedPoint* pickedPoint = rayPickAction.getPickedPoint();
  if (pickedPoint)
  {
    info.isShapePicked = true;
    info.worldCoords = pickedPoint->getPoint();
    info.normal = pickedPoint->getNormal();
    info.texCoord = pickedPoint->getTextureCoords();
  }

  if (m_pickCallback)
    m_pickCallback(info);
}

//------------------------------------------------------------------------------
void
Viewer::redraw()
{
  if (m_animationRate > 0.f )
  {
    if (m_rebuildShapesList)
    {
      m_shapesCollection.clear();
      SoSearchAction sa;
      sa.setInterest(SoSearchAction::ALL);
      sa.setType(SoShape::getClassTypeId());

      sa.apply(m_shapesGroup);

      SoPathList pathList = sa.getPaths();
      for (int i = 0; i < pathList.getLength(); i++)
      {
        SoPath* path = pathList[i];
        SoNode* node = path->getTail();

        if (node->isOfType(SoShape::getClassTypeId()))
          m_shapesCollection.push_back((SoShape*)node);
      }

      m_rebuildShapesList = false;
    }

    m_framesCount++;

    double limit = 1.0 / (double)m_framesCount;
    static double bias = 0.0;

    if ((double)m_animationRate >= limit)
    {
      std::vector<SoShape*>::const_iterator it;
      for (it = m_shapesCollection.begin(); it != m_shapesCollection.end(); it++)
      {
        if ((*it)->isOfType(SoBufferedShape::getClassTypeId()))
        {
          SoBufferedShape* bufferedShape = (SoBufferedShape*)*it;
          SoCpuBufferObject* cpuBuf = new SoCpuBufferObject;
          cpuBuf->ref();

          SoBufferObject* bufferObject = bufferedShape->vertexBuffer.getValue();
          bufferObject->map(cpuBuf, SoBufferObject::READ_WRITE);
          float* ptr = (float*)cpuBuf->map(SoBufferObject::READ_WRITE);

          size_t componentsCount = cpuBuf->getSize() / sizeof(float);

          for (size_t i = 0; i < componentsCount; i+=3)
          {
            float angle = ptr[i]*2.f + bias;
            while(angle > 2.f * 3.141592f)
              angle -= 2.f * 3.141592f;

            ptr[i+2] = sin(angle);
          }

          bias += 0.1;

          if(bias > 2.f * 3.141592f)
              bias -= 2.f * 3.141592f;

          cpuBuf->unmap();
          bufferObject->unmap(cpuBuf);
          cpuBuf->unref();
        }
        else
          if ((*it)->isOfType(SoVertexShape::getClassTypeId()))
          {
            SoVertexShape* shape = (SoVertexShape*)*it;
            SoVertexProperty* vp = (SoVertexProperty*)shape->vertexProperty.getValue();
            
            if (vp)
            {
              SbVec3f* ptr = vp->vertex.startEditing();

              size_t componentsCount = vp->vertex.getNum();

              for (size_t i = 0; i < componentsCount; i++)
              {
                float angle = ptr[i][0]*2.f + bias;
                while(angle > 2.f * 3.141592f)
                  angle -= 2.f * 3.141592f;

                ptr[i][2] = sin(angle);
              }

              bias += 0.1;

              if(bias > 2.f * 3.141592f)
                bias -= 2.f * 3.141592f;

              vp->vertex.finishEditing();
            }
          }
      }

      m_framesCount = 0;
    }
  }

  SoQtExaminerViewer::redraw();

  if (m_animationRate > 0.f)
    scheduleRedraw();
}


//------------------------------------------------------------------------------
void
Viewer::addNode(SoNode* node)
{
  m_shapesGroup->addChild( node );
  m_rebuildShapesList = true;
}


//------------------------------------------------------------------------------
void
Viewer::removeNode(SoNode* node)
{
  m_shapesGroup->removeChild( node );
  m_rebuildShapesList = true;
}


//------------------------------------------------------------------------------
void
Viewer::removeAllNodes()
{
  m_shapesGroup->removeAllChildren();
}

//------------------------------------------------------------------------------
void
Viewer::disbaleVBO(bool flag)
{
  m_shapeHints->useVBO = flag ? FALSE : TRUE;
}

//------------------------------------------------------------------------------
void
Viewer::setWindindType(SoShapeHints::WindingType type)
{
  m_shapeHints->windingType = type;
}

//------------------------------------------------------------------------------
void
Viewer::setNeighborTolerance(float neighborTolerance)
{
  m_shapeHints->neighborTolerance = neighborTolerance;
}

//------------------------------------------------------------------------------
