/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/viewers/SoXtWalkViewer.h>
#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/elements/SoCacheElement.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/sys/SoGL.h>

#include "ivViewMenus.h"

typedef struct {
  int count; // Number of files read
  char **names; // File names
  SoNode *original; // Un-munged scene
  SoSeparator *optimized; // Optimized scene
  SoSeparator *ivfixed; // Ivfixed scene
} FileInfo;

// Function prototypes 
static void print_usage(const char *);
static void parse_args(int , char **);
static SbBool readScene(FileInfo &); // Returns TRUE if error
static void setBusyCursor(SbBool);
static void createBusyCursor(Display *);
static void showAboutDialog();
static void getNewScene();
static void optimizationChanged(FileInfo &, SbBool);
static void newSceneCB(Widget, void *, XmFileSelectionBoxCallbackStruct *);
static void backgroundColorCB(void *, const SbColor *);
static void processTopbarEvent(Widget, ivMenuItem *, XmAnyCallbackStruct *);
static Widget buildAndLayoutMenus(Widget );
static SoXtViewer *buildAndLayoutViewer(Widget , Widget, SbBool);

// Global variables 
static SoXtViewer *viewer = NULL;
static Widget shell = NULL;
static SbBool useWalkViewer = FALSE;
static FileInfo files = { 0, NULL, NULL, NULL, NULL };
static SbBool ivfix = FALSE;
static SbBool shapeHintBackface = FALSE;

// Definitions for busy cursor:
static Cursor busyCursor = 0;
#define hourglass_width 17
#define hourglass_height 21
#define hourglass_x_hot 8
#define hourglass_y_hot 11

static char 
hourglass_bits[] = {
 (char)0x00, (char)0x00, (char)0x00, (char)0xfe, 
 (char)0xff, (char)0x00, (char)0x04, (char)0x40, 
 (char)0x00, (char)0x04, (char)0x40, (char)0x00,
 (char)0x04, (char)0x40, (char)0x00, (char)0xe8, 
 (char)0x2e, (char)0x00, (char)0xd0, (char)0x17, 
 (char)0x00, (char)0xa0, (char)0x0b, (char)0x00,
 (char)0x40, (char)0x05, (char)0x00, (char)0x40, 
 (char)0x05, (char)0x00, (char)0x40, (char)0x04, 
 (char)0x00, (char)0x40, (char)0x04, (char)0x00,
 (char)0x40, (char)0x04, (char)0x00, (char)0x20, 
 (char)0x09, (char)0x00, (char)0x10, (char)0x11, 
 (char)0x00, (char)0x88, (char)0x23, (char)0x00,
 (char)0xc4, (char)0x47, (char)0x00, (char)0xe4, 
 (char)0x4f, (char)0x00, (char)0xf4, (char)0x5f, 
 (char)0x00, (char)0xfe, (char)0xff, (char)0x00,
 (char)0x00, (char)0x00, (char)0x00
};
 
static char 
hourglass_mask_bits[] = {
 (char)0xff, (char)0xff, (char)0x01, (char)0xff, 
 (char)0xff, (char)0x01, (char)0xfe, (char)0xff, 
 (char)0x00, (char)0xfe, (char)0xff, (char)0x00,
 (char)0xfe, (char)0xff, (char)0x00, (char)0xfc, 
 (char)0x7f, (char)0x00, (char)0xf8, (char)0x3f, 
 (char)0x00, (char)0xf0, (char)0x1f, (char)0x00,
 (char)0xe0, (char)0x0f, (char)0x00, (char)0xe0, 
 (char)0x0f, (char)0x00, (char)0xe0, (char)0x0f, 
 (char)0x00, (char)0xe0, (char)0x0f, (char)0x00,
 (char)0xe0, (char)0x0f, (char)0x00, (char)0xf0, 
 (char)0x1f, (char)0x00, (char)0xf8, (char)0x3f, 
 (char)0x00, (char)0xfc, (char)0x7f, (char)0x00,
 (char)0xfe, (char)0xff, (char)0x00, (char)0xfe, 
 (char)0xff, (char)0x00, (char)0xfe, (char)0xff, 
 (char)0x00, (char)0xff, (char)0xff, (char)0x01,
 (char)0xff, (char)0xff, (char)0x01
};

//////////////////////////////////////////////////////////////////////
//
// Print the help message
//
static void
print_usage(const char *progname)
{
  fprintf(stderr, "Usage: %s [-hpw] [infiles]\n", progname);
  fprintf(stderr,
          "\t-h : Print this message (help) and exit\n"
          "\t-p : Enable performance meter\n"
          "\t-w : Use walk viewer (examiner viewer is default)\n");
  exit(99);
}

/////////////////////////////////////////////////////////////////////
//
// Parse the command line arguments
//
static void
parse_args(int argc, char **argv)
{
  int err = FALSE; // Flag: error in options?
  int c;
  
  while ((c = getopt(argc, argv, "hqw")) != -1) {
    switch(c) {
    case 'h': // Help
      err = TRUE;
      break;
    case 'q':
      // no-op for backward compatibility only
      break;
    case 'w':
      useWalkViewer = TRUE;
      break;
    default:
      err = TRUE;
      break;
    }
  }
  
  /* Handle multiple filenames */
  if (optind == argc) {
    // Act like one argument "-" was given
    argc++;
    argv[optind] = (char *)"-";
  }
  
  files.count = argc - optind;
  files.names = (char **) malloc (files.count * sizeof(char *));
  for (int i=0; i<files.count; i++, optind++) 
    files.names[i] = argv[optind];
  
  if (err) {
    print_usage(argv[0]);
  }
}


/////////////////////////////////////////////////////////////////////
//
// Read the scene from the file(s).
// If there are multiple files, combine their contents under one separator.
// Returns TRUE if read was OK, FALSE if read error.
//
static SbBool
readScene(FileInfo &files)
{
  SoInput in;
  SoSeparator *root;
  
  if (files.count == 0) 
    return 0;
  
  // Watch out! There's a hidden dependence here between this code
  // and the code in the optimizationChanged routine that runs files
  // through ivfix. In particular, the optimizationChanged code
  // assumes that, if stdin is being read from, that no other
  // filenames are given.
  if (files.count==1 && strcmp(files.names[0], "-")==0) {
    // Read from stdin
    if (isatty(fileno(stdin))) {
      fprintf(stderr, "Trying to read from standard input, ");
      fprintf(stderr, "but standard input is a tty!\n");
      return 0;
    } 
    else {
      in.setFilePointer(stdin);
    }    
    root = SoDB::readAll(&in);
    if (root == NULL) {
      fprintf(stderr, "Error reading file %s\n", files.names[0]);
    }
    else {
      root->ref();
    }    
    // SoInput destructor automatically closes file.
  }
  else { // Not reading from stdin:
    root = new SoSeparator;
    root->ref();
    
    //
    // Combine all the files under one separator
    //
    for (int i=0; i<files.count; i++) {
      if (!in.openFile(files.names[i])) {
        root->unref();
        return 0;
      }
      SoSeparator *inputRoot = SoDB::readAll(&in);
      if (inputRoot == NULL) {
        root->unref();
        return 0;
      }
      root->addChild(inputRoot);
      in.closeFile();
    } 
  }
  
  // Nuke old scene(s):
  if (files.original) {
    files.original->unref();
    files.original = NULL;
  }
  if (files.optimized) {
    files.optimized->unref();
    files.optimized = NULL;
  }
  if (files.ivfixed) {
    files.ivfixed->unref();
    files.ivfixed = NULL;
  }
  
  // And create new ones:
  if (root) {
    files.original = root;
    optimizationChanged(files, FALSE);
  }
  
  return (root != NULL);
}

/////////////////////////////////////////////////////////////////////
//
// Called when optimization options change, or when a new file is
// read.
//
static void
optimizationChanged(FileInfo &files, SbBool maintainCamera)
{
  // If no original file, have nothing to do:
  if (files.original == NULL) return;
  
  // We'll recreate the optimized graph:
  if (files.optimized) {
    files.optimized->unref();
    files.optimized = NULL;
  }
  if (ivfix) {
    SbBool err = FALSE;
    char *tempFile = NULL;
    setBusyCursor(TRUE);
    
    if (files.ivfixed == NULL) {
      do {
        
        // First, special case-- read from stdin:
        if (files.count==1 && strcmp(files.names[0], "-")==0) {
           //tempFile = tempnam(NULL, "ivFix");
           tempFile = (char *)"__tgsivfixfile.temp";
           mkstemp(tempFile);
          if (tempFile == NULL) {
            err = 1;
            break;
          }
          // 
          // Run ivfix on the original scene
          //
          
          // String long enough for temp filename plus "ivFix >"
          char ivfixCommand[L_tmpnam + 10];
          sprintf(ivfixCommand, "ivFix > %s", tempFile);
          FILE *ivfixPipe = popen(ivfixCommand, "w");
          if (ivfixPipe == NULL) {
            err = TRUE;
            break;
          }
          
          // Write to ivFix through pipe:
          SoOutput out;
          out.setBinary(TRUE);
          out.setFilePointer(ivfixPipe);
          SoWriteAction wa(&out);
          wa.apply(files.original);
          if (ferror(ivfixPipe)) {
            err = TRUE;
            break;
          }
          pclose(ivfixPipe);
          
          // Now read tempFile:
          SoInput in;
          if (!in.openFile(tempFile)) {
            err = TRUE;
            break;
          }
          files.ivfixed = SoDB::readAll(&in);
          if (files.ivfixed == NULL) {
            err = TRUE;
            break;
          } 
          else {
            files.ivfixed->ref();
          }
          in.closeFile();
        }
        else { // Not reading from stdin:
          files.ivfixed = new SoSeparator;
          files.ivfixed->ref();
          for (int i = 0; i < files.count; i++) {
            const char *filename = files.names[i];
            char *ivfixCommand = (char *)
              malloc(strlen("ivFix ")+strlen(filename)+1);
            sprintf(ivfixCommand, "ivFix %s", filename);
            
            // Read from ivFix:
            FILE *ivfixPipe = popen(ivfixCommand, "r");
            if (ivfixPipe == NULL) {
              files.ivfixed->unref();
              files.ivfixed = NULL;
              err = TRUE;
              break;
            }
            
            // If the filename includes a directory path, add
            // the directory name to the list of directories
            // where to look for input files:
            const char *slashPtr;
            char *searchPath = NULL;
            if ((slashPtr = strrchr(filename, '/')) != NULL) {
              searchPath = strdup(filename);
              searchPath[slashPtr - filename] = '\0';
              SoInput::addDirectoryFirst(searchPath);
            }
            
            SoInput in;
            in.setFilePointer(ivfixPipe);
            SoSeparator *root = SoDB::readAll(&in);
            
            if (searchPath) {
              SoInput::removeDirectory(searchPath);
              free(searchPath);
            }
            
            if (root == NULL) {
              files.ivfixed->unref();
              files.ivfixed = NULL;
              err = TRUE;
              break;
            }
            files.ivfixed->addChild(root);
            
            pclose(ivfixPipe);
          }
        }
      } while (0);
    }
    if (files.ivfixed) {
      files.optimized = files.ivfixed;
      files.optimized->ref();
    }
    
    setBusyCursor(FALSE);
    
    // Clean up temporary file (whether or not there was an error):
    if (tempFile && unlink(tempFile) < 0) {
      fprintf(stderr, "Warning: Error removing %s after ivFix\n",
        tempFile);
    }
    if (tempFile) free(tempFile);
    
    // If there was an error:
    if (err) {
      system("xconfirm -t 'Error running ivFix' > /dev/null");
    }
  }
  
  if (shapeHintBackface) {
    SoSeparator *optRoot = new SoSeparator;
    optRoot->boundingBoxCaching = SoSeparator::OFF;
    optRoot->pickCulling = SoSeparator::OFF;
    
    if (files.optimized != NULL) {
      // user also wants ivFix....
      optRoot->addChild(files.ivfixed);
      files.optimized->unref();
      files.optimized = NULL;
    } else {
      optRoot->addChild(files.original);
    }
    files.optimized = optRoot;
    files.optimized->ref();
    
    // Insert a ShapeHints node as the first child of files.optimized
    SoShapeHints *hints = new SoShapeHints;
    hints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
    hints->shapeType = SoShapeHints::SOLID;
    hints->setOverride(TRUE);
    files.optimized->insertChild(hints, 0);
  }
  
  //
  // Set scene graph to either optimized or original scene graph:
  //
  SoNode *currentCamera = NULL;
  if (maintainCamera) {
    SoNode *viewerCamera = viewer->getCamera();
    if (viewerCamera) {
      // Make a copy:
      currentCamera = viewerCamera->copy(FALSE);
      currentCamera->ref();
    }
  }
  
  if (files.optimized) {
    // optimized AND/OR ivfixed
    if (viewer->getSceneGraph() != files.optimized)
      viewer->setSceneGraph(files.optimized);
  } else {
    if (viewer->getSceneGraph() != files.original) {
      viewer->setSceneGraph(files.original);
    }
  }
  if (currentCamera) {
    SoNode *viewerCamera = viewer->getCamera();
    if (viewerCamera) {
      // Copy back:
      viewerCamera->copyFieldValues(currentCamera, FALSE);
    }
    currentCamera->unref();
  }
}

////////////////////////////////////////////////////////////////////
//
// Turn busy cursor on/off
//
static void
setBusyCursor(SbBool showBusy)
{
  Display *display = shell ? XtDisplay(shell) : NULL;
  Window window = shell ? XtWindow(shell) : (Window)NULL;
  
  if (!shell || !display || !window)
    return;
  
  // make sure we have defined the cursor
  if (! busyCursor) {
    createBusyCursor(display);
  }
  if (showBusy) {
    XDefineCursor(display, window, busyCursor);
  }
  else {
    XUndefineCursor(display, window);
  }
  if (viewer) {
    viewer->setCursorEnabled(!showBusy);
  }  
  // force the busy cursor to show up right now
  if (showBusy) {
    XSync(display, False);
  }
}

////////////////////////////////////////////////////////////////////
//
// Define busy cursor
//
static void
createBusyCursor(Display *display)
{
  Drawable d = DefaultRootWindow(display);
  XColor foreground, background;
  foreground.red = 65535;
  foreground.green = foreground.blue = 0;
  background.red = background.green = background.blue = 65535;
  
  Pixmap source = 
    XCreateBitmapFromData(display, d, hourglass_bits, hourglass_width, hourglass_height);
  Pixmap mask = 
    XCreateBitmapFromData(display, d, hourglass_mask_bits, hourglass_width, hourglass_height);

  busyCursor = 
    XCreatePixmapCursor(display, source, mask, &foreground, &background, hourglass_x_hot, hourglass_y_hot);
  XFreePixmap(display, source);
  XFreePixmap(display, mask);
}

///////////////////////////////////////////////////////////////////
//
// Brings up the "About..." dialog
//
static void
showAboutDialog()
{
  if (access("/usr/share/help/ivview/ivview.about", R_OK) != 0) {
    system("xconfirm -t 'Sorry, could not find "
           "/usr/share/help/ivview/ivview.about' > /dev/null");
    return;
  }
  
  char command[100];
  sprintf(command, "showcase -v /usr/share/help/ivview/ivview.about");
  
  int err = system(command);
  if (err) {
    system("xconfirm -t 'You must install showcase"
           " for this function to work' > /dev/null");
    return;
  }
}

///////////////////////////////////////////////////////////////////
//
// Use a motif file selection dialog to get the new filename. 
// Set up a callback to open the file and create the new scenegraph.
//
static void
getNewScene()
{
  static Widget fileDialog = NULL;

  if (fileDialog == NULL) {
    Arg args[5];
    int n = 0;
    
    // Unmanage when ok/cancel are pressed
    XtSetArg(args[n], XmNautoUnmanage, TRUE); n++;
    fileDialog = 
      XmCreateFileSelectionDialog(shell, (char *)"File Dialog", args, n);
    
    XtAddCallback(fileDialog, XmNokCallback, (XtCallbackProc)newSceneCB, NULL);
  }
  
  // Manage the dialog
  XtManageChild(fileDialog);
}

//
// Callback routine that gets called when the new filename
// has been entered
//
static void
newSceneCB(Widget, void *, XmFileSelectionBoxCallbackStruct *data)
{
  // Get the file name
  char *filename;
  XmStringGetLtoR(data->value,
                  (XmStringCharSet) XmSTRING_DEFAULT_CHARSET, &filename);
  
  files.count = 1;
  files.names[0] = strdup(filename);
  XtFree(filename);
  
  setBusyCursor(TRUE);
  SbBool readOk = readScene(files);
  setBusyCursor(FALSE);
  
  if (!readOk) {
    char str[100];
    strcpy(str, "Error reading file: ");
    strcat(str, files.names[0]);
    SoXt::createSimpleErrorDialog(shell, (char *)"File Error Dialog", str);
    return;
  } 
}

//
// Callback routine that gets called when the background
// color editor values change
//
static void
backgroundColorCB(void *, const SbColor *color)
{
  viewer->setBackgroundColor(*color);
}

///////////////////////////////////////////////////////////////////
//
// Process the topbar menu events.
static void
processTopbarEvent(Widget, ivMenuItem *data, XmAnyCallbackStruct *)
{
  switch (data->id) {
    //
    // File
    //
  case IV_FILE_ABOUT:
    showAboutDialog();
    break;
  case IV_FILE_OPEN:
    getNewScene();
    break;
  case IV_FILE_QUIT:
    exit (0);
    break; 
    
    //
    // Edit
    //
  case IV_EDIT_TRANSPARENCY:
    // Toggle the transparency mode
    if (viewer->getTransparencyType() == SoGLRenderAction::NO_SORT) {
      viewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    }
    else {
      viewer->setTransparencyType(SoGLRenderAction::NO_SORT);
    }
    break;

    //
    // Optimize
    //
  case IV_OPTIMIZE_IVFIX:
    ivfix = !ivfix;
    optimizationChanged(files, TRUE);
    break;
  case IV_OPTIMIZE_SHAPEHINTS:
    shapeHintBackface = !shapeHintBackface;
    optimizationChanged(files, TRUE);
    break;
  } 
}

///////////////////////////////////////////////////////////////////////
//
// Create the top menu bar and the associated menus
//
static Widget
buildAndLayoutMenus(Widget parent)
{
#ifndef _sgi
  WidgetClass zwidgetClass;
#endif
  ivMenuItem *menuItems = new ivMenuItem[IV_MENU_NUM];
  int i;
  for (i=0; i<IV_MENU_NUM; i++) {
    menuItems[i].id = i;
    menuItems[i].widget = NULL;
  }
  
  // 
  // Create the topbar menu
  //
  Widget menuWidget = XmCreateMenuBar(parent, (char *)"menuBar", NULL, 0);
  
  Arg popupargs[4];
  int popupn = 0;
#ifdef MENUS_IN_POPUP
  Widget shell = SoXt::getShellWidget(menuWidget);
  SoXt::getPopupArgs(XtDisplay(menuWidget), NULL, popupargs, &popupn);
#endif
  
  int itemCount = XtNumber(pulldownData);
  WidgetList buttons = (WidgetList) XtMalloc(itemCount * sizeof(Widget));
  
  Arg args[12];
  int n;
  for (i=0; i<itemCount; i++) {
    //
    // Make Topbar menu button
    //
    Widget subMenu = 
      XmCreatePulldownMenu(menuWidget, (char *)"", popupargs, popupn);
    
#ifdef MENUS_IN_POPUP
    // register callbacks to load/unload the pulldown colormap when the
    // pulldown menu is posted.
    SoXt::registerColormapLoad(subMenu, shell);
#endif
    
    int id = pulldownData[i].id;
    menuItems[id].widget = subMenu;
    
    XtSetArg(args[0], XmNsubMenuId, subMenu);
    buttons[i] = 
      XtCreateWidget(pulldownData[i].name, xmCascadeButtonGadgetClass, menuWidget, args, 1);
    
    //
    // Make subMenu buttons
    //
    int subItemCount = pulldownData[i].subItemCount;
    WidgetList subButtons = 
      (WidgetList) XtMalloc(subItemCount * sizeof(Widget));
    
    for (int j=0; j<subItemCount; j++) {
      if (pulldownData[i].subMenu[j].buttonType == IV_SEPARATOR) {
        subButtons[j] = 
          XtCreateWidget(NULL, xmSeparatorGadgetClass, subMenu, NULL, 0);
      }
      else {
        String callbackReason;
        
        switch (pulldownData[i].subMenu[j].buttonType) {
        case IV_PUSH_BUTTON:
#ifdef _sgi
          widgetClass = xmPushButtonGadgetClass;
#else
          zwidgetClass = xmPushButtonGadgetClass;
#endif
          callbackReason = XmNactivateCallback;
          n = 0;
          break;
        case IV_TOGGLE_BUTTON:
#ifdef _sgi
          widgetClass = xmToggleButtonGadgetClass;
#else
          zwidgetClass = xmToggleButtonGadgetClass;
#endif
          callbackReason = XmNvalueChangedCallback;
          n = 0;
          break;
        case IV_RADIO_BUTTON:
#ifdef _sgi
          widgetClass = xmToggleButtonGadgetClass;
#else
          zwidgetClass = xmToggleButtonGadgetClass;
#endif
          callbackReason = XmNvalueChangedCallback;
          XtSetArg(args[0], XmNindicatorType, XmONE_OF_MANY);
          n = 1;
          break;
        default:
          fprintf(stderr, 
            "ivview INTERNAL ERROR: bad buttonType\n");
          break;
        }
        
        //
        // Check for keyboard accelerator
        //
        char *accel = (char *)pulldownData[i].subMenu[j].accelerator;
        char *accelText = (char *)pulldownData[i].subMenu[j].accelText;
        XmString xmstr = NULL;
        if (accel != NULL) {
          XtSetArg(args[n], XmNaccelerator, accel); n++;
          
          if (accelText != NULL) {
            xmstr = XmStringCreate(accelText, XmSTRING_DEFAULT_CHARSET);
            XtSetArg(args[n], XmNacceleratorText, xmstr); n++;
          }
        }
        
        subButtons[j] = 
          XtCreateWidget(pulldownData[i].subMenu[j].name,
#ifdef _sgi
                         widgetClass, subMenu, args, n);
#else
                         zwidgetClass, subMenu, args, n);
#endif
        if (xmstr != NULL) {
          XmStringFree(xmstr);
        }
        id = pulldownData[i].subMenu[j].id;
        menuItems[id].widget = subButtons[j];
        XtAddCallback(subButtons[j], callbackReason,
                      (XtCallbackProc)processTopbarEvent,
                      (XtPointer) &menuItems[id]);
      }
    }
    XtManageChildren(subButtons, subItemCount);
    XtFree((char *)subButtons);
  }
   XtManageChildren(buttons, itemCount);
   XtFree((char *)buttons);
 
   //
   // Layout the menu bar
   //
   n = 0;
   XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
   XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
   XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
   XtSetValues(menuWidget, args, n);
   XtManageChild(menuWidget);
 
   return (menuWidget);
}

///////////////////////////////////////////////////////////////////
// Create the viewer and lay it out so that it is attached
// to the bottom of the menu bar
//
static SoXtViewer *
buildAndLayoutViewer(Widget parent, Widget menuWidget, SbBool walk)
{
  //
  //
  Arg args[12];
  int n = 0;
  SoXtViewer *viewer = NULL;
  if (!walk) {
    viewer = new SoXtExaminerViewer(parent);
  }
  else {
    viewer = new SoXtWalkViewer(parent);
  }
  viewer->setViewing(TRUE); // default to VIEW
  
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(args[n], XmNtopWidget, menuWidget); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetValues(viewer->getWidget(), args, n);
  
  return (viewer);
}

///////////////////////////////////////////////////////////////////
//
int
main(int argc, char **argv)
{
  XInitThreads();

  XtAppContext appContext;
  shell = 
    XtAppInitialize(&appContext, // context will be returned
                    "Inventor", // application class
                    NULL, 0, // options
                    (int *) &argc, argv, // command line args in/out
                    NULL, // fallback_resources
                    NULL, 0); // args, num_args
  
  // Parse command line arguments
  parse_args(argc, argv);
  
  // build the main window - this is done FIRST (before initialzing
  // Inventor and reading) to make the application startup feel
  // faster.
  
  Widget form = XtCreateWidget("Form", xmFormWidgetClass, shell, NULL, 0);
  XtVaSetValues(shell, XmNwidth, 520, XmNheight, 510, NULL);
  
  Widget menuWidget = buildAndLayoutMenus(form); 
  XtManageChild(form); 
  XtRealizeWidget(shell);
  XmUpdateDisplay(shell); // better than XSync() because it redraws too
  setBusyCursor(TRUE);
  
  // Initialize Inventor 
  SoXt::init(shell);
  
  // Create and lay out the viewer
  viewer = buildAndLayoutViewer(form, menuWidget, useWalkViewer);
  viewer->show();
  XSync(XtDisplay(shell), False);
  
  // Read the input scene from the file(s)
  readScene(files);

  setBusyCursor(FALSE);
  
  SoXt::mainLoop();
  return 0;
}


