/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <limits>
#include <vector>

#include <Inventor/helpers/SbFileHelper.h>
#include <LDM/readers/SoVolumeReader.h>
#include <LDM/SoLDMTopoOctree.h>

#include "metrics.h"
#include "SoVolumeAnalyzer.h"


//*****************************************************************************
SoVolumeAnalyzer::SoVolumeAnalyzer( const SbString& codec, int level )
: m_codec( codec ), m_level( level )
{

}

//*****************************************************************************
SoVolumeAnalyzer::~SoVolumeAnalyzer()
{

}

bool SoVolumeAnalyzer::init( const SbString& filename )
{
  m_reader = SoVolumeReader::getAppropriateReader( filename );

  if ( !m_reader->isDataConverted() )
  {
    SoDebugError::post( "SoVolumeAnalyzer::init", "%s is not a LDM file.", filename.toLatin1() );
    return false;
  }

  m_reader->setFilename( filename );

  //---------------------------------------------------------------------------
  // Get volume characterictics
  SbBox3f volumeSize;
  SbVec3i32 tileSize;
  SoDataSet::DataType datatype;

  m_reader->getDataChar( volumeSize, datatype, m_info.dimensions );
  m_info.datatype = static_cast<SbDataType>( datatype );
  m_info.isRGBA = m_reader->isRGBA();
  m_reader->getTileSize( tileSize );
  SbBool success = m_reader->getMinMax( m_info.range[0], m_info.range[1] );

  if ( !success )
  {
    if ( m_info.datatype.isInteger() )
    {
      m_info.range[0] = m_info.datatype.getMin();
      m_info.range[1] = m_info.datatype.getMax();
    }
    else
    {
      m_info.range[0] = -m_info.datatype.getMax();
      m_info.range[1] = m_info.datatype.getMax();
    }
  }

  m_topology.init( m_info.dimensions, tileSize, 0 );

  return true;
}

//*****************************************************************************
void SoVolumeAnalyzer::analyze( SbVolumeAnalyzeData &data ) const
{
  data.codecName = m_codec;
  size_t fileIdsCount = m_topology.getNumFileIDs();

  SbVec3i32 tileSize;
  m_reader->getTileSize( tileSize );

  //---------------------------------------------------------------------------
  // Tiles analysis

  data.tileData.resize( fileIdsCount );

  const SoTileAnalyzer tileAnalyzer( m_info, m_codec, m_level, tileSize );

  for ( int fileId = 0; fileId < static_cast<int>( fileIdsCount ); ++fileId )
  {
    SoLDMTileID tileId = m_topology.getTileID( fileId );

    SoBufferObject* tileBuffer = m_reader->readTile( fileId, m_topology.getTilePos(tileId) );
    tileAnalyzer.analyze( tileBuffer, data.tileData[fileId] );
    tileBuffer->clearInstance();

    std::cout << "\r                                                       \r";
    std::cout << ((double) fileId / (double) fileIdsCount)*100.0 << "%";
  }

  std::cout << "\r                                                       \r";

  //---------------------------------------------------------------------------
  // Volume metrics computation

  size_t voxelCount = m_info.dimensions[0];
  voxelCount *= m_info.dimensions[1];
  voxelCount *= m_info.dimensions[2];
  size_t voxelSize = m_info.datatype.getSize();

  data.totalUncompressedSize = voxelCount * voxelSize;
  data.totalCompressedSize = 0;
  data.totalEncodingTime = SbTime( 0.0 );
  data.totalDecodingTime = SbTime( 0.0 );
  data.maxError = m_info.range[0];
  data.minError = m_info.range[1];

  std::vector<double> originalMean( fileIdsCount );
  std::vector<double> originalVariance( fileIdsCount );

  std::vector<double> decodedMean( fileIdsCount );
  std::vector<double> decodedVariance( fileIdsCount );

  std::vector<double> covariance( fileIdsCount );

  std::vector<double> ssim( fileIdsCount );

  std::vector<double> absoluteError( fileIdsCount );
  std::vector<double> squareError( fileIdsCount );

  for ( int i = 0; i < static_cast<int>( fileIdsCount ); ++i )
  {
    const SbTileAnalyzeData &tileData = data.tileData[i];

    data.totalCompressedSize += tileData.compressedSize;
    data.totalEncodingTime   += tileData.encodingTime;
    data.totalDecodingTime   += tileData.decodingTime;

    originalMean[i]     = tileData.originalMean;
    originalVariance[i] = tileData.originalVariance;
    decodedMean[i]      = tileData.decodedMean;
    decodedVariance[i]  = tileData.decodedVariance;
    covariance[i]       = tileData.covariance;
    ssim[i]             = tileData.ssim;
    absoluteError[i]    = tileData.mae;
    squareError[i]      = tileData.mse;

    data.maxError = SbMathHelper::Max( data.maxError, tileData.maxError );
    data.minError = SbMathHelper::Min( data.minError, tileData.minError );
  }

  double sumNormalize = static_cast<double>( fileIdsCount );

  data.originalMean     = kahanSum( originalMean.begin(), originalMean.end() ) / sumNormalize;
  data.originalVariance = kahanSum( originalVariance.begin(), originalVariance.end() ) / sumNormalize;
  data.decodedMean      = kahanSum( decodedMean.begin(), decodedMean.end() ) / sumNormalize;
  data.decodedVariance  = kahanSum( decodedVariance.begin(), decodedVariance.end() ) / sumNormalize;
  data.covariance       = kahanSum( covariance.begin(), covariance.end() ) / sumNormalize;
  data.ssim             = kahanSum( ssim.begin(), ssim.end()) / sumNormalize;
  data.mae              = kahanSum( absoluteError.begin(), absoluteError.end() ) / sumNormalize;
  data.mse              = kahanSum( squareError.begin(), squareError.end() ) / sumNormalize;

  data.rmse = std::sqrt( data.mse );

  //---------------------------------------------------------------------------
  // PSNR computation

  if ( data.mse > 0.0 )
  {
    data.psnr = psnr( data.mse, m_info.range );
  }
  else
  {
    data.psnr = std::numeric_limits<double>::infinity();
  }
}
