/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <iostream>
#include <locale>
#include <limits>

#include <Inventor/helpers/SbFileHelper.h>

#include <LDM/readers/SoLDMReader.h>

#include <VolumeViz/nodes/SoVolumeRendering.h>

#include "SoVolumeAnalyzer.h"



bool parseArgs( int argc, char** argv, SbString& input, SbString& codec, int& level )
{
  SbString argument;
  int i = 1;
  while ( i < argc )
  {
    argument = argv[i++];
    if ( argument == "-c" )
    {
      if ( i >= argc )
      {
        return false;
      }

      codec = argv[i++];
    }
    else if ( argument == "-l" )
    {
      if ( i >= argc )
      {
        return false;
      }

      level = atoi( argv[i++] );
    }
    else if ( input.isEmpty() )
    {
      input = argument;
    }
    else
    {
      return false;
    }
  }

  if ( i == 1 )
  {
    return false;
  }

  return true;
}

int main( int argc, char** argv )
{
  SoDB::init();
  SoVolumeRendering::init();

  SbString filename;
  SbString codec= "gzip";
  int level = -1;

  bool success = parseArgs( argc, argv, filename, codec, level );

  if ( !success )
  {
    std::cout << "Usage: analyser filename [-c codec] [-l level]" << std::endl;
    std::cout << "\tfilename:\tThe file to analyze compression from. Must be a tiled format." << std::endl;
    std::cout << "\tcodec:\tThe codec used to test compression. By default use Gzip compression." << std::endl;
    std::cout << "\tlevel:\tCompression level." << std::endl;
    std::cout << std::endl;
    std::cout << "launching run test mode..." << std::endl;

    filename = "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm";
  }

  if ( SoDataCompressor::getAppropriateCompressor( codec ) == NULL )
  {
    std::cerr << codec << " is not a valid compressor name." << std::endl;
    std::cerr << "Available compressors: ";
    SbStringList compressorList = SoDataCompressor::getAvailableCompressors();
    for ( int i = 0; i < compressorList.getLength(); ++i )
    {
      std::cerr << compressorList[i]->toLatin1();

      if ( i + 1 < compressorList.getLength() )
      {
        std::cerr << ", ";
      }
    }
    std::cerr << std::endl;

    exit( 1 );
  }

  if ( !SbFileHelper::isAccessible( filename ) )
  {
    std::cerr << filename << " is not accesssible." << std::endl;
    exit( 1 );
  }

  std::locale::global( std::locale("") );
  std::cout.imbue( std::locale() );
  std::cout.precision(4);

  SoVolumeAnalyzer analyzer( codec, level );
  success = analyzer.init( filename );

  if ( !success )
  {
    exit(1);
  }

  SbVolumeAnalyzeData data;
  analyzer.analyze(data);

  double compressionRate = static_cast<double>( data.totalUncompressedSize ) / static_cast<double>( data.totalCompressedSize );

  double encodingRate = static_cast<double>( data.totalUncompressedSize ) / data.totalEncodingTime.getValue();
  double decodingRate = static_cast<double>( data.totalUncompressedSize ) / data.totalDecodingTime.getValue();

  std::cout << "Analysis Results" << std::endl;
  std::cout << "-- Codec: " << data.codecName << std::endl;
  std::cout << "-- Tile Count: " << data.tileData.size() << std::endl;
  std::cout << "-- Uncompressed: " << data.totalUncompressedSize << std::endl;
  std::cout << "-- Compressed: " << data.totalCompressedSize << std::endl;
  std::cout << "-- Compression Rate: " << compressionRate << std::endl;
  std::cout << "-- Encoding: " << data.totalEncodingTime.format("%H:%m:%s.%i").toLatin1() << std::endl;
  std::cout << "-- Encoding Rate: " << encodingRate << std::endl;
  std::cout << "-- Decoding: " << data.totalDecodingTime.format("%H:%m:%s.%i").toLatin1() << std::endl;
  std::cout << "-- Decoding Rate: " << decodingRate << std::endl;
  std::cout << "-- Original Mean: " << data.originalMean << std::endl;
  std::cout << "-- Original Variance: " << data.originalVariance << std::endl;
  std::cout << "-- Decoded Mean: " << data.decodedMean << std::endl;
  std::cout << "-- Decoded Variance: " << data.decodedVariance << std::endl;
  std::cout << "-- Covariance: " << data.covariance << std::endl;
  std::cout << "-- SSIM: " << data.ssim << std::endl;
  std::cout << "-- Max Error: " << data.maxError << std::endl;
  std::cout << "-- Min Error: " << data.minError << std::endl;
  std::cout << "-- MAE: " << data.mae << std::endl;
  std::cout << "-- MSE: " << data.mse << std::endl;
  std::cout << "-- RMSE: " << data.rmse << std::endl;
  std::cout << "-- PSNR: ";
  if ( data.psnr != std::numeric_limits<double>::infinity() )
  {
    std::cout << data.psnr;
  }
  else
  {
    std::cout << "Infinite";
  }
  std::cout << std::endl;

  SoVolumeRendering::finish();
  SoDB::finish();
  return 0;
}
