/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

/**
 * This example displays difference between two volumes. In order for it to
 * work, the volume must be the same size, and the tile size must be the same.
 * This tool is mainly aimed at compression error vizualization.
 *
 * Usage: errorDisplay original_filename decoded_filename min_error max_error
 *
 * The min_error and max_error parameter are optionnal and are used to set the
 * error visualization band. The default is min_error=0.01 and max_error=0.10.
 */

#include <iostream>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoMultiDataSeparator.h>

const SbString SHADER_FILENAME = "$OIVHOME/tools/source/LDMCompressionTools/errorDisplay/errorDisplay.glsl";

//*****************************************************************************
bool parseArgs( int argc, char** argv, SbString& reference, SbString& encoded, SbVec2f& range )
{
  SbString argument;

  int i = 1;
  while ( i < argc )
  {
    argument = argv[i++];
    if ( argument == "-r" )
    {
      if ( i + 1 >= argc )
      {
        return false;
      }

      range[0] = static_cast<float>( atof( argv[i++] ) );
      range[1] = static_cast<float>( atof( argv[i++] ) );
    }
    else if ( reference.isEmpty() )
    {
      reference = argument;
    }
    else if ( encoded.isEmpty() )
    {
      encoded = argument;
    }
    else
    {
      return false;
    }
  }

  return true;
}

//*****************************************************************************
int main(int argc, char **argv)
{
  SoDB::init();
  SoVolumeRendering::init();

  SbString reference;
  SbString encoded;
  SbVec2f range(-1.f, -1.f);

  bool success = parseArgs(argc, argv, reference, encoded, range);

  if ( !success || reference.isEmpty() || encoded.isEmpty() )
  {
    std::cout << "Usage: ErrorDisplay reference encoded [-r min max]" << std::endl;
    std::cout << "\treference:\t The original volume." << std::endl;
    std::cout << "\tencoded:\t The encoded volume." << std::endl;
    std::cout << "\tmin and max:\t The the error range to display." << std::endl;
    std::cout << std::endl;
    std::cout << "launching run test mode..." << std::endl;

    reference = "$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm";
    encoded = "$OIVHOME/examples/data/VolumeViz/3DHEAD-jpeg.ldm";
  }

  if ( !SbFileHelper::isAccessible( reference ) )
  {
    std::cerr << reference << " is not accessible." << std::endl;
    exit( 1 );
  }

  if ( !SbFileHelper::isAccessible( encoded ) )
  {
    std::cerr << encoded << " is not accessible." << std::endl;
    exit( 1 );
  }

  Widget myWindow = SoXt::init( argv[0] );

  if (!myWindow)
    return 0;

  //---------------------------------------------------------------------------
  // building scene graph

  SoRef<SoTransferFunction> pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::GLOW;
  pTransFunc->transferFunctionId = 0;

  SoRef<SoVolumeData> pVolData1 = new SoVolumeData;
  pVolData1->fileName = reference;
  pVolData1->dataSetId = 1;

  SoRef<SoVolumeData> pVolData2 = new SoVolumeData;
  pVolData2->fileName = encoded;
  pVolData2->dataSetId = 2;

  SbVec3i32 tileSize1 = pVolData1->getTileDimension();
  SbVec3i32 tileSize2 = pVolData2->getTileDimension();

  if ( tileSize1 != tileSize2 )
  {
    std::cerr << "The two volumes are incompatible, because of different tile sizes."  << std::endl;
    exit(1);
  }

  double min, max;

  SoRef<SoDataRange> pRange1 = new SoDataRange;
  pRange1->dataRangeId = 1;
  pVolData1->getMinMax( min, max );
  pRange1->min = min;
  pRange1->max = max;

  SoRef<SoDataRange> pRange2 = new SoDataRange;
  pRange2->dataRangeId = 2;
  pVolData1->getMinMax( min, max );
  pRange2->min = min;
  pRange2->max = max;

  SoRef<SoFragmentShader> pFragmentShader = new SoFragmentShader;
  pFragmentShader->sourceProgram.setValue( SHADER_FILENAME );

  pFragmentShader->addShaderParameter1i( "data1", 1 );
  pFragmentShader->addShaderParameter1i( "data2", 2 );

  // custom error band
  if ( range[0] >= 0.f && range[1] > 0.f)
  {
    pFragmentShader->addShaderParameter1f( "minError", range[0] );
    pFragmentShader->addShaderParameter1f( "maxError", range[1] );
  }

  SoRef<SoVolumeShader> pVolShader = new SoVolumeShader;
  pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, pFragmentShader.ptr() );
  pVolShader->forVolumeOnly = true;

  SoRef<SoVolumeRender> pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
  pVolRender->interpolation = SoVolumeRender::NEAREST;

  SoRef<SoMultiDataSeparator> root = new SoMultiDataSeparator;

  root->addChild( pVolShader.ptr() );
  root->addChild( pTransFunc.ptr() );

  root->addChild( pVolData1.ptr() );
  root->addChild( pRange1.ptr() );

  root->addChild( pVolData2.ptr() );
  root->addChild( pRange2.ptr() );

  root->addChild( pVolRender.ptr() );

  //---------------------------------------------------------------------------
  // setup UI

  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer( myWindow );
  myViewer->setSceneGraph( root.ptr() );
  myViewer->setTitle( "Error Display" );
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  SoXt::finish();

  //---------------------------------------------------------------------------
  // cleaning

  delete myViewer;

  pVolShader = NULL;
  pFragmentShader = NULL;
  pTransFunc = NULL;
  pRange1 = NULL;
  pRange2 = NULL;
  pVolData1 = NULL;
  pVolData2 = NULL;
  pVolRender = NULL;
  root = NULL;

  SoVolumeRendering::finish();

  return 0;
}
