/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

/**
 * This Benchmark tool will display a scene contained in an OpenIventor file
 * for a fixed number of frames. This tool currently record the following
 * information for each frame:
 * 
 * - Frame Duration: the amount of time elapsed to render the frame.
 * - Frame Time: The timestamp when the frame finished to render.
 * - CPU Memory Usage: the CPU memory peak all along the frame rendering.
 * - GPU Memory Usage: the GPU memory peak reached all along the frames
 *                     rendering. Warning: this can be unavailable on recent
 *                     AMD GPU.
 */

#include <iostream>

#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SbString.h>
#include <Inventor/Qt/SoQt.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoSeparator.h>

#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <recorder.h>

BenchmarkRecorder* g_recorder = NULL;
BenchmarkRecord g_record;

struct CmdArgs
{
  SbString ivFile;
  size_t frameCount;
  bool forceInteractive;
};

CmdArgs g_args;

void usage()
{
  std::cout << "Usage: VVizBenchmark -i <iv file path> -c <frame count> [--force-interactive]" << std::endl;
  std::cout << "\t-i: The input file to benchmark." << std::endl;
  std::cout << "\t-c: The number of frame to record." << std::endl;
  std::cout << "\t--force-interactive: Force the use of interactive mode, as it affect performances." << std::endl;
}

bool parseArgs( int argc, char** argv )
{
  g_args.ivFile = "";
  g_args.frameCount = 0;
  g_args.forceInteractive = false;

  for ( size_t i = 1; i < argc; ++i )
  {
    SbString arg(argv[i]);

    if ( arg == "-i" )
    {
      ++i;
      g_args.ivFile = SbString(argv[i]);
    }
    else if ( arg == "-c" )
    {
      ++i;
      g_args.frameCount = atoi(argv[i]);
    }
    else if ( arg == "--force-interactive" )
    {
      g_args.forceInteractive = true;
    }
  }

  return true;
}

int
main( int argc, char** argv )
{
  parseArgs(argc, argv);

  if ( g_args.ivFile == "" || g_args.frameCount == 0 )
  {
    usage();
    return 1;
  }

  SoWidget window = SoQt::init( "VVizBenchmark" );
  if ( !window )
  {
    std::cerr << "Unable to init Xt window." << std::endl;
    return 1;
  }

  SoDB::init();
  SoVolumeRendering::init();

  g_recorder = new BenchmarkRecorder( window );
  
  // Add root scene graph
  SoInput input;
  if (!input.openFile( g_args.ivFile ))
  {
    std::cerr << "cannot open iv file" << std::endl;
    return 1;
  }

  SoNode* root;
  if (!SoDB::read( &input, root ))
  {
    std::cerr << "cannot read iv file" << std::endl;
    return 1;
  }

  SoRef<SoInteractiveComplexity> complexity( new SoInteractiveComplexity );

  if ( g_args.forceInteractive )
    complexity->interactiveMode = SoInteractiveComplexity::FORCE_INTERACTION;

  SoRef<SoSeparator> scene( new SoSeparator );
  scene->addChild( complexity.ptr() );
  scene->addChild( root );

  g_recorder->setSceneGraph( scene.ptr() );

  g_recorder->show();
  SoQt::show( window );

  g_recorder->record( g_record, g_args.frameCount );
  
  std::cout << "Frame Duration"
    << ";" << "Frame Time"
    << ";" << "CPU Memory Usage"
    << ";" << "GPU Memory Usage"
    << std::endl;

  for ( size_t i = 0; i < g_args.frameCount; ++i )
  {
    const BenchmarkFrameRecord& frameRecord = g_record[i];
    std::cout << frameRecord.duration
      << ";" << frameRecord.time
      << ";" << frameRecord.cpuMemory
      << ";" << frameRecord.gpuMemory
      << std::endl;
  }

  delete g_recorder;

  SoVolumeRendering::finish();
  SoDB::finish();
  SoQt::finish();

  return 0;
}
